/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package org.jvnet.glassfish.comms.clb.core;

import com.sun.enterprise.admin.server.core.AdminService;
import com.sun.enterprise.config.ConfigContext;
import com.sun.enterprise.config.ConfigException;
import com.sun.enterprise.config.serverbeans.AvailabilityService;
import com.sun.enterprise.config.serverbeans.Cluster;
import com.sun.enterprise.config.serverbeans.ClusterHelper;
import com.sun.enterprise.config.serverbeans.Config;
import com.sun.enterprise.config.serverbeans.ConvergedLbClusterRef;
import com.sun.enterprise.config.serverbeans.ConvergedLbConfig;
import com.sun.enterprise.config.serverbeans.ConvergedLbConfigs;
import com.sun.enterprise.config.serverbeans.ConvergedLoadBalancer;
import com.sun.enterprise.config.serverbeans.Domain;
import com.sun.enterprise.config.serverbeans.ServerBeansFactory;
import com.sun.enterprise.config.serverbeans.ServerHelper;
import com.sun.enterprise.config.serverbeans.ServerRef;
import com.sun.enterprise.server.ServerContext;
import java.util.logging.Logger;

import org.jvnet.glassfish.comms.clb.admin.CLBConfigurator;
import org.jvnet.glassfish.comms.clb.core.sip.SipLoadBalancerManager;
import org.jvnet.glassfish.comms.httplayers.HttpLayer;
import org.jvnet.glassfish.comms.util.LogUtil;


import java.util.logging.Level;


public class ConvergedLoadBalancerFactory {
    private static SipLoadBalancerManager sipLBM;
    private static HttpLoadBalancingManager httpLBM;
    private static final Logger _logger = LogUtil.CLB_LOGGER.getLogger();
    private static String dcrFileName;
    private static boolean loadBalancingManagersInitialized = false;
    private static boolean loadBalancingManagersCreated = false;
    private static String clbConfigFileName;
    private static String configName;

    /**
     * Gets the SIP load balancing manager layer and also initializes. Used by the
     * {@link com.ericsson.ssa.config.LayerHandler} (via reflection) when
     * setting up the SIP and HTTP processing chains at start-up.
     *
     */
    public static SipLoadBalancerManager getInstance() {
        createLoadBalancingManagers();
        //Initiliazing the converged load balancer as both HttpLoadBalancingManager
        //and SipLoadBalancingManager must be initialized together
        //This is done here as SIP container will be available now
        //for SIP component to initialize
        initialize();

        return sipLBM;
    }

    private static synchronized void initialize() {
        if (loadBalancingManagersInitialized) {
            return;
        }
        
        loadBalancingManagersInitialized = true;
        
        if ((httpLBM != null) && (sipLBM != null)) {
            if(_logger.isLoggable(Level.FINE))
                _logger.log(Level.FINE, "clb.initializing_the_converged_load_balancer");

            CLBConfigurator clbConfigurator = CLBConfigurator.createInstance();
            clbConfigurator.init(configName, clbConfigFileName, sipLBM, httpLBM);
            dcrFileName = clbConfigurator.getDCRFileName();
            if(_logger.isLoggable(Level.FINE))
                _logger.log(Level.FINE,
                        "clb.completed_initialization_of_the_converged_load_balancer");
        } else {
            if(_logger.isLoggable(Level.FINE))
                _logger.log(Level.FINE, "clb.not_initializing_the_converged_load_balancer");
        }
    }

    private static synchronized void createLoadBalancingManagers() {
        if (loadBalancingManagersCreated) {
            return;
        }

        loadBalancingManagersCreated = true;

        if(_logger.isLoggable(Level.FINE))
            _logger.log(Level.FINE, "clb.configuring_the_converged_load_balancer");

        ServerContext sc = com.sun.enterprise.server.ondemand.OnDemandServer.getServerContext();
        ConfigContext instanceConfigContext = sc.getConfigContext();

        //Added check whether this instance is DAS, as DAS cannot act as a converged load-balancer
        //This check may not be needed in case config of DAS will never have converged-load-balancer
        //under availability-service
        try {
            if (ServerHelper.isDAS(instanceConfigContext, sc.getInstanceName())) {
                _logger.log(Level.INFO, 
                    "clb.clb_not_enabled_on_DAS");

                return;
            }
        } catch (ConfigException ex) {
            _logger.log(Level.SEVERE, 
                    "clb.unable_to_determine_whether_instance_is_DAS_or_not",
                    new Object[]{ex.getMessage()});
            if(_logger.isLoggable(Level.FINE))
                _logger.log(Level.FINE,
                        "clb.caught_an_exception",
                        ex);
        }

        Config instanceConfig = null;

        try {
            instanceConfig = ServerBeansFactory.getConfigBean(instanceConfigContext);
        } catch (ConfigException ex) {
            _logger.log(Level.SEVERE, 
                    "clb.configexception_will_getting_instance_config",
                    new Object[]{ex.getMessage()});
            if(_logger.isLoggable(Level.FINE))
                _logger.log(Level.FINE,
                        "clb.caught_an_exception",
                        ex);
            return;
        }

        AvailabilityService haService = instanceConfig.getAvailabilityService();

        if (haService == null) {
            _logger.log(Level.SEVERE,
                        "clb.no_availability_service_found");
            return;
        }

        ConvergedLoadBalancer clb = haService.getConvergedLoadBalancer();

        if (clb == null) {
            _logger.log(Level.INFO,
                    "clb.no_clb_entry_found");
            return;
        }

        _logger.log(Level.INFO,
                    "clb.clb_entry_found");
        
        //Setting config-name and  clb-config-file-name
        configName = instanceConfig.getName();
        clbConfigFileName = clb.getConfigFile();

        //Creating instance of HttpLoadBalancingManager
        //and SipLoadBalancerManager
        //They will be initialized later
        if (httpLBM == null) {
            httpLBM = HttpLoadBalancingManager.getInstance();
        }

        if (sipLBM == null) {
            sipLBM = SipLoadBalancerManager.getInstance();
        }

        if(_logger.isLoggable(Level.FINE))
            _logger.log(Level.FINE, "clb.configured_the_converged_load_balancer");
    }

    /**
     * This is the method used by the {@link org.jvnet.glassfish.comms.httplayers.HttpLayerHandler} when initializing from dispatcher.xml.
     * @return http load-balancing layer;
     */
    public static HttpLayer getHttpLayerInstance() {
        createLoadBalancingManagers();

        return httpLBM;
    }
    
    /**
     * This method is used to determine whether converged load-balancer layer is active
     * in this instance.
     * 
     * @return true if converged load-balancer layer is active
     */
    public static boolean isLayerActive(){
        if(sipLBM == null || httpLBM == null)
            return false;
        return true;
    }

    /**
     * Returns the DCR file name
     *
     * @return the name of dcr file
     */
    public static String getDCRFileName() {
        return dcrFileName;
    }
    
    /**
     * This method is used to find out whether this instance is front-ended 
     * by CLB or not.
     * 
     * @return true, if instance is front-ended by CLB
     */
    public static boolean isInstanceFrontEndedByCLB() {

        boolean matchFound = false;

        try {
            ServerContext sc =
                com.sun.enterprise.server.ondemand.OnDemandServer.getServerContext();
            ConfigContext instanceConfigContext = sc.getConfigContext();
            ConfigContext adminConfigContext =
                AdminService.getAdminService().getAdminContext().getAdminConfigContext();
            String instanceName = sc.getInstanceName();
            String clusterName = null;
            boolean isCluster = false;
            //Determine whether this instance is a standalone instance
            //or a clustered instance
            if (ServerHelper.isServerClustered(adminConfigContext,
                instanceName)) {
                Cluster cluster =
                    ClusterHelper.getClusterForInstance(adminConfigContext,
                    instanceName);
                clusterName = cluster.getName();
                isCluster = true;
            }

            //get all converged lb configs
            Domain domain = (Domain) adminConfigContext.getRootConfigBean();
            ConvergedLbConfigs clbConfigs = domain.getConvergedLbConfigs();
            // Null in case of a DAS or developer profile instance.Hence no CLB 
            // frontending  this instance.
            if( clbConfigs == null ) return false;
            ConvergedLbConfig[] clbConfigArray = clbConfigs.getConvergedLbConfig();
            //get all converged lb ref and determine if one points to this instance
            if (isCluster) {
                for (int i = 0; i < clbConfigArray.length && !matchFound; i++) {
                    ConvergedLbClusterRef[] clusterRefs =
                        clbConfigArray[i].getConvergedLbClusterRef();
                    for (int j = 0; j < clusterRefs.length; j++) {
                        if (clusterRefs[j].getRef().equals(clusterName)) {
                            matchFound = true;
                            break;
                        }
                    }
                }
            } else {
                for (int i = 0; i < clbConfigArray.length && !matchFound; i++) {
                    ServerRef[] serverRefs =
                        clbConfigArray[i].getServerRef();
                    for (int j = 0; j < serverRefs.length; j++) {
                        if (serverRefs[j].getRef().equals(instanceName)) {
                            matchFound = true;
                            break;
                        }
                    }
                }
            }
        //May be a further check whether any converged-load-balancer refers to this 
        //converged lb config can be done
        } catch (ConfigException ex) {
            _logger.log(Level.SEVERE,
                "clb.exception_when_detecting_whether_instance_is_front_ended_by_CLB", ex);
        }
        if (_logger.isLoggable(Level.FINE)) {
            _logger.log(Level.FINE, "clb.is_instance_front_ended_by_CLB",
                new Object[]{matchFound});
        }
        return matchFound;
    }
}

