/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.jvnet.glassfish.comms.clb.core;

import java.util.logging.Logger;
import org.jvnet.glassfish.comms.clb.proxy.http.util.HttpRequest;
import org.jvnet.glassfish.comms.util.LogUtil;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.logging.Level;


/**
 * Router implementation for round-robin loadbalancing policy
 * @author kshitiz
 */
public class RoundRobinRouter implements Router {
    private static final Logger _logger = LogUtil.CLB_LOGGER.getLogger();

    /* Clusters to which this router can route requests */
    private List<ServerCluster> clusters;

    /* List holding all the available server instances. This include healthy/unhealthy,
     * enabled/disabled instances */
    private ArrayList<ServerInstance> completeInstanceList;

    /* List holding only healthy and enabled instances */
    private ArrayList<ServerInstance> activeInstanceList;

    /* Flag to decide, whether to route to only active instances */
    private boolean activeRouting;

    /* index of instance which will service the request */
    private int index;

    /* controller with which this request group is associated */
    protected Controller controller;

    /* lock to be used when modifying the active list*/
    private final Object lock;

    /** Creates a new instance of RoundRobinRouter */
    public RoundRobinRouter(List<ServerCluster> clusters, boolean activeRouting) {
        this.clusters = clusters;
        this.activeRouting = activeRouting;
        lock = new Object();
    }

    public boolean initialize() {
        activeInstanceList = new ArrayList<ServerInstance>();
        completeInstanceList = new ArrayList<ServerInstance>();

        for (ServerCluster cluster : clusters) {
            Collection<ServerInstance> instances = cluster.getAllInstances();
            completeInstanceList.addAll(instances);

            for (ServerInstance instance : instances) {
                instance.addRouter(this);

                if (instance.isEnabled() && instance.isHealthy()) {
                    if(_logger.isLoggable(Level.FINE))
                        _logger.log(Level.FINE,
                                "clb.adding_instance_to_active_list",
                                new Object[]{instance.getName()});
                    addToActiveList(instance);
                } else {
                    if(_logger.isLoggable(Level.FINE))
                        _logger.log(Level.FINE,
                                "clb.not_adding_instance_to_active_list",
                                new Object[]{instance.getName(), instance.isEnabled(),
                                instance.isHealthy()});
                }
            }
        }

        return true;
    }

    public boolean reinitialize(ServerInstance instance) {
        if (instance.isEnabled() && instance.isHealthy()) {
            return addToActiveList(instance);
        }

        return removeFromActiveList(instance);
    }

    public ServerInstance selectInstance(HttpRequest req) {
        return getActiveInstance();
    }

    protected ServerInstance getActiveInstance() {
            if(_logger.isLoggable(Level.FINE))
                _logger.log(Level.FINE,
                        "clb.active_list_size",
                        new Object[]{activeInstanceList.size()});
            ServerInstance instanceSelected;
            int indexSelected;
        synchronized (lock) {
            if(activeInstanceList.size() == 0){
                _logger.log(Level.SEVERE, "clb.no_active_instance");
                return null;
            }
            index = (index + 1) % activeInstanceList.size();
            indexSelected = index;
            instanceSelected = activeInstanceList.get(index);
        }
            if(_logger.isLoggable(Level.FINE))
                _logger.log(Level.FINE,
                        "clb.index_selected",
                        new Object[]{indexSelected});

            return instanceSelected;
    }

    public void handleEnableEvent(ServerInstance instance) {
        if(_logger.isLoggable(Level.FINE))
            _logger.log(Level.FINE,
                    "clb.handling_enable_event",
                    new Object[]{instance.getName()});
        if(instance.isHealthy())
            addToActiveList(instance);
    }

    public void handleDisableEvent(ServerInstance instance) {
        if(_logger.isLoggable(Level.FINE))
            _logger.log(Level.FINE,
                    "clb.handling_disable_event",
                    new Object[]{instance.getName()});
        if(instance.isHealthy())
            removeFromActiveList(instance);
    }

    public void handleRecoveryEvent(ServerInstance instance) {
        if(_logger.isLoggable(Level.FINE))
            _logger.log(Level.FINE,
                    "clb.handling_recovery_event",
                    new Object[]{instance.getName()});
        if(instance.isEnabled())
            addToActiveList(instance);
    }

    public void handleFailureEvent(ServerInstance instance) {
        if(_logger.isLoggable(Level.FINE))
            _logger.log(Level.FINE,
                    "clb.handling_failure_event",
                    new Object[]{instance.getName()});
        if(instance.isEnabled())
            removeFromActiveList(instance);
    }

    private boolean addToActiveList(final ServerInstance instance) {
        if(_logger.isLoggable(Level.FINE))
            _logger.log(Level.FINE,
                    "clb.adding_to_active_list",
                    new Object[]{instance.getName()});
        synchronized (lock) {
            if(!activeInstanceList.contains(instance))
                return activeInstanceList.add(instance);
        }
        return false;
    }
    
    private boolean removeFromActiveList(final ServerInstance instance) {
        if(_logger.isLoggable(Level.FINE))
            _logger.log(Level.FINE,
                    "clb.removing_from_active_list",
                    new Object[]{instance.getName()});
        synchronized (lock) {
            if(activeInstanceList.contains(instance))
                return activeInstanceList.remove(instance);
        }
        return false;
    }
    
    public ServerInstance selectInstance(ConsistentHashRequest req) {
        // Not relevant for RR
        throw new UnsupportedOperationException(
            "Not applicable for Round-Robin");
    }

    public void setController(Controller controller) {
        this.controller = controller;
    }
}
