/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package org.jvnet.glassfish.comms.clb.core.sip;

import com.ericsson.ssa.container.datacentric.DataCentricUtil;

import com.sun.enterprise.admin.server.core.AdminService;
import com.sun.enterprise.config.ConfigContext;
import com.sun.enterprise.config.ConfigException;
import com.sun.enterprise.config.serverbeans.Cluster;
import com.sun.enterprise.config.serverbeans.ClusterHelper;
import com.sun.enterprise.config.serverbeans.Server;
import com.sun.enterprise.config.serverbeans.ServerHelper;
import com.sun.enterprise.server.ServerContext;

import org.jvnet.glassfish.comms.clb.core.CLBRuntimeException;
import org.jvnet.glassfish.comms.clb.core.util.ConsistentHash;
import org.jvnet.glassfish.comms.clb.core.util.LoadbalancerUtil;
import org.jvnet.glassfish.comms.gms.GMSEventListenerException;
import org.jvnet.glassfish.comms.util.LogUtil;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.jvnet.glassfish.comms.gms.GMSEventListener;


public class DataCentricUtilImpl extends DataCentricUtil{
    private static Logger logger = LogUtil.CLB_LOGGER.getLogger();
    private static DataCentricUtil original;
    private static final DataCentricUtilImpl instance = new DataCentricUtilImpl();
    private String clusterName;
    private DCUIGMSEventListenerImpl gmsEventListenerImpl;
    private ConsistentHash<String, ServerInstance> idealHash;
    private ConsistentHash<String, ServerInstance> actualHash;
    private Map<String, Boolean> instanceHealthy = new HashMap<String, Boolean>();
    private String instanceName;
    private String localBeKey;

    private DataCentricUtilImpl() {
    }

    private void init() {
        try {
            initHash();
            if (clusterName != null) {
                gmsEventListenerImpl = new DCUIGMSEventListenerImpl(clusterName);
            }
        } catch (GMSEventListenerException e) {
            logger.log(Level.SEVERE,
                "org.jvnet.glassfish.comms.clb.core.sip.DataCentricUtilImpl.failed_start_callback",
                e);
        }
    }

    public static void install() {
        original = DataCentricUtil.getInstance();
        instance.init();
        DataCentricUtil.setInstance(instance);
    }
    
    public static void deinstall() {
        instance.cleanup();
        DataCentricUtil.setInstance(original);
    }
    

    @Override
    public boolean isDataCentricEnabled() {
        return true;
    }

    @Override
    public String getActualServerInstance(String key) {
        String serverInstance;
        if (actualHash == null) {
            serverInstance = instanceName;
        } else {
            serverInstance = actualHash.get(key).getName();
        }
        
        if (logger.isLoggable(Level.FINEST)) {
            logger.log(Level.FINEST, "State of actualHash: " + actualHash + "; key = " + key + "; server=" + serverInstance);
        }
        return serverInstance;
    }

    @Override
    public String getIdealServerInstance(String key) {
        String serverInstance;
        if (idealHash == null) {
            serverInstance = instanceName;
        } else {
            serverInstance = idealHash.get(key).getName();
        }

        if (logger.isLoggable(Level.FINEST)) {
            logger.log(Level.FINEST, "State of idealHash: " + idealHash + "; key = " + key + "; server=" + serverInstance);
        }
        return serverInstance;
    }

    @Override
    public boolean isLocal(String key) {
        String serverInstance = getActualServerInstance(key);
        boolean result = true;
        if (actualHash != null) {
            return instanceName.equals(serverInstance);
        }
        if (logger.isLoggable(Level.FINEST)) {
            logger.log(Level.FINEST, "Result = " + result);
        }
        return result;
    }
    
    @Override
    public String getLocalKey() {
        // TODO the localBeKey must be set to null when a server instance is added to the cluster
        if (localBeKey == null) {
            for (int i = 0; i < 10000; i++) {
                String instance = getIdealServerInstance(""+i);
                if (instanceName.equals(instance)) {
                    localBeKey = ""+i;
                    break;
                }
            }
            if (localBeKey == null) throw new RuntimeException("Could not generate a local back-end key.");
        }
        return localBeKey;
    }

    private void initHash() {
        try {
            ServerContext sc = com.sun.enterprise.server.ondemand.OnDemandServer.getServerContext();
            ConfigContext instanceConfigContext = sc.getConfigContext();
            instanceName = sc.getInstanceName();
            ConfigContext adminCtx = AdminService.getAdminService()
                                                 .getAdminContext()
                                                 .getAdminConfigContext();

            if (ServerHelper.isServerClustered(instanceConfigContext,
                        instanceName)) {
                Cluster cluster = ClusterHelper.getClusterForInstance(instanceConfigContext,
                        instanceName);
                clusterName = cluster.getName();

                Server[] servers = ServerHelper.getServersInCluster(adminCtx,
                        clusterName);

                idealHash = new ConsistentHash<String, ServerInstance>(1024, "DCU_IdealHash");
                actualHash = new ConsistentHash<String, ServerInstance>(1024, "DCU_ActualHash");

                for (int i = 0; i < servers.length; i++) {
                    String serverName = servers[i].getName();
                    instanceHealthy.put(serverName, true);
                    idealHash.addNode(new ServerInstance(clusterName, serverName));
                    actualHash.addNode(new ServerInstance(clusterName, serverName));
                }

                if (logger.isLoggable(Level.FINEST)) {
                    logger.log(Level.FINEST, "Setting up idealHash");
                }
                idealHash.setup();
                
                if (logger.isLoggable(Level.FINEST)) {
                    logger.log(Level.FINEST, "Setting up actualHash");
                }
                actualHash.setup();
                
            }
        } catch (ConfigException e) {
            logger.log(Level.SEVERE,
                "org.jvnet.glassfish.comms.clb.core.sip.DataCentricUtilImpl.failed_setup_ch",
                e);
        }
    }
    
    public void cleanup() {
        gmsEventListenerImpl.cleanup();
    }
    
    private static class ServerInstance {
        private String clusterName;
        private String name;
        
        public ServerInstance(String clusterName, String name) {
            super();
            this.clusterName = clusterName;
            this.name = name;
        }

        public String getClusterName() {
            return clusterName;
        }

        public String getName() {
            return name;
        }

        @Override
        public String toString() {
            return LoadbalancerUtil.getServerString(clusterName, name);
        }

        @Override
        public int hashCode() {
            final int prime = 31;
            int result = 1;
            result = prime * result
                    + ((clusterName == null) ? 0 : clusterName.hashCode());
            result = prime * result + ((name == null) ? 0 : name.hashCode());
            return result;
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj)
                return true;
            if (obj == null)
                return false;
            if (getClass() != obj.getClass())
                return false;
            final ServerInstance other = (ServerInstance) obj;
            if (clusterName == null) {
                if (other.clusterName != null)
                    return false;
            } else if (!clusterName.equals(other.clusterName))
                return false;
            if (name == null) {
                if (other.name != null)
                    return false;
            } else if (!name.equals(other.name))
                return false;
            return true;
        }
    }
    
    class DCUIGMSEventListenerImpl extends GMSEventListener{

        public DCUIGMSEventListenerImpl(String clusterName) throws GMSEventListenerException {
            super(clusterName);
        }

        public void onRecovery(String clusterName, String instanceName) {
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "clb.recieved_recovery_notification_for_instance",
                    new Object[]{instanceName, clusterName});
            }

            instanceHealthy.put(instanceName, true);
            actualHash.addNode(new ServerInstance(clusterName, instanceName));

            if (logger.isLoggable(Level.FINEST)) {
                logger.log(Level.FINEST, "Setting up actualHash after recovery of " + clusterName + "/" + instanceName);
            }
            actualHash.setup();
        }

        public void onFailure(String clusterName, String instanceName) {
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "clb.recieved_failure_notification_for_instance",
                    new Object[]{instanceName, clusterName});
            }

            instanceHealthy.put(instanceName, false);
            actualHash.removeNode(new ServerInstance(clusterName, instanceName));

            if (logger.isLoggable(Level.FINEST)) {
                logger.log(Level.FINEST, "Setting up actualHash after failure of " + clusterName + "/" + instanceName);
            }
            actualHash.setup();
        }

        public void onDisable(String clusterName, String instanceName) {
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "clb.recieved_disable_notification_for_instance",
                    new Object[]{instanceName, clusterName});
            }

            actualHash.removeNode(new ServerInstance(clusterName, instanceName));

            if (logger.isLoggable(Level.FINEST)) {
                logger.log(Level.FINEST, "Setting up actualHash after disable of " + clusterName + "/" + instanceName);
            }
            actualHash.setup();
        }

        public void onEnable(String clusterName, String instanceName) {
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "clb.recieved_enable_notification_for_instance",
                    new Object[]{instanceName, clusterName});
            }

            if (instanceHealthy.get(instanceName)) {
                actualHash.addNode(new ServerInstance(clusterName, instanceName));
                if (logger.isLoggable(Level.FINEST)) {
                    logger.log(Level.FINEST, "Setting up actualHash after enable of " + clusterName + "/" + instanceName);
                }
                actualHash.setup();
            }
        }
    }
}
