/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.jvnet.glassfish.comms.clb.core.sip;

import java.util.ListIterator;
import java.util.StringTokenizer;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.servlet.sip.ServletParseException;
import javax.servlet.sip.SipFactory;
import javax.servlet.sip.SipURI;

import org.jvnet.glassfish.comms.clb.core.common.chr.HashKeyExtractor;
import org.jvnet.glassfish.comms.clb.core.common.chr.StickyHashKeyExtractor;
import org.jvnet.glassfish.comms.util.LogUtil;

import com.ericsson.ssa.sip.AddressImpl;
import com.ericsson.ssa.sip.Header;
import com.ericsson.ssa.sip.RemoteLockRuntimeException;
import com.ericsson.ssa.sip.SipApplicationSessionImpl;
import com.ericsson.ssa.sip.SipServletMessageImpl;
import com.ericsson.ssa.sip.SipServletRequestImpl;
import com.ericsson.ssa.sip.SipServletResponseImpl;
import com.ericsson.ssa.sip.SipSessionBase;
import com.ericsson.ssa.sip.ViaImpl;


/**
 * This class implements the SIP load balancer routing logic. It implements both
 * the front-end logic (incoming requests and responses) and back-end logic
 * (both incoming and outgoing requests and responses)
 */
public class SipLoadBalancerBackend {
    private static Logger logger = LogUtil.CLB_LOGGER.getLogger();
    private HashKeyExtractor hashKeyExtractor;
    private Socket localSipTcpSocket; // Socket for traffic between F-E to B-E and responses

    /**
     * Creates an instance and associates it with the specified hash key
     * extractor and server instance lookup.
     *
     * @param hashKeyExtractor the hash key extractor
     */
    public SipLoadBalancerBackend(HashKeyExtractor hashKeyExtractor,
        Socket localSipTcpSocket) {
        this.hashKeyExtractor = hashKeyExtractor;
        this.localSipTcpSocket = localSipTcpSocket;
    }

    /**
     * Handle a request that as been received by this instance. It shall either
     * be served by this instance, or be proxied to another instance.
     *
     * @param req the request; the request may be modified (headers added or
     *                changed)
     */
    public void handleIncomingRequest(SipServletRequestImpl req) {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Handle incoming request: " + req);
        }

        if (req.getBeKey() == null) {
            // The SipLoadBalancerIncomingHandler was not present (this is a
            // pure backend). Extract BEKey that was extracted by front-end
            String beKey = req.getHeader(StickyHashKeyExtractor.PROXY_BEKEY_HEADER);

            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE, "Extracted BE key: " + beKey);
            }

            if (beKey != null) {
                req.setBeKey(beKey);
            }
        }

        // Unconditional remove of Proxy-Bekey header
        req.removeHeader(StickyHashKeyExtractor.PROXY_BEKEY_HEADER);
    }

    /**
     * Handle a request that as been created or proxied by this instance.
     *
     * @param request the request; the request may be modified (headers added or
     *                changed)
     * @return a possible new connection; if null continue with the connection
     *         in the transaction stack.
     * @throws SipRoutingException thrown in case request was malformed
     */
    public Connection handleOutgoingRequest(SipServletRequestImpl request)
        throws SipRoutingException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Handle outgoing request: " + request);
        }

        // Check if hash key has been previously saved
        String hashkey = null;
        try {
            hashkey = getHashKey(request);
        } catch (RemoteLockRuntimeException e) {
            // Should never happen
            if (logger.isLoggable(Level.WARNING)) {
                logger.log(Level.WARNING, "clb.sip.warning.remote_lock");
            }
            throw new SipRoutingException(e);
        }

        if (hashkey == null) {
            // New request, save hash key
            hashkey = getHashKeyExtractor().getHashKey(request);

            SipSessionBase ss = request.getSessionImpl();

            if (ss != null) {
                SipApplicationSessionImpl sas;
                sas = request.getApplicationSessionImpl();

                if (sas != null && sas.isValid()) {
                    if (sas.getBeKey() == null) {
                        sas.setBeKey(hashkey);
                    }
                }
            }

            request.setBeKey(hashkey);
        }

        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Hashkey: " + hashkey);
        }

        // Set BERoute parameter on topmost Via
        ViaImpl via = getTopVia(request);
        via.setParameter(SipLoadBalancerIncomingHandler.BE_ROUTE_PARAM,
            encodeBeRoute(getLocalSipTcpSocket()));

        // Check if the application has set Contact
        if (request.isContactIndicated()) {
            // Check if this request is initiated from this instance

			Header contactHeader = request.getRawHeader(Header.CONTACT);
			if (contactHeader != null) {
				// There is a contact, we need to add the BE-KEY to it
				try {
					AddressImpl addressValue = (AddressImpl) contactHeader
							.getAddressValue();
					if (addressValue != null) {
						// Fetch current read-only status
						boolean readOnly = addressValue.isReadOnly();
						// Allow writes on Address object
						addressValue.setReadOnly(false);
						StickyHashKeyExtractor.encodeHashKeyToBeKey(
								addressValue.getURI(), hashkey);
						// Reset read-only status
						addressValue.setReadOnly(readOnly);
					}
				} catch (ServletParseException e) {
					throw new SipRoutingException("Malformed Contact", e);
				}
			}

            replaceTopVia(request, via);

            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Was initial, altered request: " + request);
            }

            return null;
        } else if (request.isRecordRouteIndicated()) {
            // This request was proxied, save ID of incoming connection.
            String encodedConnection = getEncondedConnection(request);

            if (encodedConnection != null) {
                via.setParameter(SipLoadBalancerIncomingHandler.CONNID_PARAM,
                    getEncondedConnection(request));
            }

            Header rrHeader = request.getRawHeader(Header.RECORD_ROUTE);
            AddressImpl rrAddr;

            try {
                rrAddr = (AddressImpl) rrHeader.getAddressValue();
            } catch (ServletParseException e) {
                throw new SipRoutingException("Malformed record-Route", e);
            }

            // The R-R was added by the application, thus we shall set 'bekey'
            // Fetch current read-only status
            boolean readOnly = rrAddr.isReadOnly();
            // Allow writes on Address object
            rrAddr.setReadOnly(false);
            StickyHashKeyExtractor.encodeHashKeyToBeKey(rrAddr.getURI(), hashkey);
            // Reset read-only status
            rrAddr.setReadOnly(readOnly);

            replaceTopVia(request, via);

            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Was proxied, altered request: " + request);
            }

            return null;
        } else {
            // This request was proxied but not Record-Routed, save ID of incoming 
            // connection so that response can travel back.
            String encodedConnection = getEncondedConnection(request);

            if (encodedConnection != null) {
                via.setParameter(SipLoadBalancerIncomingHandler.CONNID_PARAM,
                    encodedConnection);
            }

            replaceTopVia(request, via);

            return null;
        }
    }

    private String getEncondedConnection(SipServletRequestImpl request) {
        Connection c = (Connection) request.getAttribute(SipLoadBalancerIncomingHandler.CONNID_ATTR);

        if (c != null) {
            return c.getEncodedValue();
        }

        return null;
    }

    /**
     * Handle a response that has been created or proxied by this instance.
     *
     * @param response the response; the response may be modified (headers added
     *                or changed)
     * @return a possible new connection; if null continue with the connection
     *         in the transaction stack.
     * @throws SipRoutingException thrown if message was malformed
     */
    public Connection handleOutgoingResponse(SipServletResponseImpl response)
        throws SipRoutingException {
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Handle outgoing response: " + response);
        }

        String hashkey = null;
        try {
            hashkey = getHashKey(response);
            SipApplicationSessionImpl sas;
            if (hashkey != null && response.getSessionImpl() != null && 
                    (sas = response.getApplicationSessionImpl()) != null && sas.isValid() && sas.getBeKey() == null) {
                sas.setBeKey(hashkey);
            }
        } catch (RemoteLockRuntimeException e) {
            response.setStatus(500);
            response.setHeader(Header.RETRY_AFTER, "5");
        }

        if (hashkey != null) {
            if (response.isContactIndicated()) {
                // This is a UAS.
                Header contactHeader = response.getRawHeader(Header.CONTACT);

                if (contactHeader != null) {
                    AddressImpl contactAddress;

                    try {
                        contactAddress = (AddressImpl) (contactHeader.getAddressValue());
                    } catch (ServletParseException e) {
                        throw new SipRoutingException("Malformed Contact", e);
                    }

                    SipURI contactUri = ((SipURI) contactAddress.getURI());

                    contactAddress = ((AddressImpl) contactAddress.clone(true, true));
                    contactUri = ((SipURI) contactUri.clone());
                    StickyHashKeyExtractor.encodeHashKeyToBeKey(contactUri, hashkey);
                    // Allow writes on cloned Address object
                    contactAddress.setReadOnly(false);
                    contactAddress.setURI(contactUri);
                    contactHeader.setReadOnly(false);
                    contactHeader.removeValues();
                    contactHeader.setValue(contactAddress.toString(), true);
                    contactHeader.setReadOnly(true);
                }
            } else if (response.getMethod().equalsIgnoreCase("INVITE") && response.getStatus() >= 300){
                // This is a non-2xx response to an INVITE, set bekey in the To-header
                Header toHeader = response.getRawHeader(Header.TO);

                if (toHeader != null) {
                    AddressImpl toAddress;

                    try {
                        toAddress = (AddressImpl) (toHeader.getAddressValue());
                    } catch (ServletParseException e) {
                        throw new SipRoutingException("Malformed To-header", e);
                    }

                    SipURI toUri = ((SipURI) toAddress.getURI());

                    toAddress = ((AddressImpl) toAddress.clone(true, true));
                    toUri = ((SipURI) toUri.clone());
                    StickyHashKeyExtractor.encodeHashKeyToBeKey(toUri, hashkey);
                    // Allow writes on cloned Address object
                    toAddress.setReadOnly(false);
                    toAddress.setURI(toUri);
                    toHeader.setReadOnly(false);
                    toHeader.removeValues();
                    toHeader.setValue(toAddress.toString(), true);
                    toHeader.setReadOnly(true);
                }
            }
        }

        // Get the connection that was extracted from the Via in the (possible) incoming response
        Connection connection = (Connection) response.getAttribute(SipLoadBalancerIncomingHandler.CONNID_ATTR);

        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "Altered response: " + response);
        }

        return connection;
    }

    private String getHashKey(SipServletMessageImpl msg) {
        String hashkey = getBeKeyFromSession(msg);

        if (hashkey == null) {
            // Could not get it from session, try the message
            hashkey = msg.getBeKey();
        }

        return hashkey;
    }

    private String getBeKeyFromSession(SipServletMessageImpl msg) {
        String hashkey = null;
        SipSessionBase ss = msg.getSessionImpl();

        if (ss != null) {
            SipApplicationSessionImpl sas = ss.getApplicationSessionImpl();
            if (sas != null) hashkey = sas.getBeKey();
        }

        return hashkey;
    }

    private ViaImpl getTopVia(SipServletMessageImpl msg) {
        return new ViaImpl(msg.getRawHeader(Header.VIA).getValue());
    }

    private void replaceTopVia(SipServletMessageImpl msg, ViaImpl newTopVia) {
        Header viaHeader = msg.getRawHeader(Header.VIA);
        viaHeader.setReadOnly(false);

        ListIterator<String> viaIter = viaHeader.getValues();

        while (viaIter.hasNext()) {
            viaIter.next();
            viaIter.remove();

            break;
        }

        viaHeader.setValue(newTopVia.toString(), true);
        viaHeader.setReadOnly(true);
    }

    private HashKeyExtractor getHashKeyExtractor() {
        return hashKeyExtractor;
    }

    private String encodeBeRoute(Socket address) {
        String routeString = '"' + address.getHostName() + ":" +
            address.getPort() + '"';

        return routeString;
    }

    static Socket decodeBeRoute(String beRouteParameter)
        throws SipRoutingException {
        beRouteParameter = beRouteParameter.substring(1,
                beRouteParameter.length() - 1);

        StringTokenizer st = new StringTokenizer(beRouteParameter, ":", false);
        String[] addrArr = new String[2];
        int i = 0;

        while (st.hasMoreTokens() && (i < addrArr.length)) {
            addrArr[i++] = st.nextToken();
        }

        // Now addrArr[0]=IP address and
        // addrArr[1]=port
        if (i == addrArr.length) {
            try {
                return new Socket(addrArr[0], Integer.parseInt(addrArr[1]));
            } catch (NumberFormatException e) {
                throw new SipRoutingException("Malformed 'beroute' parameter", e);
            }
        }

        throw new SipRoutingException("Could not decode 'beroute' parameter");
    }

    private Socket getLocalSipTcpSocket() {
        return localSipTcpSocket;
    }
}
