/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.jvnet.glassfish.comms.clb.core.util;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.jvnet.glassfish.comms.util.LogUtil;


/**
 * A consistent hash function. Maps arbitrary objects to one of a finite set of
 * nodes.
 * <p>
 * Usage details:
 * <p>
 * <ul>
 * <li>{@link #setup()} must be called after that nodes has been added or
 * removed.
 * <li>{@link #get(Object)} is used to get the node to which an arbitrary
 * object maps.
 * </ul>
 *
 * For more info, see, e.g.,
 * {@linkplain http://en.wikipedia.org/wiki/Consistent_hashing}
 */
public class ConsistentHash<K, V> {
    private static Logger logger = LogUtil.CLB_LOGGER.getLogger();
    private int pointsPerNode;
    private HashMap<V, Integer> nodes = new HashMap<V, Integer>();
    private int nPoints;
    private Point<V>[] points;
    private MessageDigest md5;
    private String name;

    /**
     * Creates the hash with the specified points per node.
     * @param pointsPerNode points per node
     */
    public ConsistentHash(int pointsPerNode, String name) {
        this.pointsPerNode = pointsPerNode;
        this.name = name;

        try {
            md5 = MessageDigest.getInstance("md5");
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("can't get md5 instance");
        }
    }

    
    @Override
    public String toString() {
        return super.toString() + "(" + name + ")" + "=" + nodes.keySet();
    }


    /**
     * Add a node to the mapped set.
     * <p>
     * <b>NOTE!</b> {@link #setup()} must be called before using the consistent
     * hash after one or more node(s) has been added.
     *
     * @param node the node
     * @param factor multiplication factor for points per node
     */
    public void addNode(V node, double factor) {
        nodes.put(node, new Integer((int) ((factor * pointsPerNode) + .5)));
        points = null; // crash and burn if user forgets setup()
    }

    /**
     * Add a node to the mapped set with factor = 1.0.
     * <p>
     * <b>NOTE!</b> {@link #setup()} must be called before using the consistent
     * hash after one or more node(s) has been added.
     *
     * @param node the node
     */
    public void addNode(V node) {
        addNode(node, 1.0);
    }

    /**
     * Remove node from the mapped set.
     * <p>
     * <b>NOTE!</b> {@link #setup()} must be called before using the consistent
     * hash after one or more node(s) has been removed.
     *
     * @param node the node to remove
     */
    public void removeNode(V node) {
        nodes.remove(node);
        points = null; // crash and burn if user forgets setup()
    }

    public void setup() {
        nPoints = 0;

        for (Iterator<Integer> it = nodes.values().iterator(); it.hasNext();)
            nPoints += (it.next()).intValue();

        points = new Point[nPoints];

        int point = 0;

        for (Iterator<V> keys = nodes.keySet().iterator(); keys.hasNext();) {
            Object node = keys.next();
            int n = nodes.get(node).intValue();

            for (int i = 0; i < n; i++)
                points[point++] = new Point(md5hash(node.toString() + "/" + i),
                        node);
        }

        Arrays.sort(points);

        if (logger.isLoggable(Level.FINEST)) {
            logger.log(Level.FINEST, "Setup done: Current state: " + this);
        }
    }

    /**
     * Gets the node that the specified value maps to.
     *
     * @param value
     * @return the node that the specified value maps to
     */
    public V get(K value) {
        int hashValue = hash(value.toString());

        if (points.length == 0) {
            return null;
        }

        if (hashValue < points[0].start) {
            return points[nPoints - 1].node;
        }

        int lo = 0;
        int hi = nPoints;

        while (lo < (hi - 1)) {
            int mid = (lo + hi) / 2;

            if (points[mid].start > hashValue) {
                hi = mid;
            } else {
                lo = mid;
            }
        }

        return points[lo].node;
    }

    private int hash(String s) {
        int h = s.hashCode();

        // All the "entropy" is in the low bits; shift them up...
        return (h << 28) ^ (h << 24) ^ (h << 16) ^ h;
    }

    private int md5hash(String s) {
        md5.reset();

        byte[] digest = md5.digest(s.getBytes());

        return ((digest[12] & 0xff) << 24) | ((digest[13] & 0xff) << 16) |
        ((digest[14] & 0xff) << 8) | (digest[15] & 0xff);
    }

    private static class Point<V> implements Comparable<Point<V>> {
        int start;
        V node;

        Point(int start, V node) {
            this.start = start;
            this.node = node;
        }

        public int compareTo(Point<V> o) {
            int start2 = o.start;

            if (start < start2) {
                return -1;
            }

            if (start > start2) {
                return 1;
            }

            return 0;
        }

        @Override
        public String toString() {
            return "{" + node + "," + start + "}";
        }
    }
}
