/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.jvnet.glassfish.comms.admin.config;

import com.ericsson.ssa.config.Config;
import com.ericsson.ssa.config.ConfigFactory;
import com.ericsson.ssa.config.event.ConfigAddEvent;
import com.ericsson.ssa.config.event.ConfigRemoveEvent;
import com.ericsson.ssa.config.event.ConfigUpdateEvent;

import com.sun.enterprise.admin.event.AdminEventListenerRegistry;
import com.sun.enterprise.config.ConfigBean;
import com.sun.enterprise.config.ConfigException;
import com.sun.enterprise.server.ApplicationServer;

import org.jvnet.glassfish.comms.util.LogUtil;

import org.netbeans.modules.schema2beans.BaseBean;

import java.util.HashMap;
import java.util.Map;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipContainerEvent;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipContainerEventListener;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipConnectionPoolEvent;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipConnectionPoolEventListener;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipKeepAliveEvent;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipKeepAliveEventListener;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipLinkEvent;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipLinkEventListener;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipListenerEvent;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipListenerEventListener;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipProtocolEvent;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipProtocolEventListener;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipRequestProcessingEvent;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipRequestProcessingEventListener;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipSslEvent;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipSslEventListener;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipServiceEvent;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipServiceEventListener;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipTimersEvent;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipTimersEventListener;
import org.netbeans.modules.schema2beans.BaseBean;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipAccessLogEvent;
import org.jvnet.glassfish.comms.admin.event.extensions.sip.SipAccessLogEventListener;


/**
 * Maps the ConfigBean model as defined by the domain.dtd with the sip container
 * internal model. Note that the mapping is historically not 121.
 * The intention is to grow to a 121 mapping. The intention is to keep
 * an internal model to allow proper dependencies between the modules.
 */
public class ConfigAdapter {
    private static final String DEFAULT_MAPPER = "DEFAULT_MAPPER";
    private Map<String, ConfigBean2InternalMapper> mappers;
    private Logger log = LogUtil.SIP_LOGGER.getLogger();
    private SimpleConfig simpleConfig;

    public static final ConfigAdapter INSTANCE = new ConfigAdapter();
    private boolean initialized;
    private boolean registered;

    
    /**
     * Creates a new instance of ConfigAdapter
     */
    private ConfigAdapter() {
    }

    public static final ConfigAdapter instance() {
        return INSTANCE;
    }

//Lifecycle support
    public void init() {
        if (!initialized) {
            mappers = discoverConfigBean2InternalMappers();
            simpleConfig = initInternalConfig();
            initialized=true;
        }
    }
    
    public void startup() { 
        init();
        configureSipContainer(mappers);
    }

    public void register() {
        if (!registered) {
            registerEventListeners(mappers);
            registered=true;
        }
    }

    public void configureSpecificContext(ConfigBean bean, String ctx) {
        //This is not ideal yet, need to know how ctx is mapped to internal, not wanted ofcourse!
        //eventMask = ctx + one child
        //Capturing Add and Update events because ElementProperty related events
        //may register as Update event.
        simpleConfig.captureEvents("/" + ctx + "(([^/]+)([/]?))?",
            new String[] { ConfigAddEvent.TYPE, ConfigUpdateEvent.TYPE });
        configureSipContainer(bean, ctx, ConfigFactory.getConfig(), mappers);
        try {
            simpleConfig.processEvents();
        } finally {
            if (log.isLoggable(Level.INFO)) {
                log.log(Level.INFO, "Updated internal Sip Container config: "+
                        ConfigFactory.getConfig());
            }
        }
    }

    void reconfigureSpecificContext(ConfigBean bean, String ctx) {
        //This is not ideal yet, need to know how ctx is mapped to internal, not wanted ofcourse!
        //eventMask = ctx + any decending child
        simpleConfig.captureEvents("/" + ctx + "(([^/]+)([/]?))*",
            new String[] { ConfigAddEvent.TYPE, ConfigUpdateEvent.TYPE });
        reconfigureSipContainer(bean, ctx, ConfigFactory.getConfig(), mappers);
        try {
            simpleConfig.processEvents();
        } finally {
            if (log.isLoggable(Level.INFO)) {
                log.log(Level.INFO, "Updated internal Sip Container config: "+
                        ConfigFactory.getConfig());
            }
        }
    }

    void unconfigureSpecificContext(ConfigBean bean, String ctx) {
        //This is not ideal yet, need to know how ctx is mapped to internal, not wanted ofcourse!
        //eventMask = ctx + one child
        //Capturing Remove and Update events because ElementProperty related events
        //may register as Update event.
        simpleConfig.captureEvents("/" + ctx + "(([^/]+)([/]?))?",
            new String[] { ConfigRemoveEvent.TYPE, ConfigUpdateEvent.TYPE });
        unconfigureSipContainer(bean, ctx, ConfigFactory.getConfig(), mappers);
        try {
            simpleConfig.processEvents();
        } finally {
            if (log.isLoggable(Level.INFO)) {
                log.log(Level.INFO, "Updated internal Sip Container config: "+
                        ConfigFactory.getConfig());
            }
        }
    }

    /**
     * Discovers the mappers that are active in the current setup.
     * Mappers can be added, i.e. to deal with changes in the domain.dtd
     */
    protected Map<String, ConfigBean2InternalMapper> discoverConfigBean2InternalMappers() {
        Map<String, ConfigBean2InternalMapper> result = 
                new HashMap<String, ConfigBean2InternalMapper>();

        //No proper dynamic extention mechanism yet.
        //Hardcode here your mappers.
        //Key is the element.name of the ConfigBeans the mapper is to be applicable for.
        result.put("SipContainer", new ConfigBean2InternalMapperImpl(SipContainerEvent.eventType, SipContainerEventListener.class, false));
        result.put("StackLayer", new DefaultMapper(true));
        result.put("ElementProperty", new ElementPropertyMapper());
        result.put("SipService", new ConfigBean2InternalMapperImpl(SipServiceEvent.eventType, SipServiceEventListener.class, false));
        result.put("AccessLog", new ConfigBean2InternalMapperImpl(SipAccessLogEvent.eventType, SipAccessLogEventListener.class, false));        
        result.put("SipListener", new ConfigBean2InternalMapperImpl(SipListenerEvent.eventType, SipListenerEventListener.class, true));        
        result.put("Ssl", new ConfigBean2InternalMapperImpl(SipSslEvent.eventType, SipSslEventListener.class, false));        
        result.put("RequestProcessing", new ConfigBean2InternalMapperImpl(SipRequestProcessingEvent.eventType, SipRequestProcessingEventListener.class, false));        
        result.put("KeepAlive", new ConfigBean2InternalMapperImpl(SipKeepAliveEvent.eventType, SipKeepAliveEventListener.class, false));        
        result.put("ConnectionPool", new ConfigBean2InternalMapperImpl(SipConnectionPoolEvent.eventType, SipConnectionPoolEventListener.class, false));        
        result.put("SipProtocol", new ConfigBean2InternalMapperImpl(SipProtocolEvent.eventType, SipProtocolEventListener.class, false));       
        result.put("SipLink", new ConfigBean2InternalMapperImpl(SipLinkEvent.eventType, SipLinkEventListener.class, false));
        result.put("SipTimers", new ConfigBean2InternalMapperImpl(SipTimersEvent.eventType, SipTimersEventListener.class, false));
        result.put(DEFAULT_MAPPER, new DefaultMapper());        
        
        return result;
    }

    protected void registerEventListeners(
        Map<String, ConfigBean2InternalMapper> mappers) {
        for (ConfigBean2InternalMapper mapper : mappers.values()) {
            if (mapper.requiresEventListerenerRegistration()) {

                if (log.isLoggable(Level.FINEST)) {
                    log.log(Level.FINEST, "Registrering EventListener "+mapper.getEventListenerClass()+" for "+mapper.getEventType());
                }
                
                try {
                AdminEventListenerRegistry.registerEventListener(mapper.getEventType(),
                    mapper.getEventListenerImpl(this),
                    mapper.getEventListenerClass());
                } catch (IllegalArgumentException e) {
                    //This should not happen (after issue 116).
                    if (log.isLoggable(Level.FINEST)) { 
                       log.log(Level.FINEST, "Type already registered, just adding listener.");
                    }
                    //Note: Overrides any previous registered listener.
                    AdminEventListenerRegistry.addEventListener(mapper.getEventType(),
                        mapper.getEventListenerImpl(this));
                }
            }
        }
    }
    
    protected void configureSipContainer(
        Map<String, ConfigBean2InternalMapper> mappers) {
        //Approach is to recursivly travers the complete tree and map it to the internal config starting from the
        // for the sip container root nodes.

        try {
            configureSipContainer(SipBeansFactory.getSipContainerBean(), "",
                simpleConfig, mappers);
        } catch (ConfigException ex) {
            if (log.isLoggable(Level.WARNING)) {
                log.log(Level.WARNING, "config_adaptation_failure_continue", "SipContainer");
                log.log(Level.WARNING, ex.getMessage(), ex);
            }
        }

        try {
            configureSipContainer(SipBeansFactory.getCurrentConfig().getSipService(), "", 
                simpleConfig, mappers);
        } catch (ConfigException ex) {
            if (log.isLoggable(Level.WARNING)) {
                log.log(Level.WARNING, "config_adaptation_failure_continue", "SipService");
                log.log(Level.WARNING, ex.getMessage(), ex);
            }
        }
        if (log.isLoggable(Level.FINE)) {
            log.fine("Using internal Sip Container config: " + simpleConfig);
        }
    }

    protected void configureSipContainer(ConfigBean bean, String accumCtx,
        Config config, Map<String, ConfigBean2InternalMapper> mappers) {
        if (log.isLoggable(Level.FINE)) {
            log.log(Level.FINE,
                "Configuring bean " + bean.name() + " to context " + accumCtx);
            dumpBean(bean, Level.FINEST);
        }

        String ctx = accumCtx + bean.name();

        ConfigBean2InternalMapper mapper = mappers.get(bean.name());

        if (mapper == null) {
            mapper = mappers.get(DEFAULT_MAPPER);
        }

        String[] ctxHolder = new String[] { ctx };
        BaseBean[] childeren2Traverse = mapper.map2Internal(bean, ctxHolder,
                config);

        if (childeren2Traverse != null) {
            for (BaseBean baseBean : childeren2Traverse) {
                configureSipContainer((ConfigBean) baseBean,
                    ctxHolder[0] + "/", config, mappers);
            }
        }
    }

    protected void reconfigureSipContainer(ConfigBean bean, String accumCtx,
        Config config, Map<String, ConfigBean2InternalMapper> mappers) {

        if (log.isLoggable(Level.FINE)) {
            log.log(Level.FINE,
                "Reconfiguring bean " + bean.name() + " to context " +
                accumCtx);
            dumpBean(bean, Level.FINEST);
        }

        String ctx = accumCtx + bean.name();

        ConfigBean2InternalMapper mapper = mappers.get(bean.name());

        if (mapper == null) {
            mapper = mappers.get(DEFAULT_MAPPER);
        }

        String[] ctxHolder = new String[] { ctx };
        BaseBean[] childeren2Traverse = mapper.remap2Internal(bean, ctxHolder,
                config);

        if (childeren2Traverse != null) {
            for (BaseBean baseBean : childeren2Traverse) {
                reconfigureSipContainer((ConfigBean) baseBean,
                    ctxHolder[0] + "/", config, mappers);
            }
        }
    }

    protected void unconfigureSipContainer(ConfigBean bean, String accumCtx,
        Config config, Map<String, ConfigBean2InternalMapper> mappers) {

        if (log.isLoggable(Level.FINE)) {
            log.log(Level.FINE,
                "Unconfiguring bean " + bean.name() + " from context " +
                accumCtx);
            dumpBean(bean, Level.FINEST);
        }

        String ctx = accumCtx + bean.name();

        ConfigBean2InternalMapper mapper = mappers.get(bean.name());

        if (mapper == null) {
            mapper = mappers.get(DEFAULT_MAPPER);
        }

        String[] ctxHolder = new String[] { ctx };
        BaseBean[] childeren2Traverse = mapper.unmap2Internal(bean, ctxHolder,
                config);

        if (childeren2Traverse != null) {
            for (BaseBean baseBean : childeren2Traverse) {
                unconfigureSipContainer((ConfigBean) baseBean,
                    ctxHolder[0] + "/", config, mappers);
            }
        }
    }

    private SimpleConfig initInternalConfig() {
        String asInstanceName = ApplicationServer.getServerContext()
                                                 .getInstanceName();

        if (log.isLoggable(Level.FINEST)) {
            log.log(Level.FINEST,
                "AS Instance Name used for default context: " + asInstanceName);
        }

        SimpleConfig config = new SimpleConfig(asInstanceName);
        ConfigFactory.instance().registerConfig(config);

        return config;
    }

    private void dumpBean(ConfigBean bean, Level lvl) {
        StringBuffer buf = new StringBuffer(bean.name());

        for (String attr : bean.getAttributeNames()) {
            buf.append("\n ");
            buf.append(attr);
            buf.append(" = ");
            buf.append(bean.getAttributeValue(attr));
        }

        if (log.isLoggable(lvl)) {
            log.log(lvl, "Bean details: " + buf.toString());
        }
    }

    class DefaultMapper extends BaseMapper {
        public DefaultMapper() {
            super();
        }

        public DefaultMapper(boolean isMultiple) {
            super(isMultiple);
        }

        @Override
        public boolean requiresEventListerenerRegistration() {
            return false;
        }
    }

    /* ElementProperties are mapped differently than regular elements.
     * Eventhough in the ConfigBean model each property is represented as
     * a single ElementProperty element with a name and value attribute in
     * The container Config model all properties are mapped to a single 
     * ElementProperty node containing all properties as key value pairs.
     * This makes that the BaseMapper behaviour needs to be overriden at some 
     * point.
     */
    class ElementPropertyMapper extends BaseMapper {
        @Override
        public BaseBean[] unmap2Internal(ConfigBean bean, String[] ctxHolder, Config config) {
            String ctx = ctxHolder[0];
            
            if (log.isLoggable(Level.FINE)) {
                log.log(Level.FINE,
                    "Unmapping [ctx = " + ctx + ", bean = " +
                    ((bean != null) ? bean.name() : "null") + "] to config");
            }

            //Remove individual properties.
            //Only clear remove the ElementProperty node when no more properties
            //are on it.
            String name = bean.getAttributeValue("name");            
            
            if (name!=null) {
                config.remove(ctx, name);
            } else {
                //Should never happen.
                if (log.isLoggable(Level.FINEST)) {
                        log.log(Level.FINEST, "Unmapping ElementProperty: encountered non standard property, ignoring.");
                        dumpBean(bean, Level.FINEST);
                }
            }
            
            Map<String, String> keyValues = config.getAll(ctx);
            if (keyValues==null || keyValues.isEmpty()) {
                config.remove(ctx);
            }
            
            return null;
        }
        
        @Override
        protected void mapAttributes2Internal(ConfigBean bean, String ctx, Config config) {
            //Map the name and value attributes to key value pair.
            String name = bean.getAttributeValue("name");            
            String value = bean.getAttributeValue("value");
            
            if (name!=null && value!=null) {
                config.set(ctx, name, value);
            } else {
                //Should never happen.
                if (log.isLoggable(Level.FINEST)) {
                        log.log(Level.FINEST, "Mapping ElementProperty: encountered non standard property, ignoring.");
                        dumpBean(bean, Level.FINEST);
                }
            }
        }
        
        @Override
        protected void remapAttributes2Internal(ConfigBean bean, String ctx, Config config) {
            //No difference. ElementProperty has only the name and value attributes.
            mapAttributes2Internal(bean, ctx, config);
        }                
    }    
}
