/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package org.jvnet.glassfish.comms.replication.dialogmgmt;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.logging.Logger;
import java.util.logging.Level;
import com.sun.logging.LogDomains;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import org.apache.catalina.Container;
import javax.servlet.ServletContext;
import org.apache.catalina.Context;
import org.apache.catalina.Globals;
import org.apache.catalina.Loader;
import org.apache.catalina.session.IOUtilsCaller;
import org.apache.catalina.util.CustomObjectInputStream;
import com.sun.enterprise.ee.web.sessmgmt.BaseFileSync;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationManager;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationState;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationUtil;

/**
 *
 * @author Larry White
 */
public class DialogFragmentFileSync extends BaseFileSync {
    
    public final static String LOGGER_MEM_REP 
        = ReplicationState.LOGGER_MEM_REP;    
    
    /**
     * The logger to use for logging ALL web container related messages.
     */
    private static final Logger _logger 
        = Logger.getLogger(LOGGER_MEM_REP);     
    
    /**
     * The extension to use for serialized DialogFragment filenames.
     */
    protected static final String FILE_DF_ACTIVE_EXT = ".dfactive";
    
    /**
     * The extension to use for serialized DialogFragment replica filenames.
     */
    protected static final String FILE_DF_REPLICA_EXT = ".dfreplica";

    /**
     * The extension to use for serialized DialogFragment replica update filenames.
     */
    protected static final String FILE_DF_REPLICA_UPDATE_EXT = ".dfreplicaupd"; 
    
    /**
     * The descriptive information about this implementation.
     */
    private static final String INFO = "DialogFragmentFileSync/1.0";

    /**
     * Name to register for this Store, used for logging.
     */
    private static final String STORENAME = "dialogFragmentFileSync";

    /**
     * Name to register for the background thread.
     */
    private static final String THREADNAME = "DialogFragmentFileSync";     


    // ------------------------------------------------- Properties

    /** Creates a new instance of DialogFragmentFileSync */
    public DialogFragmentFileSync(ReplicationManager mgr) {
        manager = mgr;
    }
    
    /**
     * Load the DialogFragment artifacts from backup files
     *
     * @exception ClassNotFoundException if a deserialization error occurs
     * @exception IOException if an input/output error occurs
     */
    public void load()
        throws ClassNotFoundException, IOException {           
        // Open an input stream to the specified pathname, if any
        long startTime = System.currentTimeMillis();
        CountDownLatch doneSignal = new CountDownLatch(3);        
        ReplicationDialogFragmentManager mgr = (ReplicationDialogFragmentManager)manager;
        //load df active cache
        SessionLoad sessionLoad 
            = new SessionLoad(this, mgr, doneSignal);
        Thread sessionLoadThread = new Thread(sessionLoad);
        sessionLoadThread.start();

        //load df replica cache
        ReplicatedSessionLoad replicatedSessionLoad 
            = new ReplicatedSessionLoad(this, mgr, doneSignal);
        Thread replicatedSessionLoadThread = new Thread(replicatedSessionLoad);
        replicatedSessionLoadThread.start();        

        //load df replica update cache
        SessionUpdateLoad sessionUpdateLoad 
            = new SessionUpdateLoad(this, mgr, doneSignal);
        Thread sessionUpdateLoadThread = new Thread(sessionUpdateLoad);
        sessionUpdateLoadThread.start();

        try {
            doneSignal.await(30, TimeUnit.SECONDS);
        } catch(InterruptedException ex) {}        
        if (_logger.isLoggable(Level.FINE)) {
            _logger.fine("DialogFragmentFileSync load after a wait: wait time = " + (System.currentTimeMillis() - startTime));
        }        
    }
    
    private ObjectInputStream getObjectInputStream(String fileExtension) throws IOException {    
        // Open an input stream to the specified pathname, if any 
        String fileKey = getFileKey(manager);        
        File file = file(fileKey, fileExtension);        
        if (file == null) {
            return null;
        } else {
            if (_logger.isLoggable(Level.FINE)) {
                _logger.fine("file name for ois: " + file.getCanonicalPath());
            }            
        }
        if (! file.exists()) {
            return (null);
        }

        FileInputStream fis = null;
        ObjectInputStream ois = null;
        Loader loader = null;
        ClassLoader classLoader = null;
        try {
            fis = new FileInputStream(file.getAbsolutePath());
            BufferedInputStream bis = new BufferedInputStream(fis);
            ois = new ObjectInputStream(bis); 
        } catch (FileNotFoundException e) {
            return (null);
        } catch (IOException e) {
            if (ois != null) {
                try {
                    ois.close();
                } catch (IOException f) {
                    ;
                }
                ois = null;
            }
            throw e;
        }
        return ois;
    }

    /**
     * Remove the Session with the specified session identifier from
     * this Store, if present.  If no such Session is present, this method
     * takes no action.
     *
     * @param id Session identifier of the Session to be removed
     *
     * @exception IOException if an input/output error occurs
     */
    public void remove(String id) throws IOException {

        File file = file(id, FILE_DF_ACTIVE_EXT);
        if (file == null) {
            return;
        }       
        file.delete();

    }
    
    /**
     * Save the DialogFragment artifacts to backup files
     *
     * @exception IOException if an input/output error occurs
     */
    public void save() throws IOException {
        long startTime = 0L;
        if (_logger.isLoggable(Level.FINE)) {
            startTime = System.currentTimeMillis();
        }
        ReplicationDialogFragmentManager mgr = (ReplicationDialogFragmentManager)manager;        
        CountDownLatch doneSignal = new CountDownLatch(3);        
        //save df active cache
        SessionSave sessionSave 
            = new SessionSave(this, mgr, doneSignal);
        Thread sessionSaveThread = new Thread(sessionSave);
        sessionSaveThread.start();

        //save df replica cache
        ReplicatedSessionSave replicatedSessionSave 
            = new ReplicatedSessionSave(this, mgr, doneSignal);
        Thread replicatedSessionSaveThread = new Thread(replicatedSessionSave);
        replicatedSessionSaveThread.start();

        //save df replica update cache
        SessionUpdateSave sessionUpdateSave 
            = new SessionUpdateSave(this, mgr, doneSignal);
        Thread sessionUpdateSaveThread = new Thread(sessionUpdateSave);
        sessionUpdateSaveThread.start();

        try {
            doneSignal.await(30, TimeUnit.SECONDS);
        } catch(InterruptedException ex) {}        
        if (_logger.isLoggable(Level.FINE)) {
            _logger.fine("DialogFragmentFileSync save after a wait: wait time = " + (System.currentTimeMillis() - startTime));
        }        
                
        try {
            load();
        } catch (Exception ex) {
            ex.printStackTrace();
        }
        
    }
    
    /**
     * Return a File object representing the pathname to our
     * session persistence directory, if any.  The directory will be
     * created if it does not already exist.
     */
    protected File directory() {

        if (this.directory == null) {
            return (null);
        }
        if (this.directoryFile != null) {
            // NOTE:  Race condition is harmless, so do not synchronize
            return (this.directoryFile);
        }
        File file = new File(this.directory);
        try {
            if (_logger.isLoggable(Level.FINE)) {
                _logger.fine("first file:" + file.getCanonicalPath());
                _logger.fine("first file isAbsolute:" + file.isAbsolute());                
            }            
        } catch (Exception ex) {ex.printStackTrace();}
        if (!file.isAbsolute()) {
            throw new IllegalArgumentException
                ("directory is not correctly defined");
        }
        if (!file.exists() || !file.isDirectory()) {
            file.delete();
            file.mkdirs();
        }
        this.directoryFile = file;
        return (file);

    }
    
    private class SessionSave implements Runnable {
        
        DialogFragmentFileSync _fileSync = null;
        ReplicationDialogFragmentManager _mgr = null;
        CountDownLatch _doneSignal = null;
        
        public SessionSave(DialogFragmentFileSync fileSync, ReplicationDialogFragmentManager mgr, CountDownLatch doneSignal) {
            _fileSync = fileSync;
            _mgr = mgr;
            _doneSignal = doneSignal;
        }        

        public void run() {
            ObjectOutputStream oos = null;
            try {
                oos = _fileSync.getObjectOutputStream(FILE_DF_ACTIVE_EXT);
                _mgr.writeDialogFragments(oos);
            } catch (IOException ex) {
                //log warning
            } finally {
                try {
                    oos.close();
                } catch (IOException ex2) {
                    ;
                }
                _doneSignal.countDown();
            }             
        }
        
    } 
    
    private class ReplicatedSessionSave implements Runnable {
        
        DialogFragmentFileSync _fileSync = null;
        ReplicationDialogFragmentManager _mgr = null;
        CountDownLatch _doneSignal = null;
        
        public ReplicatedSessionSave(DialogFragmentFileSync fileSync, ReplicationDialogFragmentManager mgr, CountDownLatch doneSignal) {
            _fileSync = fileSync;
            _mgr = mgr;
            _doneSignal = doneSignal;
        }        

        public void run() {
            ObjectOutputStream oos = null;
            try {
                oos = _fileSync.getObjectOutputStream(FILE_DF_REPLICA_EXT);
                writeReplicatedSessions(_mgr.getReplicatedDialogFragments(), oos);
            } catch (IOException ex) {
                //log warning
            } finally {
                try {
                    oos.close();
                } catch (IOException ex2) {
                    ;
                }
                _doneSignal.countDown();
            }             
        }
        
    } 
    
    private class SessionUpdateSave implements Runnable {
        
        DialogFragmentFileSync _fileSync = null;
        ReplicationDialogFragmentManager _mgr = null;
        CountDownLatch _doneSignal = null;
        
        public SessionUpdateSave(DialogFragmentFileSync fileSync, ReplicationDialogFragmentManager mgr, CountDownLatch doneSignal) {
            _fileSync = fileSync;
            _mgr = mgr;
            _doneSignal = doneSignal;
        }        

        public void run() {
            ObjectOutputStream oos = null;
            try {
                oos = _fileSync.getObjectOutputStream(FILE_DF_REPLICA_UPDATE_EXT);
                _mgr.writeReplicatedDialogFragmentUpdates(oos);
            } catch (IOException ex) {
                //log warning
            } finally {
                try {
                    oos.close();
                } catch (IOException ex2) {
                    ;
                }
                _doneSignal.countDown();
            }             
        }
        
    } 
    
    private class SessionLoad implements Runnable {
        
        DialogFragmentFileSync _fileSync = null;
        ReplicationDialogFragmentManager _mgr = null;
        CountDownLatch _doneSignal = null;
        
        public SessionLoad(DialogFragmentFileSync fileSync, ReplicationDialogFragmentManager mgr, CountDownLatch doneSignal) {
            _fileSync = fileSync;
            _mgr = mgr;
            _doneSignal = doneSignal;
        }        

        public void run() {
            ObjectInputStream ois = null;
            try {
                ois = _fileSync.getObjectInputStream(FILE_DF_ACTIVE_EXT);
                _mgr.readDialogFragments(ois);
            } catch (IOException ex) {
                //log warning
            } catch (ClassNotFoundException ex2) {
                //log warning
            } finally {
                try {
                    ois.close();
                } catch (IOException ex2) {
                    ;
                }
                _doneSignal.countDown();
            }             
        }
        
    }
    
    private class ReplicatedSessionLoad implements Runnable {
        
        DialogFragmentFileSync _fileSync = null;
        ReplicationDialogFragmentManager _mgr = null;
        CountDownLatch _doneSignal = null;
        
        public ReplicatedSessionLoad(DialogFragmentFileSync fileSync, ReplicationDialogFragmentManager mgr, CountDownLatch doneSignal) {
            _fileSync = fileSync;
            _mgr = mgr;
            _doneSignal = doneSignal;
        }        

        public void run() {
            ObjectInputStream ois = null;
            try {
                ois = _fileSync.getObjectInputStream(FILE_DF_REPLICA_EXT);
                readReplicatedSessions(_mgr.getReplicatedDialogFragments(), ois);
            } catch (IOException ex) {
                //log warning
            } catch (ClassNotFoundException ex2) {
                //log warning
            } finally {
                try {
                    ois.close();
                } catch (IOException ex2) {
                    ;
                }
                _doneSignal.countDown();
            }             
        }
        
    }       
    
    private class SessionUpdateLoad implements Runnable {
        
        DialogFragmentFileSync _fileSync = null;
        ReplicationDialogFragmentManager _mgr = null;
        CountDownLatch _doneSignal = null;
        
        public SessionUpdateLoad(DialogFragmentFileSync fileSync, ReplicationDialogFragmentManager mgr, CountDownLatch doneSignal) {
            _fileSync = fileSync;
            _mgr = mgr;
            _doneSignal = doneSignal;
        }        

        public void run() {
            ObjectInputStream ois = null;
            try {
                ois = _fileSync.getObjectInputStream(FILE_DF_REPLICA_UPDATE_EXT);
                _mgr.readReplicatedDialogFragmentUpdates(ois);
            } catch (IOException ex) {
                //log warning
            } catch (ClassNotFoundException ex2) {
                //log warning
            } finally {
                try {
                    ois.close();
                } catch (IOException ex2) {
                    ;
                }
                _doneSignal.countDown();
            }             
        }
        
    }    

}
