/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.jvnet.glassfish.comms.replication.dialogmgmt;

import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import com.ericsson.ssa.sip.DialogFragment;
import com.ericsson.ssa.sip.DialogSet;
import com.ericsson.ssa.sip.PersistentDialogFragmentManager;
import com.ericsson.ssa.sip.RemoteLockException;
import com.sun.appserv.ha.spi.BackingStore;
import com.sun.appserv.ha.spi.BackingStoreException;
import com.sun.appserv.ha.spi.BackingStoreFactory;
import com.sun.appserv.ha.spi.BackingStoreRegistry;
import com.sun.appserv.ha.spi.SimpleMetadata;
import com.sun.appserv.util.cache.BaseCache;
import com.sun.enterprise.ee.web.sessmgmt.FederatedListRequestProcessor;
import com.sun.enterprise.ee.web.sessmgmt.FederatedQueryListElement;
import com.sun.enterprise.ee.web.sessmgmt.JxtaBackingStoreFactory;
import com.sun.enterprise.ee.web.sessmgmt.JxtaBackingStoreImpl;
import com.sun.enterprise.ee.web.sessmgmt.JxtaReplicationReceiver;
import com.sun.enterprise.ee.web.sessmgmt.JxtaReplicationSender;
import com.sun.enterprise.ee.web.sessmgmt.PurgeCapable;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationHealthChecker;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationManager;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationMessageRouter;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationResponseRepository;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationState;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationStateRemovedEntry;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationStateQueryResponse;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationUtil;
import com.sun.enterprise.ee.web.sessmgmt.StorePool;
import com.sun.enterprise.ee.web.sessmgmt.StorePoolElement;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationSessionMonitors;
import com.sun.logging.LogDomains;

import java.io.IOException;
import java.security.AccessController;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.security.AccessController;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import org.apache.catalina.Globals;
import org.apache.catalina.security.SecurityUtil;
import org.jvnet.glassfish.comms.replication.sessmgmt.ActivationHelper;
import org.jvnet.glassfish.comms.util.LogUtil;
import com.sun.enterprise.ee.web.sessmgmt.PrivilegedGetReplicationMessageRouter;

/**
 * In-memory replicating DialogFragmentManager.
 * 
 * Get this singleton instance using {@link com.ericsson.ssa.sip.DialogFragmentManager#getInstance()}.
 */
public class ReplicationDialogFragmentManager
        extends PersistentDialogFragmentManager
        implements ReplicationManager, PurgeCapable {
    private static final Logger _logger = LogUtil.SSR_LOGGER.getLogger();

    // Singleton class so app_id is always this constant.
    private static final String APP_ID = "DialogFragmentManager";

    protected static final String MODE_SIP = ReplicationState.MODE_SIP;
    final static String DUPLICATE_IDS_SEMANTICS_PROPERTY 
        = ReplicationState.DUPLICATE_IDS_SEMANTICS_PROPERTY; 
    final static String SUPPRESS_LOAD_ACK_PROPERTY 
        = ReplicationState.SUPPRESS_LOAD_ACK_PROPERTY;
    final static String COMMAND_MAP = "commandmap";
    private static final Level TRACE_LEVEL = Level.FINE;
    protected static int _maxBaseCacheSize = 4096;
    protected static float _loadFactor = 0.75f;
    protected static final int _concurrencyLevel = 100;
    
    private static final Map dialogFragmentCommandMap = new HashMap();
   
    public final static String SAVE_COMMAND 
        = ReplicationState.SAVE_COMMAND;
     public final static String VALVE_SAVE_COMMAND 
        = ReplicationState.VALVE_SAVE_COMMAND;
    public final static String REMOVE_COMMAND 
        = ReplicationState.REMOVE_COMMAND;    
    public final static String UNDEPLOY_COMMAND 
        = ReplicationState.UNDEPLOY_COMMAND;
    public final static String REMOVE_EXPIRED_COMMAND 
        = ReplicationState.REMOVE_EXPIRED_COMMAND;
    public final static String UPDATE_LAST_ACCESS_TIME_COMMAND 
        = ReplicationState.UPDATE_LAST_ACCESS_TIME_COMMAND;
    public final static String MESSAGE_BROADCAST_QUERY 
        = ReplicationState.MESSAGE_BROADCAST_QUERY;
    public final static String MESSAGE_BROADCAST_LOAD_RECEIVED
        = ReplicationState.MESSAGE_BROADCAST_LOAD_RECEIVED;
    
    public final static String SAVE_DIALOG_FRAGMENT_COMMAND = "savedialogfragment";
    public final static String REMOVE_DIALOG_FRAGMENT_COMMAND = "removedialogfragment";    
    public final static String UPDATE_LAST_ACCESS_TIME_DIALOG_FRAGMENT_COMMAND 
        = "updatelastaccesstimedialogfragment";
    public final static String MESSAGE_BROADCAST_QUERY_DIALOG_FRAGMENT 
        = "broadcastfinddialogfragment"; 
    public final static String MESSAGE_BROADCAST_LOAD_RECEIVED_DIALOG_FRAGMENT 
        = "broadcastloadreceiveddialogfragment";
    
    public final static String MESSAGE_BROADCAST_DIALOG_FRAGMENT_ID_QUERY
        = "broadcastfinddialogfragmentids";
    
    public final static String MESSAGE_BROADCAST_ROLLING_UPGRADE_ADVISORY
        = "rollingupgradeadvisory";    

    /**
     * the list of method names that are broadcasts
     */ 
    private static List broadcastMethods 
        = Arrays.asList(MESSAGE_BROADCAST_QUERY_DIALOG_FRAGMENT,
                        MESSAGE_BROADCAST_LOAD_RECEIVED_DIALOG_FRAGMENT,
                        MESSAGE_BROADCAST_DIALOG_FRAGMENT_ID_QUERY,
                        MESSAGE_BROADCAST_ROLLING_UPGRADE_ADVISORY);    
    
    private StorePool dialogFragmentStorePool;

    /**
     * Our cache of replicated DialogFragment ReplicationState objects
     * keyed by id
     */
    protected BaseCache replicatedDialogFragments = new BaseCache();

    /**
     * Our cache of replicated DialogFragement ReplicationStateUpdate
     * objects
     * keyed by id
     */
    protected BaseCache replicatedDialogFragmentUpdates = null;
    
    /**
    * Our cache of monitor objects
    * keyed by id
    */
    protected ReplicationSessionMonitors replicatedSessionMonitors = null;
    
    /**
    * Our cache of dialog fragment removals
    * keyed by id
    */
    protected ConcurrentHashMap replicatedDialogRemovals = null;    
    
    /**
    * Our Replicator instance for DialogFragments (for SimpleMetadata)
    */
    private BackingStore dialogFragmentBackingStore = null; 
    
    protected String _passedInPersistenceType = null;
    protected boolean _duplicateIdsSemanticsAllowed = false;  
  
    private AtomicBoolean isReplicationInitialized = new AtomicBoolean(); //defaults to false
    private AtomicBoolean isReplicationStarted = new AtomicBoolean(); //defaults to false

    private final String instanceName;
   
    static {
        checkSessionCacheProperties();
        initializeCommandMap();
    }

    /**
     * Constructor, which registers the new instance with the
     * ReplicationMessageRouter.
     *
     * @see com.ericsson.ssa.sip.DialogFragmentManager#getInstance()
     */
    public ReplicationDialogFragmentManager() {
        super();
        instanceName = ReplicationUtil.getInstanceName();
    }

    protected static boolean checkSessionCacheProperties() {
        boolean result = false;

        try {
            Properties props = System.getProperties();
            String cacheSize = props.getProperty(
                    "HTTP_SESSION_CACHE_MAX_BASE_CACHE_SIZE");

            if (null != cacheSize) {
                _maxBaseCacheSize = (new Integer(cacheSize).intValue());
            }

            String loadFactor = props.getProperty(
                    "HTTP_SESSION_CACHE_MAX_BASE_LOAD_FACTOR");

            if (null != loadFactor) {
                _loadFactor = (new Float(loadFactor).floatValue());
            }
        } catch (Exception e) {
            //do nothing accept defaults
        }

        return result;
    }
    
    protected static void initializeCommandMap() {
        dialogFragmentCommandMap.put(SAVE_COMMAND, SAVE_DIALOG_FRAGMENT_COMMAND);
        dialogFragmentCommandMap.put(VALVE_SAVE_COMMAND, SAVE_DIALOG_FRAGMENT_COMMAND);
        dialogFragmentCommandMap.put(REMOVE_COMMAND, REMOVE_DIALOG_FRAGMENT_COMMAND);
        dialogFragmentCommandMap.put(UPDATE_LAST_ACCESS_TIME_COMMAND, UPDATE_LAST_ACCESS_TIME_DIALOG_FRAGMENT_COMMAND);
        dialogFragmentCommandMap.put(MESSAGE_BROADCAST_QUERY, MESSAGE_BROADCAST_QUERY_DIALOG_FRAGMENT);   
    }  
    
    /**
     * Initialize ReplicationDialogFragmentManager replication services.
     *
     * @param passedInPersistenceType 
     * @param haStorePoolSize 
     * @param haStorePoolUpperSize 
     * @param haStorePoolPollTime 
     */
    public void initializeReplication(String passedInPersistenceType, int haStorePoolSize, 
                                      int haStorePoolUpperSize, int haStorePoolPollTime) {
        if (isInitialized.get() == false ) {
            IllegalStateException ex = new IllegalStateException("DialogFragmentManager must be initialized prior to initializing replication.");
            _logger.log(Level.SEVERE, "df_init_replication_error");
            _logger.log(Level.SEVERE, ex.getMessage(), ex);
            throw ex;
        }


        // Only initialize once.
        if (isReplicationInitialized.compareAndSet(false, true)) {
            //initialize replicated DialogFragment cache
            replicatedDialogFragments = new BaseCache();
            replicatedDialogFragments.init(_maxBaseCacheSize, _loadFactor,
                                            null);
        
            replicatedDialogFragmentUpdates = new BaseCache();
            replicatedDialogFragmentUpdates.init(_maxBaseCacheSize,
                                                 _loadFactor, null);
            replicatedSessionMonitors = new ReplicationSessionMonitors(_logger, _maxBaseCacheSize, _loadFactor);
            replicatedDialogRemovals = new ConcurrentHashMap(_maxBaseCacheSize, _loadFactor, _concurrencyLevel);

            setPassedInPersistenceType(passedInPersistenceType);
            StorePool dialogFragmentStorePool = 
                    new StorePool(haStorePoolSize, haStorePoolUpperSize, haStorePoolPollTime, 
                                  new DialogFragmentStoreFactory());
            setDialogFragmentStorePool(dialogFragmentStorePool);
            setDialogFragmentStore(new DialogFragmentStoreImpl());
        }
    }
    
    public void startReplication() {
        if (isStarted.get() == false ) {
            IllegalStateException ex = new IllegalStateException("DialogFragmentManager must be started prior to starting replication.");
            _logger.log(Level.SEVERE, "df_start_replication_error");
            _logger.log(Level.SEVERE, ex.getMessage(), ex);
            throw ex;
        }

        // Only start once.
        if (isReplicationStarted.compareAndSet(false, true)) {
            registerWithMessageRouter();
        }
    }

    public boolean isReplicationStarted() {
        return isReplicationStarted.get();
    }
    
    @Override
    protected void doStop() {
        super.doStop();        
        deregisterWithMessageRouter();
    }
    
     /**
     * Factory method to instansiate DialogFragments
     * @param ds
     * @return
     */
    public DialogFragment createDialogFragment(DialogSet ds) {
        return new HADialogFragment(ds);
    }
     
    
    private void registerWithMessageRouter() {
        ReplicationMessageRouter router = null;
        if (Globals.IS_SECURITY_ENABLED) {
            router = (ReplicationMessageRouter)
                AccessController.doPrivileged(
                    new PrivilegedGetReplicationMessageRouter());
        } else {
            router = ReplicationMessageRouter.createInstance();
        }         
        if(router != null) {
            router.addReplicationManager(this.getApplicationId(), (ReplicationManager)this);
            router.registerBroadcastMethodList(broadcastMethods);
        }
    }  
    
    private void deregisterWithMessageRouter() {
        ReplicationMessageRouter router = null;
        if (Globals.IS_SECURITY_ENABLED) {
            router = (ReplicationMessageRouter)
                AccessController.doPrivileged(
                    new PrivilegedGetReplicationMessageRouter());
        } else {
            router = ReplicationMessageRouter.createInstance();
        }         
        if(router != null) {
            router.removeReplicationManager(this.getApplicationId());
            // FIXME currently no means to deregister Broadcast methodlist.
        }
    }  
    
    public String getApplicationId() {
        return APP_ID;
    }
    
     /**
    * get the backingStore for DialogFragment
    */ 
    synchronized public BackingStore getDialogFragmentBackingStore() {
        if (dialogFragmentBackingStore == null) {
            dialogFragmentBackingStore = createBackingStore(dialogFragmentCommandMap); 
        }
        return dialogFragmentBackingStore;
    }
    
   /**
    * create and set the backing store
    * @param commandMap map used to translate commands
    */     
    BackingStore createBackingStore(Map commandMap) {
        BackingStoreFactory storeFactory = new JxtaBackingStoreFactory();
        BackingStoreRegistry backingStoreRegistry 
            = BackingStoreRegistry.getInstance();
        Properties inputEnv = backingStoreRegistry.getFactoryClassEnv(getPassedInPersistenceType());
        Properties env = (Properties)inputEnv.clone();
        env.put(DUPLICATE_IDS_SEMANTICS_PROPERTY, Boolean.valueOf(this.isDuplicateIdsSemanticsAllowed()));
        env.put(SUPPRESS_LOAD_ACK_PROPERTY, Boolean.TRUE);
        env.put(COMMAND_MAP, commandMap);
        BackingStore backingStore = null;
        try {
            backingStore = storeFactory.createBackingStore(
                        SimpleMetadata.class,     //type
                        this.getApplicationId(), //appid
                        env);
        } catch (BackingStoreException ex) {
            //deliberate no-op
        }
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("backingStore: " + backingStore);
        }         
        if(backingStore != null) {
            if(backingStore instanceof JxtaBackingStoreImpl) {
                ((JxtaBackingStoreImpl)backingStore).setMode(MODE_SIP);
            }        
        }
        return backingStore;
    }        
    
    public String getPassedInPersistenceType() {
        return _passedInPersistenceType;
    }    
    
    public void setPassedInPersistenceType(String persistenceType) {
        _passedInPersistenceType = persistenceType;
    }    
    
    public boolean isDuplicateIdsSemanticsAllowed() {
        return _duplicateIdsSemanticsAllowed;
    }    
    
    public void setDuplicateIdsSemanticsAllowed(boolean value) {
        _duplicateIdsSemanticsAllowed = value;
    }   
    
    //DialogFragment cache methods
    
    private Object getReplicationSessionMonitor(String id) {
        return replicatedSessionMonitors.get(id);
    }    
    
    void processExpiredReplicaMonitors() {
        replicatedSessionMonitors.processExpired();
    }
    
    void processExpiredReplicaRemovals() {
        Set<Map.Entry> mapEntries = replicatedDialogRemovals.entrySet();
        long timeNow = System.currentTimeMillis();
        for(Iterator<Map.Entry> it = mapEntries.iterator(); it.hasNext();) {
            Map.Entry nextEntry = it.next();
            ReplicationStateRemovedEntry nextRemoval
                = (ReplicationStateRemovedEntry)nextEntry.getValue();        
            if((timeNow - nextRemoval.getRemovalTime()) > (60 * 1000)) {
                replicatedDialogRemovals.remove(nextRemoval.getId(), nextRemoval);
            }
        }
    }      

    /**
     * get the replicated DialogFragments cache
     */ 
    public BaseCache getReplicatedDialogFragments() {
        return replicatedDialogFragments;
    }
    
    /**
     * set the replicated DialogFragments cache
     * @param dialogFragmentTable
     */ 
    public void setReplicatedDialogFragments(BaseCache dialogFragmentTable) {
        replicatedDialogFragments = dialogFragmentTable;
    } 
    
    /**
     * Put dialogFragment State in DialogFragment replica cache
     * @param dialogFragmentState
     */    
    protected void putInDialogFragmentReplicationCache(ReplicationState dialogFragmentState) {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("in " + this.getClass().getName()
                         + ">>putInDialogFragmentReplicationCache id: "
                         + dialogFragmentState.getId());
        } 
        if (dialogFragmentState == null) {
            return;
        }
        String id = (String)dialogFragmentState.getId();
        if(id == null) {
            return;
        }
        Object monitor = getReplicationSessionMonitor(id);
        synchronized(monitor) {
            //do not put if already removed
            if(wasRemoved(id)) {
                return;
            }            
            ReplicationState currentState = (ReplicationState)replicatedDialogFragments.get(id);
            if(_logger.isLoggable(Level.FINE)) {
                if(currentState != null) {
                    _logger.fine("currentVersion: " + currentState.getVersion()
                                 + " newVersion: " + dialogFragmentState.getVersion());
                }
            }
            if((currentState != null)
                    && currentState.getVersion() > dialogFragmentState.getVersion()) {
                return;
            }
            replicatedDialogFragments.put(dialogFragmentState.getId(), dialogFragmentState);
            if(_logger.isLoggable(Level.FINE)) {
                _logger.log(Level.FINE, "in " + this.getClass().getName()
                    + ">>putInDialogFragmentReplicationCache complete id: "
                    + dialogFragmentState.getId() + "[ver:" + dialogFragmentState.getVersion()
                    + "]");
            }
        }
    }
    
    /**
     * get DialogFragment State from replica cache based on the id
     * @param id
     */    
    protected ReplicationState getFromDialogFragmentReplicationCache(String id) {
        if (replicatedDialogFragments == null) {
            return null;
        }
        Object monitor = getReplicationSessionMonitor(id);
        ReplicationState returnState = null;
        synchronized(monitor) {        
            returnState 
                = (ReplicationState)replicatedDialogFragments.get(id);
            if(returnState != null) {
                ReplicationStateUpdate returnStateUpdate
                    = this.getFromDialogFragmentReplicationUpdateCache(id);
                //apply update if it exists and has higher version
                if(returnStateUpdate != null) {
                    if(returnStateUpdate.getVersion() > returnState.getVersion()) {
                        returnState.setVersion(returnStateUpdate.getVersion());
                        returnState.setLastAccess(returnStateUpdate.getLastAccess());
                    }
                    this.removeFromDialogFragmentReplicationUpdateCache(returnStateUpdate.getId());
                }            
            }
        }
        return returnState;
    }
    
    /**
     * remove dialogFragment State from replica cache based on the id 
     * @param id
     */    
    protected ReplicationState removeFromDialogFragmentReplicationCache(String id) {
        if(id == null || replicatedDialogFragments == null) {
            return null;
        }
        ReplicationState returnState = null;
        Object monitor = getReplicationSessionMonitor(id);
        synchronized(monitor) {    
            returnState = getFromDialogFragmentReplicationCache(id);
            replicatedDialogFragments.remove(id);
            removeFromDialogFragmentReplicationUpdateCache(id);             
        }
        return returnState;               
    }
    
    private boolean wasRemoved(String id) {
        return replicatedDialogRemovals.get(id) != null;
    }    

    //DF updates cache methods
    
    /**
     * Put DialogFragment State updates in replica cache
     * @param dialogFragmentState
     */    
    protected void putInDialogFragmentReplicationUpdateCache(ReplicationStateUpdate dialogFragmentStateUpdate) {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("in " + this.getClass().getName()
                         + ">>putInDialogFragmentReplicationUpdateCache id: "
                         + dialogFragmentStateUpdate.getId());
        }
        if(dialogFragmentStateUpdate == null) {
            return;
        }
        String id = (String)dialogFragmentStateUpdate.getId();
        if(id == null) {
            return;
        }
        Object monitor = getReplicationSessionMonitor(id);
        synchronized(monitor) { 
            //do not put if already removed
            if(wasRemoved(id)) {
                return;
            }            
            ReplicationStateUpdate currentStateUpdate = this.getFromDialogFragmentReplicationUpdateCache(id);
            if(_logger.isLoggable(Level.FINE)) {
                if(currentStateUpdate != null) {
                    _logger.fine("currentVersion: "
                                 + currentStateUpdate.getVersion()
                                 + " newVersion: "
                                 + dialogFragmentStateUpdate.getVersion());
                }
            }
            if((currentStateUpdate != null)
                    && currentStateUpdate.getVersion() > dialogFragmentStateUpdate.getVersion()) {
                return;
            }
            replicatedDialogFragmentUpdates.put(dialogFragmentStateUpdate.getId(),
                                                dialogFragmentStateUpdate);
        }
    }
    
    /**
     * get DialogFragment State update from replica cache based on the id
     * @param id
     */    
    protected ReplicationStateUpdate getFromDialogFragmentReplicationUpdateCache(String id) {    
        if( id == null || replicatedDialogFragmentUpdates == null) {
            return null;
        }
        return (ReplicationStateUpdate)replicatedDialogFragmentUpdates.get(id);
    }
    
    /**
     * remove dialogFragment State update from replica cache based on the id of
     * dialogFragmentStateUpdate
     * @param id
     */    
    protected void removeFromDialogFragmentReplicationUpdateCache(String id) {
        if(id == null || replicatedDialogFragmentUpdates == null) {
            return;
        }
        replicatedDialogFragmentUpdates.remove(id);
    }   
    
    /** Returns a store from the pool 
     * This method intializes the store with right parameters
     * @return returns DialogFragmentStoreImpl
     */
    public DialogFragmentStoreImpl getDialogFragmentStore() {
        if(_logger.isLoggable(Level.FINER)) {
            _logger.entering("ReplicationDialogFragmentManager", "getDialogFragmentStore");
        }
        if (dialogFragmentStorePool == null) {
            // Replication service not initialized yet.
            return null;
        }
        DialogFragmentStoreImpl store = null;
        try {
            store = (DialogFragmentStoreImpl) dialogFragmentStorePool.take();
            if(_logger.isLoggable(Level.FINEST)) {
                _logger.log(Level.FINEST,
                    "ReplicationDialogFragmentManager.getDialogFragmentStore returning   " + store);
            }
            return store;
        }
        catch (Exception e) {
            // FIXME evaluate log level 
            if (_logger.isLoggable(Level.FINE)) {
                _logger.log(Level.FINE, "exception occurred during getDialogFragment", e);
            }
        }
        if(_logger.isLoggable(Level.FINE)) {
            _logger.exiting("ReplicationDialogFragmentManager", "getDialogFragmentStore", store);
        }
        return store;
    } 
    
    /** 
     *  Returns (puts) a store back to the pool
     */
    private void putDialogFragmentStore(StorePoolElement store) {    
        //( (DialogFragmentStoreImpl) store).setSipSessionManager(null);
        if (store != null && dialogFragmentStorePool != null) {
            try {
                dialogFragmentStorePool.put(store);
            } catch (InterruptedException ex1) {
                // FIXME evaluate log level 
                if (_logger.isLoggable(Level.FINE)) {
                    _logger.log(Level.FINE, "exception occurred during putDialogFragmentStore", ex1);
                }
            }
        }
    }
    
    /** Returns the singleton store 
     * This method intializes the store with right parameters
     * @return returns DialogFragmentStoreImpl
     */
    public DialogFragmentStoreImpl getSingletonDialogFragmentStore() { 
        return (DialogFragmentStoreImpl)getDialogFragmentStore(); 
    }
    
    //begin send-side methods
    
    /**
     * Persists the given DialogFragment.
     *
     * @param dialogFragment The DialogFragment to be persisted
     */
    public void saveDialogFragment(DialogFragment dialogFragment)
            throws IOException {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.entering("ReplicationDialogFragmentManager", "saveDialogFragment");
        }        
        DialogFragmentStoreImpl store = null;
        try {
            store = getDialogFragmentStore();
            store.save(dialogFragment);
        }
        finally {
            this.putDialogFragmentStore(store);
        }                 
        if(_logger.isLoggable(Level.FINE)) {
            _logger.exiting("ReplicationDialogFragmentManager", "saveDialogFragment");
        }
    }
    
    /**
     * Removes the given DialogFragment from both the active cache
     * and the persistent session store of this session manager,
     *
     * @param dialogFragment The DialogFragment to be removed
     */
    public void removeDialogFragment(DialogFragment dialogFragment) {
        
        if(_logger.isLoggable(Level.FINE)) {
            _logger.entering("ReplicationDialogFragmentManager", "removeDialogFragment");
        }
        super.removeDialogFragmentFromCache(dialogFragment, false);
        DialogFragmentStoreImpl store = null;
        try {
            store = getDialogFragmentStore();   
            store.remove(dialogFragment.getDialogId());
        } catch (IOException ex) {
            _logger.log(Level.WARNING, "df_remove_error", dialogFragment.toString());
            _logger.log(Level.WARNING, ex.getMessage(), ex);
        } finally {
            this.putDialogFragmentStore(store);
        }                 
        if(_logger.isLoggable(Level.FINE)) {
            _logger.exiting("ReplicationDialogFragmentManager", "removeDialogFragment");
        }
    } 
    
    /**
     * Removes the given DialogFragment from 
     * the persistent session store of this session manager,
     *
     * @param dialogFragmentId The id of the DialogFragment replica to be removed
     */
    public void removeDialogFragmentReplica(String dialogFragmentId) {
        
        if(_logger.isLoggable(Level.FINE)) {
            _logger.entering("ReplicationDialogFragmentManager", "removeDialogFragmentReplica");
        }
        DialogFragmentStoreImpl store = null;
        try {
            store = getDialogFragmentStore();   
            store.remove(dialogFragmentId);
        } catch (IOException ex) {
            _logger.log(Level.WARNING, "df_remove_error", dialogFragmentId);
            _logger.log(Level.WARNING, ex.getMessage(), ex);
        } finally {
            this.putDialogFragmentStore(store);
        }                 
        if(_logger.isLoggable(Level.FINE)) {
            _logger.exiting("ReplicationDialogFragmentManager", "removeDialogFragmentReplica");
        }
    }     
    
    /**
     * Removes the given DialogFragment from both the active cache and the
     * persistent session store of this session manager,
     *
     * @param id The id of the DialogFragment to be removed
     */
    public void removeDialogFragment(String id, boolean hasTimedOut) {
        
        if(_logger.isLoggable(Level.FINE)) {
            _logger.entering("ReplicationDialogFragmentManager", "removeDialogFragment", id);
        }
        removeDialogFragmentFromCache(id, hasTimedOut);
        DialogFragmentStoreImpl store = null;
        try {
            store = getDialogFragmentStore();            
            store.remove(id); 
        } catch (IOException ex) {
            _logger.log(
                Level.WARNING,
                "unable to remove DialogFragment:id = " + id,
                ex);
        } finally {
            this.putDialogFragmentStore(store);
        }                 
        if(_logger.isLoggable(Level.FINE)) {
            _logger.exiting("ReplicationDialogFragmentManager", "removeDialogFragment");
        }
    }        
    
    /**
     * Look for a DialogFragment in the Store and, if found, restore
     * it in the Manager's list of active dialog fragments if appropriate.
     * The dialog fragment will be removed from the Store after swapping
     * in, but will not be added to the active dialog fragment list if it
     * is invalid or past its expiration.
     *
     * @param id the id of the DialogFragment
     *
     * @throws RemoteLockException
     */
    protected DialogFragment swapInDialogFragment(String id)
            throws IOException, RemoteLockException {
        return swapInDialogFragment(id, null);
    }    
    
    /**
     * Look for a DialogFragment in the Store and, if found, restore
     * it in the Manager's list of active dialog fragments if appropriate.
     * The dialog fragment will be removed from the Store after swapping
     * in, but will not be added to the active dialog fragment list if it
     * is invalid or past its expiration.
     * @param id the id of the Dialog Fragment
     * @param version - the version of the Dialog Fragment
     * 
     * @throws RemoteLockException
     */
    protected DialogFragment swapInDialogFragment(String id, String version)
            throws IOException, RemoteLockException {
        DialogFragmentStoreImpl store = null;
        try {
            store = this.getDialogFragmentStore();
            if (store == null)
                return null;

            DialogFragment dialogFragment = null;
            try {
                if (SecurityUtil.isPackageProtectionEnabled()){
                    try{
                        dialogFragment = (DialogFragment) AccessController.doPrivileged(new PrivilegedStoreLoadDialogFragment(id, version, store));
                    }catch(PrivilegedActionException ex){
                        Exception exception = ex.getException();
                        if (exception instanceof IOException){
                            throw (IOException)exception;
                        } else if (exception instanceof ClassNotFoundException) {
                            throw (ClassNotFoundException)exception;
                        }
                    }
                } else {
                    if (version != null) {
                         dialogFragment = store.load(id, version);
                    } else {
                         dialogFragment = store.load(id);
                    }
                }   
            } catch (ClassNotFoundException e) {
                IOException ex1 = 
                    (IOException) new IOException("Error during swapInSipApplicationSession: " + e.getMessage()).initCause(e);
                throw ex1;                 
            }

            if (dialogFragment == null)
                return (null); 
            registerDialogFragment(dialogFragment);
            return (dialogFragment);
        } finally {
            this.putDialogFragmentStore(store);
        }
    }        
    
    //start receiver side process methods
    
    //DialogFragment process methods

    /**
     * process the save of a DialogFragment
     * @param dialogFragmentState - contains the save command
     */     
    public void processSavedialogfragment(ReplicationState dialogFragmentState) {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("IN" + this.getClass().getName() + ">>processSavedialogfragment");
        }        
        this.putInDialogFragmentReplicationCache(dialogFragmentState);
    }    

    /**
     * process the removal of a DialogFragment
     * @param dialogFragmentState - contains the remove command
     */     
    public void processRemovedialogfragment(ReplicationState dialogFragmentState) {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("IN" + this.getClass().getName() + ">>processRemovedialogfragment");
        } 
        if(dialogFragmentState == null) {
            return;
        }
        String id = (String)dialogFragmentState.getId();
        //keep track of recent removals
        replicatedDialogRemovals.put(id, new ReplicationStateRemovedEntry(id, System.currentTimeMillis()));        
        removeFromDialogFragmentReplicationCache(id);
    }

    /**
     * process the update of a DialogFragment
     * @param dialogFragmentState - contains the updatelastaccesstime command
     */    
    public void processUpdatelastaccesstimedialogfragment(ReplicationState dialogFragmentState) {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("IN" + this.getClass().getName() + ">>processUpdatelastaccesstimedialogfragment");
            _logger.fine("processUpdatelastaccesstimedialogfragment:version:" + dialogFragmentState.getVersion());
        }
        String id = (String)dialogFragmentState.getId();
        ReplicationState state = null;
        Object monitor = getReplicationSessionMonitor(id);
        synchronized(monitor) {        
            state = this.getFromDialogFragmentReplicationCache(id);
            //overlay update only if version is greater
            if(state != null) {
                if(dialogFragmentState.getVersion() > state.getVersion()) {                
                    state.setLastAccess(dialogFragmentState.getLastAccess());
                    state.setVersion(dialogFragmentState.getVersion());
                    state.setContainerExtraParamsState(dialogFragmentState.getContainerExtraParamsState());
                }
            } else {
                if(_logger.isLoggable(Level.FINE)) {                
                    _logger.fine("processUpdatelastaccesstimedialogfragment: attempting to update a dialogframgnet not yet stored:id:" + dialogFragmentState.getId());
                }
                this.putInDialogFragmentReplicationUpdateCache(new ReplicationStateUpdate(id, dialogFragmentState.getVersion(), dialogFragmentState.getLastAccess()) );
            }
        }
    }
    
    /**
     * process the broadcastfinddialogfragment for DialogFragment
     * @param queryState
     */     
    public ReplicationState processBroadcastfinddialogfragment(ReplicationState queryState) {
        //complete query and send back response
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("in " + this.getClass().getName() + ">>processBroadcastfinddialogfragment:instance: " + instanceName);
            _logger.fine("in " + this.getClass().getName() + ">>processBroadcastfinddialogfragment:id=" + queryState.getId());                        
        } 
        
        //look in active cache
        HADialogFragment haDialogFragment = null;
        DialogFragment dialogFragment 
            = findDialogFragmentFromCacheOnly((String)queryState.getId());
        if(dialogFragment instanceof HADialogFragment) {
            haDialogFragment = (HADialogFragment)dialogFragment;
        }        
        
        //look in replica cache
        ReplicationState replicaState
            = getFromDialogFragmentReplicationCache((String)queryState.getId());
        ReplicationState returnState 
            = getBestDialogFragment(haDialogFragment, replicaState, queryState);          
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("in " + this.getClass().getName() + ">>processBroadcastfinddialogfragment:returnState=" + returnState);
        }
        return returnState;
    }
    
    public void processBroadcastloadreceiveddialogfragment(ReplicationState queryState) {
        //load is acknowledged safe to remove replica now
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("in " + this.getClass().getName() + ">>processbroadcastloadreceiveddialogfragment:instance: " + instanceName);
            _logger.fine("in " + this.getClass().getName() + ">>processbroadcastloadreceiveddialogfragment:id=" + queryState.getId());                        
        }
        if(queryState == null || queryState.getId() == null) {
            return;
        }
        String id = (String)queryState.getId();
        //remove active dialogfragment if present and not locked
        DialogFragment dialogFragment 
            = findDialogFragmentFromCacheOnly(id);
        if(dialogFragment != null 
                && !dialogFragment.isForegroundLocked()) {
            this.removeDialogFragmentFromCache(dialogFragment, false);
        } 
        //only safe to remove replica if we are not the replica partner
        //of the sending instance
        ReplicationHealthChecker healthChecker 
            = ReplicationHealthChecker.getInstance();
        String replicatedFromInstanceName 
            = healthChecker.getReshapeReplicateFromInstanceName(); 
        if(replicatedFromInstanceName != null && !replicatedFromInstanceName.equalsIgnoreCase(queryState.getInstanceName())) {           
            removeFromDialogFragmentReplicationCache(id);           
        }
    }   
    
    /**
     * find the best version of DialogFragment
     * removing any stale versions and return query result
     * @param activeDialogFragment DialogFragment 
     *      from active cache
     * @param replicaDialogFragment DialogFragment 
     *      from replica cache
     * @param queryVersion version requested in query (-1 means 
     *      version unaware
     */     
    private ReplicationState getBestDialogFragment(HADialogFragment activeDialogFragment, 
        ReplicationState replicaDialogFragment, ReplicationState queryState) {
        ReplicationState bestResult = null;
        long queryVersion = queryState.getVersion();
        
        //first check for none found in either active or replica caches
        if(replicaDialogFragment == null && activeDialogFragment == null) {
            //return nack
            return ReplicationState.createQueryResponseFrom(queryState, true);
        }
        
        //next check for artifacts found in both active and replica caches
        if(replicaDialogFragment != null && activeDialogFragment != null) { 
            //compare and remove the lesser version
            //keeping the higher version as (tentative) best
            if(replicaDialogFragment.getVersion() <= activeDialogFragment.getVersion()) {
                //remove stale replica - work with active
                removeFromDialogFragmentReplicationCache((String)replicaDialogFragment.getId());
                //create appropriate response from active
                bestResult 
                    = createDialogFragmentResponseFrom(activeDialogFragment, queryState);
            } else {
                //remove stale active - work with replica
                clearFromDialogFragmentManagerCache(activeDialogFragment.getDialogId(), false);               
                //create appropriate response from replica
                bestResult 
                    = createDialogFragmentResponseFrom(replicaDialogFragment, queryState);                                
            }
        } else {
            //either replica or active is null and other is non-null
            //replica is null and active is not null
            if(replicaDialogFragment == null) {                
                //create appropriate response from active
                bestResult 
                    = createDialogFragmentResponseFrom(activeDialogFragment, queryState);                               
            } else {
                //active is null & replica is not null
                //create appropriate response from replica
                bestResult 
                    = createDialogFragmentResponseFrom(replicaDialogFragment, queryState);                                 
            }
        }
        return bestResult;
    }
    
    private ReplicationState createDialogFragmentResponseFrom(ReplicationState replicaDialogFragment, 
            ReplicationState queryState) {        
        //create appropriate response from replica
        ReplicationState result = null;
        long queryVersion = queryState.getVersion();        
        if(queryVersion != -1 && replicaDialogFragment.getVersion() < queryVersion) {
            //return nack & clear stale replica
            removeFromDialogFragmentReplicationCache((String)replicaDialogFragment.getId());
            result = ReplicationState.createQueryResponseFrom(queryState, true);
        } else {
            //return real response based on replica
            result = ReplicationState.createQueryResponseFrom(replicaDialogFragment); 
        }        
        return result;
    }

    private ReplicationState createDialogFragmentResponseFrom(HADialogFragment activeDialogFragment, 
            ReplicationState queryState) {     
        //create appropriate response from active
        ReplicationState result = null;
        long queryVersion = queryState.getVersion();        
        if(queryVersion != -1 && activeDialogFragment.getVersion() < queryVersion) {
            //return nack & clear stale active
            clearFromDialogFragmentManagerCache(activeDialogFragment.getDialogId(), false);
            result = ReplicationState.createQueryResponseFrom(queryState, true);
        } else {
            //return real response based on active DF
            try {
                result = createQueryResponseFrom(activeDialogFragment);
            } catch (IOException ioe) {
                _logger.log(Level.WARNING, "df_load_error", activeDialogFragment.toString());
                _logger.log(Level.WARNING, ioe.getMessage(), ioe);
                
                // We've been unable to serialize the given active
                // DialogFragment.
                // Clear it from the active cache and return a nack instead
                clearFromDialogFragmentManagerCache(
                    activeDialogFragment.getDialogId(), false);
                result = ReplicationState.createQueryResponseFrom(queryState,
                                                                  true);
            }
        }

        return result;
    }
    
    /**
     * Converts the given DialogFragment to a ReplicationState.
     *
     * @param DF The DialogFragment to be converted
     *
     * @return The ReplicationState corresponding to the given 
     * DialogFragment
     */ 
    private ReplicationState createQueryResponseFrom(
            HADialogFragment df) throws IOException {
        byte[] containerExtraParamState = null;
        DialogFragmentExtraParams containerExtraParams 
            = df.getExtraParameters();
        if(containerExtraParams != null) {
            try {
                containerExtraParamState
                    = ReplicationUtil.getByteArray(containerExtraParams);
            } catch (IOException ex) {
                ; //deliberate no-op
            }
        }         
        return new ReplicationState(
                MODE_SIP,
                df.getDialogId(),
                getApplicationId(),  
                df.getVersion(),
                0L,
                0L,
                null,
                null,
                null,
                ReplicationState.RETURN_BROADCAST_MSG_COMMAND,
                ReplicationUtil.getByteArray(df), 
                null,
                containerExtraParamState);
    }

    protected void clearFromDialogFragmentManagerCache(String id, boolean hasTimedOut) {
        DialogFragment df = findDialogFragmentFromCacheOnly(id);
        if(df != null) {                               
            this.removeDialogFragmentFromCache(df, hasTimedOut);
        } 
    }
    
    //end receiver side process methods
    
    //start expiration related methods
    
    void processExpiredDialogFragmentReplicas() {
        //FIXME 
        //code assumes that isExpired works for this type of ReplicationState
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("in " + this.getClass().getName() + ">>processExpiredDialogFragmentReplicas");            
        }
        ArrayList expiredReplicas = new ArrayList(30);
        BaseCache replicasCache = this.getReplicatedDialogFragments();
        for(Iterator it = replicasCache.values(); it.hasNext();) {
            ReplicationState nextState = (ReplicationState)it.next();
            if(_logger.isLoggable(Level.FINE)) {
                _logger.fine("in " + this.getClass().getName() + "nextState=" + nextState);            
            } 
            if(isExpiredDialogFragmentReplica(nextState)) {
                expiredReplicas.add(nextState);
            }
        }
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("processExpiredDialogFragmentReplicas:expiredReplicas.size=" + expiredReplicas.size());            
        }        
        for(int i=0; i<expiredReplicas.size(); i++) {
            ReplicationState nextState = 
                (ReplicationState)expiredReplicas.get(i);
            this.removeFromDialogFragmentReplicationCache((String)nextState.getId());
        }
    }
    
    private DialogFragmentExtraParams getDialogFragmentExtraParamsFrom(ReplicationState state) {
        if(state == null) {
            return null;
        }
        if(state.getDeserializedExtraParam() != null) {
            return (DialogFragmentExtraParams)state.getDeserializedExtraParam();
        } else {
            //deserialize and cache result in state
            byte[] extraParamsState = state.getContainerExtraParamsState();
            DialogFragmentExtraParams extraParams = null;
            if(extraParamsState != null) {
                try {
                    extraParams
                        = (DialogFragmentExtraParams)ReplicationState.getObjectValue(extraParamsState);
                } catch (Exception ex) {
                    _logger.log(Level.WARNING, "df_extraparams_deserialization_error", state.getAppId() + state.getId());
                    _logger.log(Level.WARNING, ex.getMessage(), ex);
                }
                state.setDeserializedExtraParam(extraParams);
            }            
            return extraParams;
        }
    }     
    
    protected boolean isExpiredDialogFragmentReplica(ReplicationState state) {
        DialogFragmentExtraParams extraParams
            = getDialogFragmentExtraParamsFrom(state);
        return isExpiredDialogFragmentReplica(extraParams);
    }
    
    protected boolean isExpiredDialogFragmentReplica(DialogFragmentExtraParams extraParams) {         
        if(extraParams == null) {
            return false;
        } else {
            long nextExpirationTime = extraParams.getExpirationTime();
            //expirationTime of 0 or -1 means infinite
            //in this case we expire if duration since last access
            //is greater than SERVLET_TIMER_MAX_THRESHOLD_IN_MILLIS
            if(nextExpirationTime == 0L || nextExpirationTime == -1L) {
                if( (System.currentTimeMillis() - extraParams.getInternalLastAccessedTime()) > ActivationHelper.SERVLET_TIMER_MAX_THRESHOLD_IN_MILLIS ) {
                    return true;
                } else {
                    return false;
                }
            } else {
                return (nextExpirationTime < System.currentTimeMillis());
            }
        }
    }    
    
    void processExpiredDialogFragmentReplicasThirdPartySPI() {
        //code assumes that isExpired works for this type of ReplicationState
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("in " + this.getClass().getName() + ">>processExpiredDialogFragmentReplicasThirdPartySPI");            
        }
        ArrayList expiredReplicas = new ArrayList(30);
        List dialogFragmentExtraParams = getDialogFragmentExtraParamsThirdPartySPI();
        for(Iterator it = dialogFragmentExtraParams.iterator(); it.hasNext();) {
            DialogFragmentExtraParams nextExtraParam = (DialogFragmentExtraParams)it.next();
            if(_logger.isLoggable(Level.FINE)) {
                _logger.fine("in " + this.getClass().getName() + " nextExtraParam=" + nextExtraParam);            
            } 
            if(isExpiredDialogFragmentReplica(nextExtraParam)) {
                expiredReplicas.add(nextExtraParam);
            }
        }
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("processExpiredDialogFragmentReplicasThirdPartySPI:expiredReplicas.size=" + expiredReplicas.size());            
        }
        for(int i=0; i<expiredReplicas.size(); i++) {
            DialogFragmentExtraParams nextExtraParam = 
                (DialogFragmentExtraParams)expiredReplicas.get(i);
            //FIXME not sure if this should be true?
            this.removeDialogFragment((String)nextExtraParam.getId(), false);
        }
    }
    
    List getDialogFragmentExtraParamsThirdPartySPI() {        
        BackingStore dialogFragmentBackingStore 
            = this.getDialogFragmentBackingStore();
        DialogFragmentExtraParams dialogFragmentExtraParamCriteria 
            = DialogFragmentExtraParams.createSearchCriteriaAll();        
        Collection<DialogFragmentExtraParams> dialogFragmentColl 
            = dialogFragmentBackingStore.findByCriteria(dialogFragmentExtraParamCriteria);

        List result = new ArrayList();        
        Iterator<DialogFragmentExtraParams> dialogFragmentResults =
            (Iterator<DialogFragmentExtraParams>) dialogFragmentColl.iterator();
        while (dialogFragmentResults.hasNext()) {
            DialogFragmentExtraParams eParam = dialogFragmentResults.next();
            result.add(eParam);            
        }        
        return result;
    }     
   
    //begin implement ReplicationManager
    
    private boolean isBroadcastState(ReplicationState state) {
        return(broadcastMethods.contains(state.getCommand()));
    }
                
    public void processMessage(ReplicationState message) {
        //handle broadcast methods
        if(isBroadcastState(message)) {
            processBroadcastMessage(message);
            return;
        }
        
        //handle non-void methods
        ReplicationStateQueryResponse queryResult = null;
        //do process non-void message (and cannot be response either)
        if(!message.isResponseState() && !message.isVoidMethodReturnState()) {
            //do non-void processing including sending response
            queryResult = this.doProcessQueryMessage(message);
            ReplicationState qResponse = queryResult.getState();
            if(qResponse != null) {
                //sourceInstanceName is preserved in the response
                ReplicationState response = 
                    ReplicationState.createResponseFrom(qResponse);
                if(_logger.isLoggable(Level.FINE)) {
                    _logger.fine("RepMgrBase:responseState=" + response);
                }                
                this.doSendResponse(response);
            }
            return;
        }
        //end do process non-void message        

        boolean isResponse = this.doProcessMessage(message);
        
    }
    
    //return true if message is processResponse
    public boolean doProcessMessage(ReplicationState message) {
        boolean result = false;
        String methodName = ReplicationUtil.getProcessMethodName(message);
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("in " + this.getClass().getName() + ">>doProcessMessageName=" + methodName);
        }
        try {
            Class myClass = this.getClass();
            myClass.getMethod(
                methodName,
                    new Class[]{ message.getClass() }).invoke(
                        this, new Object[]{ message });            
        } catch (Throwable t) {
           _logger.log(Level.SEVERE, "df_unknown_replication_method_name_error", new Object[] { this.getClass().getName(), methodName });
           _logger.log(Level.SEVERE, t.getMessage(), t);
        } 
        if(methodName.equals("processResponse")) {
            result = true;
        }
        return result;
    }    
    
    /**
    * send the response
    *
    * @param dialogFragmentState 
    *   The replication state response
    */    
    public void doSendResponse(ReplicationState dialogFragmentState) {
               
        JxtaReplicationReceiver replicationReceiver = 
                JxtaReplicationReceiver.createInstance();
        ReplicationState resultState = 
                replicationReceiver.sendReplicationStateResponse(dialogFragmentState); 
    }        
    
    public void processBroadcastMessage(ReplicationState message) {
        ReplicationStateQueryResponse response = this.doProcessQueryMessage(message);
        boolean isResponse = response.isResponse();
        ReplicationState responseState = response.getState();
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("processBroadcastMessage:after doProcessQueryMessage:response=" + isResponse);
            _logger.fine("processBroadcastMessage:after doProcessQueryMessage:responseState=" + responseState);
            _logger.fine("processBroadcastMessage:after doProcessQueryMessage:responseStateTrunk=" + responseState.getTrunkState());
            _logger.fine("processBroadcastMessage:after doProcessQueryMessage:responseStateAttr=" + responseState.getState());
            _logger.fine("processBroadcastMessage:after doProcessQueryMessage:responseStateVer=" + responseState.getVersion());
        }

        //don't send a response to a response
        if(!isResponse) {
            //point-to-point response back to sender
            //System.out.println("processBroadcastMessage - need to send back result");
            doSendQueryResponse(responseState, message.getInstanceName());
        }
    }
    
    /**
    * send the response
    *
    * @param dialogFragmentState 
    *   The replication state response
    * @param instanceName  the name of the target instance
    */    
    public void doSendQueryResponse(ReplicationState dialogFragmentState, String instanceName) {
        JxtaReplicationSender replicationSender = 
                JxtaReplicationSender.createInstance();
        ReplicationState resultState = 
                replicationSender.sendReplicationStateQueryResponse(dialogFragmentState, instanceName); 
    }        
    
    //return true if message is processQueryResponse
    public ReplicationStateQueryResponse doProcessQueryMessage(ReplicationState message) {
        ReplicationState resultState = null;
        String methodName = ReplicationUtil.getProcessMethodName(message);
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("in " + this.getClass().getName() + ">>doProcessQueryMessage:methodName=" + methodName);
            _logger.fine("in " + this.getClass().getName() + ">>doProcessQueryMessage:thisInstance=" + instanceName + " returnInstance=" + message.getInstanceName() );
        }        
        try {
            Class myClass = this.getClass();
            resultState = (ReplicationState) myClass.getMethod(
                methodName,
                    new Class[]{ message.getClass() }).invoke(
                        this, new Object[]{ message });            
        } catch (Throwable t) {
           _logger.log(Level.SEVERE, "df_unknown_replication_method_name_error", new Object[] { this.getClass().getName(), methodName });
           _logger.log(Level.SEVERE, t.getMessage(), t);
        } 
        boolean isResponse = methodName.equals("processBroadcastresponse");
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("in " + this.getClass().getName() + ">>doProcessQueryMessage:resultState=" + resultState);
        }
        if (resultState != null) {
            resultState.setRouteAdvertisement(message.getRouteAdvertisement());
        }        
        //System.out.println("in " + this.getClass().getName() + ">>doProcessQueryMessage:resultState=" + resultState);
        return new ReplicationStateQueryResponse(resultState, isResponse);
    }
    
    public void processQueryMessage(ReplicationState message, String returnInstance) {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("in " + this.getClass().getName() + ">>processQueryMessage:returnInstance= " + returnInstance);
        }         
        ReplicationStateQueryResponse response = this.doProcessQueryMessage(message);
        boolean isResponse = response.isResponse();
        ReplicationState responseState = response.getState();
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("processQueryMessage:after doProcessQueryMessage:response=" + isResponse);
            _logger.fine("processQueryMessage:after doProcessQueryMessage:responseState=" + responseState);            
        }        
        //don't send a response to a response
        if(!isResponse && responseState != null) {
            //point-to-point response back to sender
            if(_logger.isLoggable(Level.FINE)) {
                _logger.fine("processQueryMessage - need to send back result to " + returnInstance);            
            }             
            doSendQueryResponse(responseState, returnInstance);
        }
    } 
    
    public ReplicationState processBroadcastresponse(ReplicationState queryResponseState) {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("in " + this.getClass().getName() + ">>processBroadcastresponse:queryResponseState=" + queryResponseState);            
        }
        ReplicationResponseRepository.putFederatedEntry(queryResponseState);
        return queryResponseState;
    }    
    
    public void repair(long repairStartTime) {
       if (!isRepairDuringFailure()) {
           return;
       }        
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("ReplicationDialogFragmentManager>>repair");                       
        }         

        if(ReplicationHealthChecker.isStopping()) {
            return;
        }

        Iterator it = dialogFragments.values().iterator();
        while (it.hasNext()) {
            DialogFragment fragment = (DialogFragment) it.next();
            //by virtue of being in the cache it is considered to 
            //but since invalidate is called before removal we must check            
            if(fragment.isValid()
                && (fragment.getDialogId() != null)
                && isDialogFragmentOlderThan(fragment, repairStartTime)) {
                if(fragment.lockBackground()) { 
                    try {
                        ((HADialogFragment)fragment).setReplicated(false);
                        ((HADialogFragment)fragment).setDirty(true);
                        saveDialogFragment(fragment);
                    } catch (IOException ex) {
                        _logger.log(Level.WARNING, "df_save_during_repair_error", fragment.toString());
                        _logger.log(Level.WARNING, ex.getMessage(), ex);
                    } finally {
                        fragment.unlockBackground();
                    }
                }                                
	    }            
        }
    } 
    
    protected boolean isDialogFragmentOlderThan(DialogFragment fragment, long aTime) {
        // FIXME Uncomment when getLastAccessedTime is implemented.
        // return (fragment.getLastAccessedTime() <= aTime); 
        return false;
    }    
    
    public void repair(long repairStartTime, boolean checkForStopping) {
       if (!isRepairDuringFailure()) {
           return;
       }         
        //FIXME
    }
    
    public void purge(String owningInstanceName, long purgeStartTime) {
        //do not purge if we are the replica partner
        //of the owning instance
        ReplicationHealthChecker healthChecker 
            = ReplicationHealthChecker.getInstance();
        String replicatedFromInstanceName 
            = healthChecker.getReshapeReplicateFromInstanceName();
        if(replicatedFromInstanceName != null && replicatedFromInstanceName.equalsIgnoreCase(owningInstanceName)) {            
            return;           
        }        
        List idsToRemove = new ArrayList();
        Iterator it = replicatedDialogFragments.values();
        while(it.hasNext()) {
            ReplicationState nextState 
                = (ReplicationState)it.next();
            byte[] extraParamsState
                = nextState.getContainerExtraParamsState();
            DialogFragmentExtraParams extraParams = null;
            if(extraParamsState != null) {
                try {
                extraParams
                    = (DialogFragmentExtraParams)ReplicationState.getObjectValue(extraParamsState);
                } catch (Exception ex) {
                    _logger.log(Level.WARNING, "df_extraparams_deserialization_error", nextState.getAppId() + nextState.getId());
                    _logger.log(Level.WARNING, ex.getMessage(), ex);
                }
            }
            if(extraParams != null) {
                String nextOwningInstance = extraParams.getCurrentOwnerInstanceName();
                if(nextOwningInstance != null 
                    && nextOwningInstance.equalsIgnoreCase(owningInstanceName)) {
                    idsToRemove.add((String)nextState.getId());
                }
            }
        }
        for(int i=0; i< idsToRemove.size(); i++) {
            removeFromDialogFragmentReplicationCache((String)idsToRemove.get(i));
        }
    }     

    public void respondToFailure(String failedInstanceName, boolean checkForStopping) {
        //FIXME Is there something specific to DialogFragments needed to be done here
        //     Following comment was copied from ReplicationDialogFragmentManager, doubt it applies to DialogFragment
        // only do timer wakeup if failedInstanceName was replication to us
        ReplicationHealthChecker healthChecker 
            = ReplicationHealthChecker.getInstance();
        String replicatedFromInstanceName 
            = healthChecker.getReshapeReplicateFromInstanceName();
        if(!failedInstanceName.equalsIgnoreCase(replicatedFromInstanceName)) {
            return;
        }
    }
  
    private void setDialogFragmentStorePool(StorePool dialogFragmentStorePool) {
        this.dialogFragmentStorePool = dialogFragmentStorePool;
    }

    public boolean isReplicationEnabled() {
        return isReplicationInitialized.get() && isReplicationStarted.get();
    }

    boolean isThirdPartyBackingStoreInUse() {
        BackingStore dialogFragmentBackingStore 
            = getDialogFragmentBackingStore();
        return (!(dialogFragmentBackingStore instanceof JxtaBackingStoreImpl));
    }
    
    // begin rolling upgrade related code
    
    public void doRollingUpgradePreShutdownProcessing(long startTime) {
        doSyncpointSave();       
    }
    
    public void doRollingUpgradePostStartupProcessing(long startTime) {
        //do syncpoint load
        doSyncpointLoad();
        doActiveCacheReconciliation();
        
        //trigger replicateFrom partner to do
        //replica cache reconciliation
        ReplicationHealthChecker healthChecker
            = ReplicationHealthChecker.getInstance();
        String replicateFromInstanceName = healthChecker.getReshapeReplicateFromInstanceName();
        sendRollingUpgradeAdvisory(replicateFromInstanceName);        
    }
    
    void doSyncpointSave() {
        DialogFragmentFileSync syncStore = new DialogFragmentFileSync((ReplicationManager)this);
        try {
            syncStore.save();
        } catch (IOException ex) {
            ex.printStackTrace();
        }
    }
    
    void doSyncpointLoad() {
        DialogFragmentFileSync syncStore = new DialogFragmentFileSync((ReplicationManager)this);
        try {
            syncStore.load();
        } catch (IOException ex) {
            ;
        } catch (ClassNotFoundException ex2) {
            ;
        }
    }
    
    void doActiveCacheReconciliation() {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("ReplicationDialogFragmentManager>>begin doActiveCacheReconciliation");                       
        }        
        HashMap stillOwnedDialogFragmentIds = getDialogFragmentIdList();
        if(_logger.isLoggable(Level.FINE)) {        
            Iterator it = stillOwnedDialogFragmentIds.values().iterator();
            while(it.hasNext()) {
                _logger.fine("next dfid = " + ((FederatedQueryListElement)it.next()).getId());
            } 
        }        
        reconcileDialogFragmentActiveCache(stillOwnedDialogFragmentIds);
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("ReplicationDialogFragmentManager>>end doActiveCacheReconciliation");                       
        }        
    }   
    
    /**
     * reconcile the active cache with a map containing
     * current set of instance ids owned by this instance
     * @param stillOwnedIds
     */
    private void reconcileDialogFragmentActiveCache(HashMap stillOwnedIds) {
        ArrayList idsToBeRemoved = new ArrayList();
        Iterator it = dialogFragments.values().iterator();
        while(it.hasNext()) {
            HADialogFragment nextDF 
                = (HADialogFragment)it.next();
            if(stillOwnedIds.get(nextDF.getDialogId()) == null) {
                idsToBeRemoved.add(nextDF.getDialogId());
            }
        }
        for(int i=0; i<idsToBeRemoved.size(); i++) {
            clearFromDialogFragmentManagerCache((String)idsToBeRemoved.get(i), false);
        }
    }
    
    /**
     * reconcile the replica cache (of your replica partner)
     * query instance1 to get a list of replica id/version data elements
     * then do 2 iterations:
     * iterate over the query result:
     * if an id from this list does not exist in this active cache 
     *    issue a remove message & load acknowledgment
     * if an id exists and the versions match  do nothing
     * if an id exists and the active version is > replica version, 
     *   - do a save
     * iterate over the active cache
     * if an id from active cache does not exist in the replica list 
     *   - do a save
     */    
    void doReplicaCacheReconciliation() {        
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("ReplicationDialogFragmentManager>>begin doReplicaCacheReconciliation");                       
        }
        cleanOutZombieReplicas();
        HashMap stillOwnedDialogFragmentIds = this.getDialogFragmentIdList();
        if(_logger.isLoggable(Level.FINE)) {        
            Iterator it = stillOwnedDialogFragmentIds.values().iterator();
            while(it.hasNext()) {
                _logger.fine("next dfid = " + ((FederatedQueryListElement)it.next()).getId());
            } 
        }              
        reconcileDialogFragmentReplicaCache(stillOwnedDialogFragmentIds);
    }
    
    void cleanOutZombieReplicas() {
        //FIXME for now issue load acknowledgement for each
        //active cache entry
        //this is functionally correct but not optimal
        //solution will be to improve query for replica
        //ids to track the source instance and then only
        //issue load acks to those ids that are not from
        //our replicate to partner
        HADialogFragment df = null;
        Iterator it = dialogFragments.values().iterator();
        while(it.hasNext()) {
            df = (HADialogFragment)it.next();
            String dfId = df.getDialogId();
            long version = df.getVersion();
            this.sendLoadAcknowledgement(dfId, version, MESSAGE_BROADCAST_LOAD_RECEIVED_DIALOG_FRAGMENT);
        }        
    }
    
    /**
     * reconcile the DialogFragment replica cache (of your replica partner)
     * with a map containing
     * current set of instance ids owned by this instance
     * @param stillOwnedIds
     */    
    private void reconcileDialogFragmentReplicaCache(HashMap stillOwnedIds) {
        HADialogFragment df = null;
        Iterator it = stillOwnedIds.values().iterator();
        while(it.hasNext()) {
            FederatedQueryListElement nextElement 
                = (FederatedQueryListElement)it.next();
            df = (HADialogFragment)this.findDialogFragmentFromCacheOnly(nextElement.getId());
            //if not present in active cache, issue remove
            //and issue load acknowledgement to clean any zombie replicas
            if(df == null) {
                String nextId = nextElement.getId();
                this.removeDialogFragmentReplica(nextId);
                this.sendLoadAcknowledgement(nextId, nextElement.getVersion(), MESSAGE_BROADCAST_LOAD_RECEIVED_DIALOG_FRAGMENT);
            } else {
                //if active version > replica version - issue save
                if(df.getVersion() > nextElement.getVersion()) {
                    try {
                        this.saveDialogFragment(df);
                    } catch (IOException ex) {}
                }
            }
        }
        Iterator it2 = dialogFragments.values().iterator();
        while(it2.hasNext()) {
            df = (HADialogFragment)it2.next();
            String activeDfId = df.getDialogId();
            //if session in active cache has no corresponding replica
            //issue a save
            if(stillOwnedIds.get(activeDfId) == null) {
                try {
                    this.saveDialogFragment(df);
                } catch (IOException ex) {}
            }
        }
    }    
    
    protected void sendLoadAcknowledgement(String id, long version, String command) {
        JxtaReplicationSender sender
            = JxtaReplicationSender.createInstance();
        ReplicationState loadReceivedState = 
            ReplicationState.createBroadcastLoadReceivedState(MODE_SIP, id, this.getApplicationId(), version, ReplicationUtil.getInstanceName(), command); 
        sender.sendBroadcastQuery(loadReceivedState);
    }    
    
    protected void readDialogFragments(ObjectInputStream ois) 
        throws ClassNotFoundException, IOException {
        int count = ois.readInt();
        for(int i=0; i<count; i++) {
            HADialogFragment nextDF = (HADialogFragment)ois.readObject();
            dialogFragments.put(nextDF.getDialogId(), nextDF);
        }
    }    
    
    protected void readReplicatedDialogFragmentUpdates(ObjectInputStream ois) 
        throws ClassNotFoundException, IOException {
        int count = ois.readInt();
        for(int i=0; i<count; i++) {
            ReplicationStateUpdate nextState = (ReplicationStateUpdate)ois.readObject();
            replicatedDialogFragmentUpdates.put(nextState.getId(), nextState);
        }
    }    
    
    protected void writeDialogFragments(ObjectOutputStream oos) 
        throws IOException {
        ReplicationUtil.writeHashMap(dialogFragments, oos);
    }     
    
    protected void writeReplicatedDialogFragmentUpdates(ObjectOutputStream oos) 
        throws IOException {
        ReplicationUtil.writeBaseCache(replicatedDialogFragmentUpdates, oos);
    } 
    
    public HashMap getDialogFragmentIdList() {
        //first get the list from local replica cache
        List localDialogFragmentIdsList = new ArrayList();
        //do multi-cast to cluster member and get back list
        HashMap result = new HashMap();
        String findDialogFragmentIdListCommand = MESSAGE_BROADCAST_DIALOG_FRAGMENT_ID_QUERY;
        //need a unique pseudo-id for this query
        String id = this.getApplicationId() + ReplicationUtil.getInstanceName() + System.currentTimeMillis();
        ReplicationState state = 
            ReplicationState.createBroadcastQueryState(MODE_SIP, id, this.getApplicationId(), -1L, ReplicationUtil.getInstanceName(), findDialogFragmentIdListCommand);

        int numberExpectedResults 
            = ReplicationUtil.getNumberExpectedRespondants();
        FederatedListRequestProcessor federatedRequestProcessor = 
            new FederatedListRequestProcessor(state, numberExpectedResults, 4000L);
       
        JxtaReplicationSender sender 
            = JxtaReplicationSender.createInstance();
        ReplicationState queryResult = sender.sendReplicationStateQuery(state, federatedRequestProcessor);
        if(queryResult != null && queryResult.getState() != null) {
            byte[] returnState = queryResult.getState();
            try {
                result = (HashMap)ReplicationState.getObjectValue(returnState);
            } catch (Exception ex) {
                ; //deliberately do nothing
            }            
        }
        result = mergeLocalListIntoMap(localDialogFragmentIdsList, result);
        return result;
    }
    
    private HashMap mergeLocalListIntoMap(List localDialogFragmentIdsList, HashMap dialogFragmentIdsMap) {
        for(int i=0; i<localDialogFragmentIdsList.size(); i++) {
            FederatedQueryListElement nextElement = (FederatedQueryListElement)localDialogFragmentIdsList.get(i);            
            if(nextElement != null) {
                //get any existing element in the map
                FederatedQueryListElement existingElement 
                    = (FederatedQueryListElement)dialogFragmentIdsMap.get(nextElement.getId());
                if(existingElement == null || existingElement.getVersion() < nextElement.getVersion()) {
                    dialogFragmentIdsMap.put(nextElement.getId(), nextElement);
                }
            }                   
        }
        return dialogFragmentIdsMap;
    }
    
    List getDialogFragmentIds(String owningInstanceName) {
        List dialogFragmentIds = new ArrayList();
        //using set to avoid dups
        HashSet dialogFragmentIdsSet = new HashSet();
        //iterate over dialog fragment replicas
        Iterator it = getReplicatedDialogFragments().values();
        long nextVersion = -1L;
        DialogFragmentExtraParams nextExtraParams = null;
        String nextOwnerInstanceName = null;
        while(it.hasNext()) {
            ReplicationState nextState 
                = (ReplicationState)it.next();
            String nextDialogFragmentId
                = (String)nextState.getId();
            if(nextState.getContainerExtraParamsState() != null) {
                try {
                    nextExtraParams
                        = (DialogFragmentExtraParams)ReplicationState.getObjectValue(nextState.getContainerExtraParamsState());
                } catch (Exception ex) {
                    ;
                }
            }
            if(nextExtraParams != null) {
                nextOwnerInstanceName
                    = nextExtraParams.getCurrentOwnerInstanceName();
            }
            if(nextOwnerInstanceName != null 
                && nextOwnerInstanceName.equalsIgnoreCase(owningInstanceName)) {
                nextVersion = nextState.getVersion();
                dialogFragmentIdsSet.add(new FederatedQueryListElement(nextDialogFragmentId, nextVersion));
            }
        }
        dialogFragmentIds.addAll(dialogFragmentIdsSet);
        return dialogFragmentIds;
    }   
    
    /**
     * process the broadcastfinddialogfragmentids for DialogFragment
     * @param queryState
     */     
    public ReplicationState processBroadcastfinddialogfragmentids(ReplicationState queryState) {
        //complete query and send back response
        String instanceName = ReplicationUtil.getInstanceName();
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("in " + this.getClass().getName() + ">>processBroadcastfinddialogfragmentids:instance: " + instanceName);
            _logger.fine("in " + this.getClass().getName() + ">>processBroadcastfinddialogfragmentids:owningInstance=" + queryState.getInstanceName());                        
        }         
        String owningInstanceName = queryState.getInstanceName(); 
        List dialogFragmentIds = null;
        //sessionIds = getDialogFragmentIds(owningInstanceName);
        if(this.isThirdPartyBackingStoreInUse()) {
            dialogFragmentIds
                = getDialogFragmentIdsThirdPartySPI(owningInstanceName);
        } else {
            dialogFragmentIds
                = getDialogFragmentIds(owningInstanceName);
        }
        byte[] resultState = null;
        try {
            resultState = ReplicationUtil.getByteArray((ArrayList)dialogFragmentIds);
        } catch (IOException ex) {
            //deliberate no-op
            ;
        }
        //first check for none found in either active or replica caches
        if(dialogFragmentIds == null || dialogFragmentIds.isEmpty()) {
            //return nack
            return ReplicationState.createQueryResponseFrom(queryState, true);
        } else {
            return ReplicationState.createQueryResponseFrom(queryState, resultState);
        }
    }  
    
    List getDialogFragmentIdsThirdPartySPI(String owningInstanceName) {        
        BackingStore backingStore 
            = this.getDialogFragmentBackingStore();
        DialogFragmentExtraParams dfExtraParamCriteria 
            = DialogFragmentExtraParams.createSearchCriteria(owningInstanceName);        
        Collection<DialogFragmentExtraParams> dfColl 
            = backingStore.findByCriteria(dfExtraParamCriteria);

        List dialogFragmentIds = new ArrayList();
        //using set to avoid dups
        HashSet dialogFragmentIdsSet = new HashSet();        
        Iterator<DialogFragmentExtraParams> dfResults =
            (Iterator<DialogFragmentExtraParams>) dfColl.iterator();
        while (dfResults.hasNext()) {
            DialogFragmentExtraParams eParam = dfResults.next();
            if (owningInstanceName.equals(eParam.getCurrentOwnerInstanceName())) {
                dialogFragmentIdsSet.add(new FederatedQueryListElement((String)eParam.getId(), -1L));
            }            
        }       
        dialogFragmentIds.addAll(dialogFragmentIdsSet);        
        return dialogFragmentIds;
    }
    
    /**
     * send a rolling upgrade advisory message to instance to trigger
     * it to do rolling upgrade reconciliation for the sending
     * instance
     *
     * @param instanceName the instance to be sent the rolling upgrade advisory
     */
    public void sendRollingUpgradeAdvisory(String instanceName) {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.entering("ReplicationDialogFragmentManager",
                             "sendRollingUpgradeAdvisory",
                             new Object[] {instanceName});
        }        
        DialogFragmentStoreImpl store = null;
        try {
            store = (DialogFragmentStoreImpl)getDialogFragmentStore();            
            store.sendRollingUpgradeAdvisory(instanceName);
        } catch (IOException ex) {
            _logger.log(Level.WARNING, ex.getMessage(), ex);
        } finally {
            this.putDialogFragmentStore(store);
        }                 
        if(_logger.isLoggable(Level.FINE)) {
            _logger.exiting("ReplicationDialogFragmentManager",
                            "sendRollingUpgradeAdvisory");
        }
    }
    
    public void processRollingupgradeadvisory(ReplicationState replicationState) {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("IN " + this.getClass().getName() + ">>processRollingupgradeadvisory");
        }    
        if(replicationState == null) {
            return;
        }
        //we have been triggered to do replica
        //cache reconciliation for our replicateTo partner
        doReplicaCacheReconciliation();
    }    
    
    // end rolling upgrade related code    
         
    private class ReplicationStateUpdate {
        private String _id = null;
        private long _lastAccess = 0L;
        private long _version = -1L;
        
        String getId() {
            return _id;
        }        
        
        long getLastAccess() {
            return _lastAccess;
        }
        
        long getVersion() {
            return _version;
        }       
        
        public ReplicationStateUpdate(String id, long version, long lastAccess) {
            _id = id;
            _version = version;
            _lastAccess = lastAccess;
        }
        
    } 
    
    private class PrivilegedStoreLoadDialogFragment
        implements PrivilegedExceptionAction {

        private String id;
        private String version;
        private DialogFragmentStoreImpl store;
            
        PrivilegedStoreLoadDialogFragment(String id, String version, DialogFragmentStoreImpl store) {     
            this.id = id;
            this.version = version;
            this.store = store;
        }

        public Object run() throws Exception{
           return store.load(id, version);
        }                       
    }
       
}
