/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.jvnet.glassfish.comms.replication.sessmgmt;

import java.util.concurrent.ConcurrentHashMap;
import java.util.Iterator;
import java.util.Random;
import java.util.Timer;
import java.util.TimerTask;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.ericsson.ssa.sip.SipApplicationSessionUtil;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationUtil;
import org.jvnet.glassfish.comms.replication.sessmgmt.ServletTimerExtraParams;
import org.jvnet.glassfish.comms.util.LogUtil;

/**
 * A class that handles restoration and migration of ServletTimers and
 * SipApplicationSessions in response to their owning instance failing. 
 *
 * The entry point is registerServletTimerExtraParam(), which indicates that
 *  the specified timer must be restored wither on this instance or on a different
 *  instance. The call to registerServletTimerExtraParam() results in a creation of
 *  a JDK Timer task that contains enough information to restore / recreate the
 *  ServletTimer either on this instance or on a different instance. The timer task
 *  itself has a periodicity that is controlled by SERVLET_TIMER_TASK_PERIODICITY_IN_MILLIS.
 *  The only reason to execute the TimerTask periodically (untill it is cancelled), is
 *  to ensure that the ServletTimer is restoredf eventually in some instance.
 *
 * Also, in order to smooth out this restoration process, the initial delay of the
 *  TimerTasks is calculated as follows:
 *
 *  1. If the difference between "now" and the ServletTimer's nextExpirationTime is below
 *      SERVLET_TIMER_MIN_THRESHOLD_IN_MILLIS, then the TimerTask is scheduled to execute
 *      immediately (0 delay).
 *
 *  2. If the difference between "now" and the ServletTimer's nextExpirationTime is after
 *      SERVLET_TIMER_MAX_THRESHOLD_IN_MILLIS
 *      then the TimerTask is scheduled to execute <b>BEFORE</b> SERVLET_TIMER_MAX_THRESHOLD_IN_MILLIS
 *      minus some random time (that is between 0 and SERVLET_TIMER_ADJUST_TIME_IN_MILLIS).
 *
 *  In all cases, the TimerTask executes with a periodicity of SERVLET_TIMER_TASK_PERIODICITY_IN_MILLIS.
 *      When the TimerTask's run() method is called, the following is done:
 *      a) A check is made to determine the "actual" owner that must "own" this ServletTimer. This is
 *          done by calling DataCentricUtil.getInstanceNameFromConsistentHash()
 *      b) if the above checks indicates that this instance must "own" the ServletTimer, then
 *          processLoadAdvisoryServletTimer(eParam) is called to restore the ServletTimer in this
 *          instance. This method is assumed to eventually call unregisterServletTimerExtraParam
 *      c) if the check in (a) indicates that some other instance must "own" the ServletTimer, then
 *          sendLoadAdvisoryServletTimer(eParam) is called to instruct the other instance to "own"
 *          the ServletTimer. The other instance will load the ServletTimer and will eventually
 *          call unregisterServletTimerExtraParam
 *
 *  Eventually, once the ServletTimer is restored (either on this instance or on another instance),
 *      the call to unregisterServletTimerExtraParam will cause the TimerTask to be cancelled.
 *
*/
public class ActivationHelper {

    private SipTransactionPersistentManager sipTxMgr;
 
    private static final Logger _logger = Logger.getLogger(LogUtil.SIP_LOG_DOMAIN);

    private static final Level TRACE_LEVEL = Level.FINE;

    private Random random = new Random(System.currentTimeMillis());

    private static final long SERVLET_TIMER_TASK_PERIODICITY_IN_MILLIS = 2 * 60 * 1000; // 2 minutes

    private static final long SAS_TIMER_TASK_PERIODICITY_IN_MILLIS = 2 * 60 * 1000; // 2 minutes

    /**
     * ServletTimers that need to execute before this threshold will result
     *  in creation of a TimerTask that will be scheduled immediately.
     */
    private static final long SERVLET_TIMER_MIN_THRESHOLD_IN_MILLIS = 6 * 60 * 1000; //6 minutes

    /**
     * ServletTimers that need to execute AFTER this threshold will result
     *  in creation of a TimerTask that will be scheduled this value minus
     *  a random time between zero and SERVLET_TIMER_ADJUST_TIME_IN_MILLIS.
     */
    public static final long SERVLET_TIMER_MAX_THRESHOLD_IN_MILLIS = 5 * 60 * 60 * 1000; //5 hours

    /**
     * @See SERVLET_TIMER_MAX_THRESHOLD_IN_MILLIS.
     */
    private static final long SERVLET_TIMER_ADJUST_THRESHOLD_IN_MILLIS = 5 * 60 * 1000; //5 minutes

    private static final float SERVLET_TIMER_ADJUST_PERCENTAGE = 0.10f;

    private static final long SAS_TIMER_MIN_THRESHOLD_IN_MILLIS = 6 * 60 * 1000; //6 minutes
    public static final long SAS_TIMER_MAX_THRESHOLD_IN_MILLIS = 5 * 60 * 60 * 1000; //5 hours
    private static final long SAS_TIMER_ADJUST_THRESHOLD_IN_MILLIS = 5 * 60 * 1000; //5 minutes
    private static final float SAS_TIMER_ADJUST_PERCENTAGE = 0.10f;
    
    private static final int SAS_TIMER_TYPE = 0;
    private static final int SERVLET_TIMER_TIMER_TYPE = 1;    

    /**
     * Our cache of TimerTasks for which ServletTimers haven't been created.
     * keyed by TimerId
     */
    private ConcurrentHashMap<String, TimerMigrationTask> timerTaskCache =
        new ConcurrentHashMap<String, TimerMigrationTask>();
    
    /**
     * Our cache of TimerTasks for which SasTimers haven't been created.
     * keyed by TimerId
     */
    private ConcurrentHashMap<String, SasTimerMigrationTask> sasTimerTaskCache =
        new ConcurrentHashMap<String, SasTimerMigrationTask>();
    
    private Timer migrationTimer = new Timer(true);

    /**
     * Constructor
     */
    ActivationHelper(SipTransactionPersistentManager sipTxMgr) {
        this.sipTxMgr = sipTxMgr;
    }

    void registerServletTimerMigrationTask(ServletTimerExtraParams eParam) {
        long timerScheduleTime = calculateServletTimerTaskScheduleTime(eParam);
        TimerMigrationTask task = new TimerMigrationTask(eParam);
        timerTaskCache.put(eParam.getId(), task);
        migrationTimer.scheduleAtFixedRate(task, timerScheduleTime,
            SERVLET_TIMER_TASK_PERIODICITY_IN_MILLIS);
    }   

    void registerSipApplicationSessionMigrationTask(SipApplicationSessionExtraParams eParam) {
        long timerScheduleTime = calculateSasTimerTaskScheduleTime(eParam);
        SasTimerMigrationTask task = new SasTimerMigrationTask(eParam);
        sasTimerTaskCache.put(eParam.getId(), task);
        migrationTimer.scheduleAtFixedRate(task, timerScheduleTime,
            SAS_TIMER_TASK_PERIODICITY_IN_MILLIS);
    }   

    boolean hasTimerTask(String timerId) {
        return timerTaskCache.contains(timerId);
    }

    boolean hasSasTimerTask(String timerId) {
        return sasTimerTaskCache.contains(timerId);
    }

    TimerTask unregisterServletTimerMigrationTask(String timerId) {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.entering("ActivationHelper",
                             "unregisterServletTimerExtraParam",
                             timerId);
        }  
        TimerMigrationTask task = timerTaskCache.remove(timerId);
        if (task != null) {
            task.stop();
            task.cancel();
        }
        if(_logger.isLoggable(Level.FINE)) {
            _logger.exiting("ActivationHelper",
                            "unregisterServletTimerExtraParam",
                            task);
        }
        return task;
    }

    TimerTask unregisterSasTimerMigrationTask(String timerId) {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.entering("ActivationHelper",
                             "unregisterSasTimerExtraParam",
                             timerId);
        }  
        SasTimerMigrationTask task = sasTimerTaskCache.remove(timerId);
        if (task != null) {
            task.stop();
            task.cancel();
        }
        if(_logger.isLoggable(Level.FINE)) {
            _logger.exiting("ActivationHelper",
                             "unregisterSasTimerExtraParam",
                             task);
        }
        return task;
    }

    long calculateServletTimerTaskScheduleTime(long expTime) {
        return this.calculateTimerTaskScheduleTime(expTime,
            SERVLET_TIMER_MIN_THRESHOLD_IN_MILLIS,
            SERVLET_TIMER_MAX_THRESHOLD_IN_MILLIS,
            SERVLET_TIMER_ADJUST_THRESHOLD_IN_MILLIS,
            SERVLET_TIMER_ADJUST_PERCENTAGE,
            SERVLET_TIMER_TIMER_TYPE);

    }

    long calculateServletTimerTaskScheduleTime(ServletTimerExtraParams eParam) {
        return this.calculateTimerTaskScheduleTime(eParam.getExpirationTime(),
            SERVLET_TIMER_MIN_THRESHOLD_IN_MILLIS,
            SERVLET_TIMER_MAX_THRESHOLD_IN_MILLIS,
            SERVLET_TIMER_ADJUST_THRESHOLD_IN_MILLIS,
            SERVLET_TIMER_ADJUST_PERCENTAGE,
            SERVLET_TIMER_TIMER_TYPE);
    }

    long calculateSasTimerTaskScheduleTime(long expTime) {
        return this.calculateTimerTaskScheduleTime(expTime,
            SAS_TIMER_MIN_THRESHOLD_IN_MILLIS,
            SAS_TIMER_MAX_THRESHOLD_IN_MILLIS,
            SAS_TIMER_ADJUST_THRESHOLD_IN_MILLIS,
            SAS_TIMER_ADJUST_PERCENTAGE,
            SAS_TIMER_TYPE);
    }

    long calculateSasTimerTaskScheduleTime(SipApplicationSessionExtraParams eParam) {
        return this.calculateTimerTaskScheduleTime(eParam.getExpirationTime(),
            SAS_TIMER_MIN_THRESHOLD_IN_MILLIS,
            SAS_TIMER_MAX_THRESHOLD_IN_MILLIS,
            SAS_TIMER_ADJUST_THRESHOLD_IN_MILLIS,
            SAS_TIMER_ADJUST_PERCENTAGE,
            SAS_TIMER_TYPE);
    }
    
    long calculateTimerTaskScheduleTime(long actualExpTime,
        long minThreshold, long maxThreshold, long adjustThreshold,
        float adjustPercentage, int timerType) {
        long now = System.currentTimeMillis();
        long timeDiff = actualExpTime - now;

        if (actualExpTime <= 0) {
            timeDiff = maxThreshold + 1;
        }

        long scheduleDelay = 0;
        if (timeDiff <= minThreshold) {
            if(timerType == SAS_TIMER_TYPE) {
                //give SAS a head start on loading
                scheduleDelay = 0;
            } else {
                //Aim is to let SAS go first and hold back the ST's a 
                //little bit to try to avoid race conditions between competing 
                //load advisories.
                scheduleDelay = calculateDelayUsingModulo(minThreshold, adjustPercentage);
                //pad the ServletTimer delay to insure it is at least 5 seconds
                if(scheduleDelay < 5000) {
                    scheduleDelay += 5000;
                }
            }
        } else if (timeDiff <= maxThreshold) {
            scheduleDelay = calculateDelayUsingModulo(timeDiff, adjustPercentage);
        } else {
            scheduleDelay = Math.abs(random.nextLong()) % adjustThreshold;
            scheduleDelay = maxThreshold - scheduleDelay;
        }

        return scheduleDelay;
    }

    /**
     * Cancels any outstanding ServletTimer and SipApplicationSession
     * migration tasks.
     */
    void cancelAllMigrationTasks() {
        
        // Cancel any SipApplicationSession migration tasks
        Iterator<String> iter = sasTimerTaskCache.keySet().iterator();
        while (iter.hasNext()) {
            unregisterSasTimerMigrationTask(iter.next());
        }

        // Cancel any ServletTimer migration tasks
        iter = timerTaskCache.keySet().iterator();
        while (iter.hasNext()) {
            unregisterServletTimerMigrationTask(iter.next());
        }
    }
    
    private long calculateDelayUsingModulo(long inputDiff, float adjustPercentage) {
        long scheduleDelay = 0;
        long modFactor = (long) (inputDiff * adjustPercentage);
        if (modFactor > 0) {
            scheduleDelay = inputDiff - (Math.abs(random.nextLong()) % modFactor);
        }

        if (scheduleDelay <= 0) {
            scheduleDelay = inputDiff; //better than setting it to zero;
        }
        return scheduleDelay;
    }

    private class TimerMigrationTask 
        extends TimerTask {

        private int runCount = 0;
        private boolean stopped = false;
        private ServletTimerExtraParams eParam;

        TimerMigrationTask(ServletTimerExtraParams extraParam) {
            this.eParam = extraParam;
        }

        String myInstanceName = null;
        String timerId = null;
        String actualOwnerName = null;

        public void stop() {
            stopped = true;
        }

        public void run() {

            if (stopped) {
                return;
            }

            try {
                myInstanceName = ReplicationUtil.getInstanceName();
                timerId = eParam.getId();
                actualOwnerName =
                    SipApplicationSessionUtil.getActualServerInstance(
                        eParam.getParentSasId());
                if (myInstanceName.equals(actualOwnerName)) {
                    if (runCount > 0) {
                        _logger.log(TRACE_LEVEL, "Executing processLoadAdvisoryServletTimer() (for " + runCount + " time)");
                    }

                    //Note: processLoadAdvisoryServletTimer must eventually
                    //  call unregisterServletTimerExtraParam()
                    sipTxMgr.processLoadadvisoryservlettimer(eParam);
                } else {
                    if (runCount > 0) {
                        _logger.log(TRACE_LEVEL, "Sending sendLoadAdvisoryServletTimer() (for " + runCount + " time) to " + actualOwnerName);
                    }

                    //Note: The acutualOwner must eventually send an ACK that
                    //  must result in a call to unregisterServletTimerExtraParam()
                    sipTxMgr.sendLoadAdvisoryServletTimer(timerId, actualOwnerName);
                }
            } catch (Exception ex) {
                LogUtil.SSR_LOGGER.getLogger().log(Level.WARNING, "st_migration_upon_expiration_error", new Object[] { timerId, myInstanceName, actualOwnerName });
                LogUtil.SSR_LOGGER.getLogger().log(Level.WARNING, ex.getMessage(), ex);
            } finally {
                runCount++;
            }
        }
    }

    private class SasTimerMigrationTask
        extends TimerTask {

        private int runCount = 0;
        private boolean stopped = false;
        private SipApplicationSessionExtraParams eParam;

        SasTimerMigrationTask(SipApplicationSessionExtraParams extraParam) {
            this.eParam = extraParam;
        }

        String myInstanceName = null;
        String sasId = null;
        String actualOwnerName = null;

        public void stop() {
            stopped = true;
        }

        public void run() {

            if (stopped) {
                return;
            }

            try {
                myInstanceName = ReplicationUtil.getInstanceName();
                sasId = eParam.getId();
                actualOwnerName = 
                    SipApplicationSessionUtil.getActualServerInstance(
                        eParam.getId());
                if (myInstanceName.equals(actualOwnerName)) {
                    if (runCount > 0) {
                        _logger.log(TRACE_LEVEL, "Executing processLoadAdvisorySas() (for " + runCount + " time)");
                    }

                    //Note: processLoadAdvisorySas must eventually
                    //  call unregisterServletTimerExtraParam()
                    sipTxMgr.processLoadadvisorysas(eParam);
                } else {
                    if (runCount > 0) {
                        _logger.log(TRACE_LEVEL, "Sending sendLoadAdvisorySas() (for " + runCount + " time) to " + actualOwnerName);
                    }

                    //Note: The acutualOwner must eventually send an ACK that
                    //  must result in a call to unregisterSas()
                    sipTxMgr.sendLoadAdvisorySipApplicationSession(sasId, actualOwnerName);
                }
            } catch (Exception ex) {
                LogUtil.SSR_LOGGER.getLogger().log(Level.WARNING, "sas_migration_upon_expiration_error", new Object[] { sasId, myInstanceName, actualOwnerName });
                LogUtil.SSR_LOGGER.getLogger().log(Level.WARNING, ex.getMessage(), ex);
            } finally {
                runCount++;
            }
        }
    }
}
