/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.jvnet.glassfish.comms.replication.sessmgmt;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.servlet.http.HttpSession;

import org.jvnet.glassfish.comms.util.LogUtil;

import com.ericsson.ssa.sip.PersistentSipSessionManagerBase;
import com.ericsson.ssa.sip.RemoteLockException;
import com.ericsson.ssa.sip.SipApplicationSessionImpl;
import com.ericsson.ssa.sip.SipApplicationSessionUtil;
import com.ericsson.ssa.sip.SipSessionBase;
import com.ericsson.ssa.sip.SipSessionDialogImpl;
import com.ericsson.ssa.sip.SipSessionManagerBase;
import com.ericsson.ssa.sip.persistence.ReplicationUnitOfWork;
import com.ericsson.ssa.sip.timer.GeneralTimer;
import com.ericsson.ssa.sip.timer.ServletTimerImpl;
import com.sun.appserv.ha.uow.ReplicableEntity;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationState;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationUtil;


public class HASipApplicationSession extends SipApplicationSessionImpl
    implements ReplicableEntity {

    /**
     * 
     */
    private static final long serialVersionUID = 1L;

    /** The serialized format versioning. 1 = first version. */
    private static short serializedFormVersion  = 1;

    // XXX make this a configurable item?
    private static final long DELTA = 10 * 1000; // 10 secs
    private transient boolean isDirty = false;
    private transient AtomicLong version;
    private transient SipApplicationSessionExtraParams extraParams;
    private transient String previousOwnerInstanceName;
    private transient String currentOwnerInstanceName;
    private transient long internalLastAccessedTime;
    private transient boolean wasLockedOnRemoteInstance = false;
    private transient boolean isBeingReplicated = false;
    protected transient boolean isReplicated = false;
    private static final Logger logger = LogUtil.SSR_LOGGER.getLogger();

    /**
     * Constructor.
     */
    public HASipApplicationSession(SipSessionManagerBase manager, String id) {
        super(manager, id);
        version = new AtomicLong(-1);
        extraParams = new SipApplicationSessionExtraParams(this);
        currentOwnerInstanceName = ReplicationUtil.getInstanceName();
    }

    /**
     * Gets the name of the instance.that previously owned this
     * SipApplicationSession.
     *
     * @return The name of the instance that previously owned this
     *         SipApplicationSession
     */
    public String getPreviousOwnerInstanceName() {
        return previousOwnerInstanceName;
    }

    /**
     * Gets the name of the instance.that currently owns this
     * SipApplicationSession.
     *
     * @return The name of the instance that currently owns this
     *         SipApplicationSession
     */
    public String getCurrentOwnerInstanceName() {
        return currentOwnerInstanceName;
    }

    /**
     * Checks whether this session is dirty or not.
     *
     * @return true if this session has been marked as dirty, false otherwise
     */
    public boolean isDirty() {
        return isDirty;
    }

    /**
     * Marks this session as dirty or non-dirty.
     *
     * @param isDirty The dirty flag
     */
    public void setDirty(boolean isDirty) {
        setDirty(isDirty, true);
    }
    
    /**
     * Marks this session as dirty or non-dirty.
     *
     * @param isDirty The dirty flag
     * @param addToUnitOfWork should we add to unit of work
     */
    public void setDirty(boolean isDirty, boolean addToUnitOfWork) {
        this.isDirty = isDirty;

        if (isDirty && addToUnitOfWork) {
            addToUnitOfWork();
        }
    } 

    /**
     * Increments the version of this SipApplicationSession.
     */
    public void incrementVersion() {
        version.incrementAndGet();
    }

    /**
     * Gets the version of this SipApplicationSession.
     *
     * @return The version of this SipApplicationSession
     */
    public long getVersion() {
        return version.get();
    }

    /**
     * Sets the version of this SipApplicationSession.
     *
     * @value The new version of this SipApplicationSession
     */
    public void setVersion(long value) {
        version.getAndSet(value);
    }
    
    public long getInternalLastAccessedTime() {
        return internalLastAccessedTime;
    }

    public void setInternalLastAccessedTime(long value) {
        internalLastAccessedTime = value;
    }    

    /**
     * is the SipApplicationSession persistent
     */
    public boolean isReplicated() {
        return isReplicated;
    }

    /**
     * this sets the persistent flag
     */
    public void setReplicated(boolean value) {
        isReplicated = value;
    }

    public SipApplicationSessionExtraParams getExtraParameters() {
        return extraParams;
    }

    /**
     * Updates this SipApplicationSession with the given extra
     * parameters.
     *
     * @param extraParams The serialized SipApplicationSessionExtraParams
     * used to update this
     * SipApplicationSession
     */
    public void update(byte[] extraParamsState) {
        SipApplicationSessionExtraParams extraParams = null;

        if (extraParamsState != null) {
            try {
                extraParams = (SipApplicationSessionExtraParams) ReplicationState.getObjectValue(extraParamsState);
                setPFieldExpirationTime(extraParams.getExpirationTime(),
                                        false);
            } catch (Exception ex) {
                LogUtil.SSR_LOGGER.getLogger().log(Level.WARNING, "sas_extraparams_deserialization_error", toString());
                LogUtil.SSR_LOGGER.getLogger().log(Level.WARNING, ex.getMessage(), ex);
            }
        }
    }
    
    @Override
    protected ServletTimerImpl findServletTimerLocally(String anId, boolean loadDependencies) {
        // ignore the loadDependancies, this is a non-persistent timer only
        PersistentSipSessionManagerBase mgr = (PersistentSipSessionManagerBase)
        getSipSessionManager();    
        return mgr.findServletTimerFromCacheOnly(anId);
    }

    /**
     * Activates the object graph belonging to this SAS.
     */
    void activateGraph() {
        // getPFieldSipSessions just returns an iterable.
        // we must iterate over the SS children in order to load them
        for (SipSessionBase ss : getPFieldSipSessions(false)) {
            // for each we ensure that the DF is also loaded:
            ((HASipSession) ss).getFragmentId();
        }

        // for the ST children it is a collection, so no iteration is needed
        getPFieldApplicationTimers(false);
    }

    /**
     * Passivates this SipApplicationSession.
     */
    public void passivate() {
        if (logger.isLoggable(Level.FINE)) {
            logger.fine("IN " + this.getClass().getName() +
                        ">>passivate: " + this);
        }
        
        // assumption is that this is never called with an active UOW
        // assert(ReplicationUnitOfWork.getThreadLocalUnitOfWork() == null);
        ReplicationUnitOfWork unitOfWork = new ReplicationUnitOfWork();
        unitOfWork.lockApplicationSession(this);
        try {
            
            notifySessionWillPassivate();
            cancelNonPersistentTimers();

            // Remove from active cache
            PersistentSipSessionManagerBase mgr = (PersistentSipSessionManagerBase)
            getSipSessionManager();
            if (mgr != null) {
                mgr.removeSipApplicationSessionFromCache(this);
            }

            if (sasTimer != null) {
                sasTimer.cancel();
            }
            
        } finally {
            // Save modified entities (if any), and unlock the dialog
            unitOfWork.saveAndUnlock();
        }
    }
    
    @Override
    public boolean activate() {
        if (logger.isLoggable(Level.FINE)) {
            logger.fine("IN " + this.getClass().getName() +
                        ">>activate: " + this);
        }
        if (ReplicationUnitOfWork.getThreadLocalUnitOfWork() == null) {
            // if not invoked in the context where there is a UOW we create
            // a new one (e.g., when called after a loadAdvisory).
            ReplicationUnitOfWork unitOfWork = new ReplicationUnitOfWork();
            unitOfWork.lockApplicationSession(this);
            try {
                return super.activate();
            } finally {
                // Save modified entities (if any), and unlock the dialog
                unitOfWork.saveAndUnlock();
            }
        } else {
            // reuse the current UOW
            return super.activate();            
        }
    }

    /**
     * Validates this SipApplicationSession to see if it has been locked on
     * remote instance.
     */
    void validate() throws RemoteLockException {
        if (wasLockedOnRemoteInstance()) {
            throw new RemoteLockException(this +
                " has been locked on remote instance");
        }
    }

    boolean wasLockedOnRemoteInstance() {
        return wasLockedOnRemoteInstance;
    }

    /**
     * @serialData See serialized form version 1 in #readObject(ObjectInputStream in)
     *  
     * @param oos the stream to write the object members
     * @throws IOException is thrown when unsupported version is detected
     * @throws ClassNotFoundException
     */    
    private void writeObject(ObjectOutputStream oos) throws IOException {
        oos.writeShort(serializedFormVersion);
        
        oos.writeLong(getVersion());
        oos.writeObject(currentOwnerInstanceName);
        // When doing normal replication, never mark the artifact as
        // foreground-locked. Foreground lock needs to be considered and
        // reflected in serialized representation only if artifact is being
        // served from active cache.
        if (isBeingReplicated) {
            oos.writeBoolean(false);
        } else {
            oos.writeBoolean(isForegroundLocked());
        }
    }

    /**
     * @serialData first field is an short and represents the serializedFormVersion.<br>
     * <h3>Data layout for serializedFormVersion = 1 follows</h3>
     * 
     * <li>field is a <b>Long</b> and represents version used by HA replication system for this instance</li>
     * <li>field is a <b>String</b> and represents the name of the owner instance when this instance was serialized</li>
     * <li>field is a <b>Boolean</b> and represents if the object is foreground locked</li>
     *  
     * @param in the stream to read the object members
     * @throws IOException is thrown when unsupported version is detected
     * @throws ClassNotFoundException
     */    
    private void readObject(ObjectInputStream in)
        throws IOException, ClassNotFoundException {
        short readSerializedFormVersion = in.readShort();
        switch(readSerializedFormVersion) {
        case 1:
            version = new AtomicLong(in.readLong());
            previousOwnerInstanceName = (String) in.readObject();
            wasLockedOnRemoteInstance = in.readBoolean();
            currentOwnerInstanceName = ReplicationUtil.getInstanceName();
            extraParams = new SipApplicationSessionExtraParams(this);
        break;
        default:
            throw new IOException("Unable to deserialize into "
                    + this.getClass().getName()
                    + " due to unknown serializedFormVersion of "
                    + readSerializedFormVersion);
        }
    }

    public String toString() {
        return super.toString() + ":version:" + getVersion();
    }

    @Override
    protected void setPFieldExpirationTime(long expTime) {
        setPFieldExpirationTime(expTime, true);
    }

    private void setPFieldExpirationTime(long expTime,
                                         boolean addToUnitOfWork) {
        super.setPFieldExpirationTime(expTime);

        if (addToUnitOfWork) {
            // load all the children and set their expiration time
            for (SipSessionBase ss : getPFieldSipSessions()) {
                if (expTime > 0) {
                    ((HASipSession) ss).setPFieldExpirationTime(expTime + DELTA);
                } else {
                    ((HASipSession) ss).setPFieldExpirationTime(-1L);
                }
            }
            addToUnitOfWork();
        }
    }

    @Override
    public void setShouldBePersisted() {
        super.setShouldBePersisted();
        addToUnitOfWork();
    }

    @Override
    protected void setSipSessionShouldBePersisted(
        SipSessionDialogImpl sipSession) {
        super.setSipSessionShouldBePersisted(sipSession);
        setDirty(true);
    }
    
    /* (non-Javadoc)
     * @see com.ericsson.ssa.sip.SipApplicationSessionBase#addSession(com.ericsson.ssa.sip.SipSessionBase)
     * WHen adding a SS we have to update its expiration time in accordance with our own
     */
    @Override
    public void addSession(SipSessionBase anSession) {
        super.addSession(anSession);
        HASipSession hass = (HASipSession) anSession;
        if (hass != null) {
            long expTime = getPFieldExpirationTime();
            if (expTime <= 0) {
                // "infinite" expiration
                hass.setPFieldExpirationTime(expTime);
            } else {
                hass.setPFieldExpirationTime(expTime + DELTA);
            }
        }
    }

    /**
     * Sets the attribute with the given name to the given value and updates
     * the dirty flag to true
     *
     * @param name The attribute name to set
     * @param value The attribute value to set
     */
    @Override
    public void setPFieldApplicationAttribute(String name, Object value) {
        super.setPFieldApplicationAttribute(name, value);
        setDirty(true);
    }

    /**
     * Removes the attribute with the given name and updates the dirty flag
     * to true
     *
     * @param name The name of the attribute to remove
     */
    @Override
    protected void removePFieldApplicationAttribute(String key) {
        super.removePFieldApplicationAttribute(key);
        setDirty(true);
    }

    @Override
    protected void addPFieldApplicationTimer(ServletTimerImpl timer) {
        super.addPFieldApplicationTimer(timer);
        setDirty(true);
    }

    @Override
    protected void addPFieldHttpProtocolSession(HttpSession session) {
        super.addPFieldHttpProtocolSession(session);
        setDirty(true);
    }

    @Override
    protected void removePFieldApplicationTimer(ServletTimerImpl timer) {
        super.removePFieldApplicationTimer(timer);
        setDirty(true);
    }

    @Override
    protected void removePFieldHttpProtocolSession(HttpSession session) {
        super.removePFieldHttpProtocolSession(session);
        setDirty(true);
    }

    @Override
    protected void removePFieldSipProtocolSession(SipSessionBase session) {
        super.removePFieldSipProtocolSession(session);
        setDirty(true);
    }

    @Override
    protected void setPFieldApplicationName(String applicationName) {
        super.setPFieldApplicationName(applicationName);
        setDirty(true);
    }

    @Override
    protected void setPFieldBeKey(String beKey) {
        super.setPFieldBeKey(beKey);
        setDirty(true);
    }

    /**
    * checks/adds this entity to unitOfWork
    */
    protected void addToUnitOfWork() {
        if (shouldBePersisted()) {
            ReplicationUnitOfWork uow =
                ReplicationUnitOfWork.getThreadLocalUnitOfWork();
            if(uow != null) {
                uow.add(this);
            } else {
                throw new IllegalStateException("HASipApplicationSession>>addToUnitOfWork: unit of work missing");
            } 
        }
    }

    public void save() {
    	if (!isValid()) return;
    	
        PersistentSipSessionManagerBase sessionManager =
            (PersistentSipSessionManagerBase) getSipSessionManager();

        synchronized (getSasObjectLock()) {
            try {
                sessionManager.saveSipApplicationSession(this);

                if (LogUtil.SSR_LOGGER.getLogger().isLoggable(Level.FINE)) {
                    LogUtil.SSR_LOGGER.getLogger().log(Level.FINE, "Saved " + toString());
                }
            } catch (IOException e) {
                LogUtil.SSR_LOGGER.getLogger().log(Level.WARNING, "sas_save_error", toString());
                LogUtil.SSR_LOGGER.getLogger().log(Level.WARNING, e.getMessage(), e);
                return;
            }
        }
    }

    @Override
    protected void invalidate(boolean hasTimedOut) {
        super.invalidate(hasTimedOut);

        // remove from unit-of-work due its already gone from SessionManager
        if (shouldBePersisted()) {
            ReplicationUnitOfWork uow =
                ReplicationUnitOfWork.getThreadLocalUnitOfWork();
            if (uow != null) {
                uow.remove(this);
            }
        }        
    }

    /**
     * Processes a timeout event for this (active) SAS upon expiration.
     *
     * This method first checks if this SAS has expired
     * on the correct (according to DCR) instance. If it hasn't, then
     * a loadadvisory is sent to the new rightful owner instance, which causes
     * this SAS to migrate to it.
     *
     * An active SAS might expire on the wrong instance, if the cluster shape
     * has changed, and the SAS was not migrated to the new rightful owner
     * instance by active traffic.
     */
    public void timeout(GeneralTimer timer) {
        if (LogUtil.SSR_LOGGER.getLogger().isLoggable(Level.FINE)) {
            LogUtil.SSR_LOGGER.getLogger().log(Level.FINE, "Timeout on SAS " + toString());
        }
        if (!isValid()) {
            return;
        }
        String myInstanceName = ReplicationUtil.getInstanceName();
        String actualOwnerName =
            SipApplicationSessionUtil.getActualServerInstance(getId());
        if (myInstanceName.equals(actualOwnerName)) {
            // We have expired on the correct instance.
            // Instantiate a unit-of-work to collect any modified
            // entities/artifacts during timer invocation.
            ReplicationUnitOfWork unitOfWork = new ReplicationUnitOfWork();
            unitOfWork.lockApplicationSession(this);
            try {
            	// fire timer
            	super.timeout(timer);
            } finally {
                // Save modified entities (if any), and unlock the dialog
                unitOfWork.saveAndUnlock();
            }
        } else {
            // Issue a load advisory to the new rightful owner instance,
            // which will cause this SAS to migrate to it
            SipTransactionPersistentManager mgr =
                (SipTransactionPersistentManager) getSipSessionManager();
            if (mgr != null) {
                mgr.sendLoadAdvisorySipApplicationSession(getId(),
                    actualOwnerName);
            }
        }
    }

    void setIsBeingReplicated(boolean isBeingReplicated) {
        this.isBeingReplicated = isBeingReplicated;
    }

    boolean isBeingReplicated() {
        return isBeingReplicated;
    }
}
