/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.jvnet.glassfish.comms.replication.sessmgmt;

import org.apache.catalina.*;

import com.ericsson.ssa.sip.RemoteLockException;
import com.ericsson.ssa.sip.SipApplicationSessionImpl;
import com.ericsson.ssa.sip.SipApplicationSessionStore;
import com.ericsson.ssa.sip.SipApplicationSessionUtil;
import com.ericsson.ssa.sip.SipSessionManager;
import com.sun.appserv.ha.spi.BackingStore;
import com.sun.appserv.ha.spi.BackingStoreException;
import com.sun.appserv.ha.spi.SimpleMetadata;

import com.sun.enterprise.ee.web.sessmgmt.JxtaBackingStoreImpl;
import com.sun.enterprise.ee.web.sessmgmt.JxtaReplicationSender;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationUtil;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationState;
import com.sun.enterprise.ee.web.sessmgmt.StorePoolElement;

import java.io.*;
import java.util.logging.Level;
import org.apache.catalina.session.IOUtilsCaller;


/**
 *
 * @author Larry White
 */
public class SipApplicationSessionStoreImpl extends SipStoreBase
    implements StorePoolElement, SipApplicationSessionStore {
    
    protected static final String MODE_SIP = ReplicationState.MODE_SIP;    
    
    /** Creates a new instance of SipApplicationSessionStoreImpl */
    public SipApplicationSessionStoreImpl() {
    }
    
    /**
     * gets the BackingStore used for replicating SipApplicationSessions
     *
     * @return the BackingStore initialized for SipApplicationSessions
     *
     */    
    protected BackingStore getBackingStore() {
        SipTransactionPersistentManager mgr
            = (SipTransactionPersistentManager)this.getSipSessionManager();
        return mgr.getSipApplicationSessionBackingStore(); 
    }
    
    /**
     * Loads and returns the SipApplicationSession with the given id from
     * this store.
     *
     * @param id The id of the SipApplicationSession to load
     *
     * @return The SipApplicationSession with the given id, or null if not
     * found
     *
     * @throws IOException
     * @throws RemoteLockException
     */
    public SipApplicationSessionImpl load(String id)
        throws IOException, RemoteLockException {
        return load(id, null);
    }    

    /**
     * Loads and returns the SipApplicationSession with the given id from
     * this store.
     *
     * @param id The id of the SipApplicationSession to load
     * @param version The version of the SipApplicationSession to load
     *
     * @return The SipApplicationSession with the given id, or null if not
     * found
     *
     * @throws IOException
     * @throws RemoteLockException
     */
    public SipApplicationSessionImpl load(String id, String version) 
        throws IOException, RemoteLockException {
        if (SipReplicationUtil.isInstanceLoadBalancedByCLB() &&
                !SipApplicationSessionUtil.isLocal(id)) {
            throw new RemoteLockException("load of SipApplicationSession id:" +
                id + " aborted. Does not map to this instance");
        }        
        HASipApplicationSession result = null;
        if(id == null) {
            return result;
        }
        SipTransactionPersistentManager repMgr 
            = (SipTransactionPersistentManager)this.getSipSessionManager();
        ReplicationState localCachedState 
            = repMgr.removeFromSipApplicationSessionReplicationCache(id);
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("SipApplicationSessionStoreImpl>>load:localCachedState=" + localCachedState);                       
        }
        //check if we got a hit from our own replica cache
        //and if so return it immediately
        if(version != null && localCachedState != null) {
            long versionLong = ReplicationUtil.parseLong(version);
            try {
                if(localCachedState.getVersion() == versionLong) {
                    result = this.getSipApplicationSession(localCachedState);
                    result.validate();
                    //save result - save will reset dirty to false
                    result.setDirty(true, false);
                    save(result);                    
                    return result;
                } 
            } catch (ClassNotFoundException ex) {
                IOException ex1 = 
                    (IOException) new IOException("Error during load: " + ex.getMessage()).initCause(ex);
                throw ex1;                
            }
        }        
        ReplicationState broadcastResultState = findSessionViaBroadcast(id, version);
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("SipApplicationSessionStoreImpl>>load:broadcastResultState from broadcast=" + broadcastResultState);                       
        }        
        ReplicationState bestState 
            = ReplicationState.getBestResult(localCachedState, broadcastResultState);
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("SipApplicationSessionStoreImpl>>load:bestState=" + bestState);                       
        }
        if(bestState != null) {
            if(_logger.isLoggable(Level.FINE)) {
                _logger.fine("SipApplicationSessionStoreImpl>>load:bestStateVERSION=" + bestState.getVersion());
                _logger.fine("SipApplicationSessionStoreImpl>>load:bestStateSTATE=" + bestState.getState());
            }            
        }

        if(bestState != null && bestState.getState() != null) {
            if(_logger.isLoggable(Level.FINE)) {
                _logger.fine("SipApplicationSessionStoreImpl>>load:before deserializing bestState:ver=" + bestState.getVersion());                       
            }            
            try {
                result = this.getSipApplicationSession(bestState); 
            } catch (ClassNotFoundException ex) {
                IOException ex1 = 
                    (IOException) new IOException("Error during load: " + ex.getMessage()).initCause(ex);
                throw ex1;                
            }                        
            if(_logger.isLoggable(Level.FINE)) {
                _logger.fine("SipApplicationSessionStoreImpl>>load:after deserializing session:ver=" + result.getVersion());                       
            }
            // For SipApplicationSession this sending of load received ack
            // has been postponed to this point so we can check the
            // deserialized result to see if it was locked on the remote side
            // (this is done inside validate(), which may throw a
            // RemoteLockedException).
            // For most other artifacts the load received ack is done
            // immediately in JxtaBackingStoreImpl
            result.validate();
            //save result - save will reset dirty to false
            result.setDirty(true, false);
            save(result);            
            //send acknowledgement of load receipt
            //no response to wait for in this case
            JxtaReplicationSender sender 
                = JxtaReplicationSender.createInstance();
            ReplicationState loadReceivedState = 
                ReplicationState.createBroadcastLoadReceivedState(MODE_SIP,
                    (String)bestState.getId(), getApplicationId(),
                    bestState.getVersion(), ReplicationUtil.getInstanceName(),
                    SipTransactionPersistentManager.MESSAGE_BROADCAST_LOAD_RECEIVED_SAS);
            sender.sendBroadcastQuery(loadReceivedState);
        }
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("SipApplicationSessionStoreImpl>>load: " + result);
        }         
        return result;
    }
    
    private ReplicationState findSessionViaBroadcast(String id, String version) 
        throws IOException {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("SipApplicationSessionStoreImpl>>findSessionViaBroadcast");                       
        }        
        BackingStore replicator = this.getBackingStore();
        JxtaBackingStoreImpl jxtaReplicator = null;
        if(replicator instanceof JxtaBackingStoreImpl) {
            jxtaReplicator = (JxtaBackingStoreImpl)replicator;
        }
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("SipApplicationSessionStoreImpl>>findSessionViaBroadcast: replicator: " + replicator);                       
        }
        
        SimpleMetadata queryResult = null;
        try {
            //use version aware load if possible
            if(jxtaReplicator != null && version != null) {
                queryResult = (SimpleMetadata)jxtaReplicator.load(id, version);
            } else {
                queryResult = (SimpleMetadata)replicator.load(id);
            }
        } catch (BackingStoreException ex) {
            IOException ex1 = 
                (IOException) new IOException("Error during load: " + ex.getMessage()).initCause(ex);
            throw ex1;
        }
        if(queryResult == null) {
            return null;
        }
        byte[] extraParamState 
            = ReplicationUtil.getByteArray((Serializable) queryResult.getExtraParam());
        ReplicationState result 
            = new ReplicationState(MODE_SIP, id, this.getApplicationId(), 
                queryResult.getVersion(), 0L, 0L, null, null, 
                null, null, queryResult.getState(), null, extraParamState);
        return result;
    }
    
    /**
    * Given a byte[] containing session data, return a session
    * object
    *
    * @param state
    *   The byte[] with the session data
    *
    * @return
    *   A newly created SipApplicationSession for the given session data, and associated
    *   with this Manager
    */
    protected HASipApplicationSession getSipApplicationSession(ReplicationState replicationState) 
        throws IOException, ClassNotFoundException 
    {
        byte[] state = replicationState.getState();
        HASipApplicationSession _session = null;
        BufferedInputStream bis = null;
        ByteArrayInputStream bais = null;
        Loader loader = null;    
        ClassLoader classLoader = null;
        ObjectInputStream ois = null;
        SipSessionManager manager 
            = this.getSipSessionManager();
        Container container = manager.getContext();
        long version = 0L;
        IOUtilsCaller utilsCaller = null;
            
        bais = new ByteArrayInputStream(state);
        bis = new BufferedInputStream(bais);

        String id = (String)replicationState.getId();
        version = replicationState.getVersion();    

        if(_logger.isLoggable(Level.FINEST)) {
            _logger.finest("loaded session from replicationstore, length = "+state.length);
        }
        if (container != null) {
            loader = container.getLoader();
        }

        if (loader != null) {
            classLoader = loader.getClassLoader();
        }                   
        if (classLoader != null) {
            if( (utilsCaller = ReplicationUtil.getWebUtilsCaller()) != null) {
                try {
                    ois = utilsCaller.createObjectInputStream(bis, true, classLoader);
                } catch (Exception ex) {}
            }
        }
        if (ois == null) {
            ois = new ObjectInputStream(bis); 
        }
        if(ois != null) {               
            try {
                _session = (HASipApplicationSession)ois.readObject();
            } 
            finally {
                if (ois != null) {
                    try {
                        ois.close();
                        bis = null;
                    }
                    catch (IOException e) {
                    }
                }
            }
        }

        _session.update(replicationState.getContainerExtraParamsState());
        //currentOwnerInstanceName is in the manager already
        _session.setInternalLastAccessedTime(System.currentTimeMillis());
        _session.setVersion(version);
        _session.setDirty(false);
        _session.setReplicated(false);

        return _session;
    } 
    
    /**
     * Saves the given SipApplicationSession to this store.
     *
     * If a SipApplicationSession with the same id already exists in this
     * store, it will be replaced.
     *
     * @param sas The SipApplicationSession to be saved
     *
     * @exception IOException
     */
    public void save(SipApplicationSessionImpl sas) throws IOException { 
        HASipApplicationSession haSas = (HASipApplicationSession)sas;
        long previousVersion = haSas.getVersion();
        haSas.incrementVersion();
        haSas.setInternalLastAccessedTime(System.currentTimeMillis());
        try {
            if( haSas.isReplicated() && !haSas.isDirty() ) {
                this.updateContainerExtraParam(haSas);
            } else {
                haSas.setIsBeingReplicated(true);
                this.doSave(haSas);
                haSas.setReplicated(true);
            }
            haSas.setDirty(false);
        } catch (IOException ex) {
            haSas.setVersion(previousVersion);
            throw ex;
        } finally {
            haSas.setIsBeingReplicated(false);
        }
    }

    /**
     * Saves the given SipApplicationSession to this store.
     *
     * If a SipApplicationSession with the same id already exists in this
     * store, it will be replaced.
     *
     * @param sas The SipApplicationSession to be saved
     *
     * @exception IOException
     */
    public void doSave(HASipApplicationSession haSas) throws IOException {
        byte[] sessionState = ReplicationUtil.getByteArray(haSas);
        BackingStore replicator = this.getBackingStore();
        SimpleMetadata simpleMetadata =        
            new SimpleMetadata(
                haSas.getVersion(),
                haSas.getInternalLastAccessedTime(),
                0L, //maxinactiveinterval
                sessionState,
                haSas.getExtraParameters() //containerExtraParam
            );
                
        try {        
            replicator.save(haSas.getId(), //id
                    simpleMetadata);  //SimpleMetadata 
        } catch (BackingStoreException ex) {
            IOException ex1 = 
                (IOException) new IOException("Error during save: " + ex.getMessage()).initCause(ex);
            throw ex1;
        }        
    }
    
    /**
     * Removes the SipApplicationSession with the given id from this store.
     *
     * @param id The id of the SipApplicationSession to be removed
     *
     * @exception IOException
     */
    public void remove(String id) throws IOException {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("SipApplicationSessionStoreImpl>>remove" + " id: " + id);                       
        } 
        BackingStore replicator = this.getBackingStore();
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("SipApplicationSessionStoreImpl>>remove: replicator: " + replicator);                       
        }        
        try {
            replicator.remove(id);
        } catch (BackingStoreException ex) {
            IOException ex1 = 
                (IOException) new IOException("Error during remove: " + ex.getMessage()).initCause(ex);
            throw ex1;       
        }
    }
    
    void sendLoadAdvisory(String id, String instanceName) 
        throws IOException {    
        //send load advisory to instanceName
        SipTransactionPersistentManager mgr
            = (SipTransactionPersistentManager)getSipSessionManager();
        String theCommand = mgr.MESSAGE_BROADCAST_LOAD_ADVISORY_SAS;
        ReplicationState loadAdvisoryState 
            = ReplicationState.createUnicastLoadAdvisoryState(MODE_SIP, id, this.getApplicationId(), 0L, mgr.getInstanceName(), theCommand);     
        JxtaReplicationSender sender 
            = JxtaReplicationSender.createInstance();
        sender.sendOverPropagatedPipe(loadAdvisoryState, instanceName, false);

    }    
    
    /**
     * update the lastaccess time of the specified SipApplicationSession into this Store.
     *
     * @param sas SipApplicationSession to be saved
     *
     * @exception IOException if an input/output error occurs
     */    
    public void updateContainerExtraParam(HASipApplicationSession sas) throws IOException {
        BackingStore replicator = this.getBackingStore();
        JxtaBackingStoreImpl jxtaReplicator = null;
        if(replicator instanceof JxtaBackingStoreImpl) {
            jxtaReplicator = (JxtaBackingStoreImpl)replicator;
        }
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("SipApplicationSessionStoreImpl>>updateContainerExtraParam: replicator: " + replicator);                       
        }         
        try {            
            replicator.updateContainerExtraParam(sas.getId(), //id
                    sas.getInternalLastAccessedTime(), //internallastaccesstime
                    sas.getVersion(), //version 
                    sas.getExtraParameters());  //containerExtraParams
        } catch (BackingStoreException ex) {
            IOException ex1 = 
                (IOException) new IOException("Error during updateContainerExtraParam: " + ex.getMessage()).initCause(ex);
            throw ex1;
        }
    }       

    /**
     * Removes all SipApplicationSessions from this store.
     */
    public void clear() throws IOException {
        //FIXME
    }

    /**
     * clean up resources
     */
    public void cleanup() {
        //FIXME        
    }

    /**
     * Removes any expired SipApplicationSessions from this store.
     *
     * This method is invoked by the background reaper thread.
     */
    public void processExpires() {
        SipTransactionPersistentManager replicationMgr =
            (SipTransactionPersistentManager) this.getSipSessionManager();
        if(!(replicationMgr.isThirdPartyBackingStoreInUse())) {
            replicationMgr.processExpiredSipApplicationSessionReplicas();
        } else {
            replicationMgr.processExpiredSipApplicationSessionReplicasThirdPartySPI();
        }
    }
    
    void sendRollingUpgradeAdvisory(String instanceName) 
        throws IOException {    
        //send rolling upgrade advisory to instanceName
        SipTransactionPersistentManager mgr
            = (SipTransactionPersistentManager)getSipSessionManager();
        String theCommand = mgr.MESSAGE_BROADCAST_ROLLING_UPGRADE_ADVISORY;
        ReplicationState loadAdvisoryState 
            = ReplicationState.createUnicastLoadAdvisoryState(MODE_SIP, "id", mgr.getApplicationId(), 0L, ReplicationUtil.getInstanceName(), theCommand);     
        JxtaReplicationSender sender 
            = JxtaReplicationSender.createInstance();
        sender.sendOverPropagatedPipe(loadAdvisoryState, instanceName, false);

    }     
       
}
