/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.jvnet.glassfish.comms.replication.sessmgmt;

import com.ericsson.ssa.sip.RemoteLockException;
import com.ericsson.ssa.sip.SipSessionDialogImpl;
import com.ericsson.ssa.sip.SipSessionManager;
import com.ericsson.ssa.sip.SipSessionStore;
import com.sun.appserv.ha.spi.BackingStore;
import com.sun.appserv.ha.spi.BackingStoreException;
import com.sun.appserv.ha.spi.SimpleMetadata;

import com.sun.enterprise.ee.web.sessmgmt.JxtaBackingStoreImpl;
import com.sun.enterprise.ee.web.sessmgmt.JxtaReplicationSender;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationUtil;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationState;
import com.sun.enterprise.ee.web.sessmgmt.StorePoolElement;

import java.util.logging.Level;
import java.util.logging.Logger;
import java.io.*;

import org.apache.catalina.*;
import org.apache.catalina.session.IOUtilsCaller;


/**
 *
 * @author Larry White
 */
public class SipSessionStoreImpl extends SipStoreBase
    implements StorePoolElement, SipSessionStore {
        
    /** Creates a new instance of SipSessionStoreImpl */
    public SipSessionStoreImpl() {
    }
    
    /**
     * gets the BackingStore used for replicating SipSessions
     *
     * @return the BackingStore initialized for SipSessions
     *
     */    
    protected BackingStore getBackingStore() {
        SipTransactionPersistentManager mgr
            = (SipTransactionPersistentManager)this.getSipSessionManager();
        return mgr.getSipSessionBackingStore(); 
    }      

    /**
     * Loads and returns the SipSession with the given id from this session
     * store.
     *
     * @param id The id of the SipSession to load
     *
     * @return The SipSession with the given id, or null if not found
     *
     * @throws IOException
     * @throws RemoteLockException
     */
    public SipSessionDialogImpl load(String id, boolean loadDependencies) 
            throws IOException, RemoteLockException {
        return load(id, null, loadDependencies);
    }

    /**
     * Loads and returns the SipSession with the given id from this session
     * store.
     *
     * @param id The id of the SipSession to load
     * @param version the version of the SipSession to load
     *
     * @return The SipSession with the given id, or null if not found
     *
     * @throws IOException
     * @throws RemoteLockException
     */
    public SipSessionDialogImpl load(String id, String version,
                                     boolean loadDependencies) 
        throws IOException, RemoteLockException {
        HASipSession result = null;
        if(id == null) {
            return result;
        }
        SipTransactionPersistentManager repMgr 
            = (SipTransactionPersistentManager)this.getSipSessionManager();
        ReplicationState localCachedState 
            = repMgr.removeFromSipSessionReplicationCache(id);
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("SipSessionStoreImpl>>load:localCachedState=" + localCachedState);                       
        }
        //check if we got a hit from our own replica cache
        //and if so return it immediately
        if(version != null && localCachedState != null) {
            long versionLong = ReplicationUtil.parseLong(version);
            try {
                if(localCachedState.getVersion() == versionLong) {
                    result = this.getSipSession(localCachedState);
                    result.validate(loadDependencies);
                    //save result - save will reset dirty to false
                    result.setDirty(true, false);
                    save(result);                     
                    return result;
                } 
            } catch (ClassNotFoundException ex) {
                IOException ex1 = 
                    (IOException) new IOException("Error during load: " + ex.getMessage()).initCause(ex);
                throw ex1;                
            }
        }        
        ReplicationState broadcastResultState = findSessionViaBroadcast(id, version);
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("SipSessionStoreImpl>>load:broadcastResultState from broadcast=" + broadcastResultState);                       
        }        
        ReplicationState bestState 
            = ReplicationState.getBestResult(localCachedState, broadcastResultState);
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("SipSessionStoreImpl>>load:bestState=" + bestState);                       
        }
        if(bestState != null) {
            if(_logger.isLoggable(Level.FINE)) {
                _logger.fine("SipSessionStoreImpl>>load:bestStateVERSION=" + bestState.getVersion());
                _logger.fine("SipSessionStoreImpl>>load:bestStateSTATE=" + bestState.getState());
            }            
        }

        if(bestState != null && bestState.getState() != null) {
            if(_logger.isLoggable(Level.FINE)) {
                _logger.fine("SipSessionStoreImpl>>load:before deserializing bestState:ver=" + bestState.getVersion());                       
            }            
            try {
                result = this.getSipSession(bestState); 
            } catch (ClassNotFoundException ex) {
                IOException ex1 = 
                    (IOException) new IOException("Error during load: " + ex.getMessage()).initCause(ex);
                throw ex1;                
            }                        
            if(_logger.isLoggable(Level.FINE)) {
                _logger.fine("SipSessionStoreImpl>>load:after deserializing session:ver=" + result.getVersion());                       
            }
            // For SipSession this sending of load received ack
            // has been postponed to this point so we can check the
            // deserialized result to see if it was locked on the remote side
            // (this is done inside validate(), which may throw a
            // RemoteLockedException).
            // For most other artifacts the load received ack is done
            // immediately in JxtaBackingStoreImpl
            result.validate(loadDependencies);
            //save result - save will reset dirty to false
            result.setDirty(true, false);
            save(result);            
            //send acknowledgement of load receipt
            //no response to wait for in this case
            JxtaReplicationSender sender 
                = JxtaReplicationSender.createInstance();
            ReplicationState loadReceivedState = 
                ReplicationState.createBroadcastLoadReceivedState(MODE_SIP,
                    (String)bestState.getId(), this.getApplicationId(),
                    bestState.getVersion(), ReplicationUtil.getInstanceName(),
                    SipTransactionPersistentManager.MESSAGE_BROADCAST_LOAD_RECEIVED_SIP_SESSION);
            sender.sendBroadcastQuery(loadReceivedState);
        }
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("SipSessionStoreImpl>>load: " + result);                       
        }         
        return result;
    } 
    
    private ReplicationState findSessionViaBroadcast(String id, String version) 
        throws IOException {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("SipSessionStoreImpl>>findSessionViaBroadcast");                       
        }        
        BackingStore replicator = this.getBackingStore();
        JxtaBackingStoreImpl jxtaReplicator = null;
        if(replicator instanceof JxtaBackingStoreImpl) {
            jxtaReplicator = (JxtaBackingStoreImpl)replicator;
        }
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("SipSessionStoreImpl>>findSessionViaBroadcast: replicator: " + replicator);                       
        }
        
        SimpleMetadata queryResult = null;
        try {
            //use version aware load if possible
            if(jxtaReplicator != null && version != null) {
                queryResult = (SimpleMetadata)jxtaReplicator.load(id, version);
            } else {
                queryResult = (SimpleMetadata)replicator.load(id);
            }
        } catch (BackingStoreException ex) {
            IOException ex1 = 
                (IOException) new IOException("Error during load: " + ex.getMessage()).initCause(ex);
            throw ex1;
        }
        if(queryResult == null) {
            return null;
        }
        byte[] extraParamState 
            = ReplicationUtil.getByteArray((Serializable) queryResult.getExtraParam());
        ReplicationState result 
            = new ReplicationState(MODE_SIP, id, this.getApplicationId(), 
                queryResult.getVersion(), 0L, 0L, null, null, 
                null, null, queryResult.getState(), null, extraParamState);
        return result;
    }    
    
    /**
    * Given a byte[] containing session data, return a session
    * object
    *
    * @param state
    *   The byte[] with the session data
    *
    * @return
    *   A newly created SipSession for the given session data, and associated
    *   with this Manager
    */
    protected HASipSession getSipSession(ReplicationState replicationState) 
        throws IOException, ClassNotFoundException 
    {
        byte[] state = replicationState.getState();
        HASipSession _session = null;
        BufferedInputStream bis = null;
        ByteArrayInputStream bais = null;
        Loader loader = null;    
        ClassLoader classLoader = null;
        ObjectInputStream ois = null;
        SipSessionManager manager 
            = this.getSipSessionManager();
        Container container = manager.getContext();
        long version = 0L;
        IOUtilsCaller utilsCaller = null;
            
        bais = new ByteArrayInputStream(state);
        bis = new BufferedInputStream(bais);

        String id = (String)replicationState.getId();
        version = replicationState.getVersion();    

        if(_logger.isLoggable(Level.FINEST)) {
            _logger.finest("loaded session from replicationstore, length = "+state.length);
        }
        if (container != null) {
            loader = container.getLoader();
        }

        if (loader != null) {
            classLoader = loader.getClassLoader();
        }                   
        if (classLoader != null) {
            if( (utilsCaller = ReplicationUtil.getWebUtilsCaller()) != null) {
                try {
                    ois = utilsCaller.createObjectInputStream(bis, true, classLoader);
                } catch (Exception ex) {}
            }
        }
        if (ois == null) {
            ois = new ObjectInputStream(bis); 
        }
        if(ois != null) {               
            try {
                _session = (HASipSession)ois.readObject();
            } 
            finally {
                if (ois != null) {
                    try {
                        ois.close();
                        bis = null;
                    }
                    catch (IOException e) {
                    }
                }
            }
        } 
        
        _session.update(replicationState.getContainerExtraParamsState());
        //currentOwnerInstanceName is in the manager already
        _session.setInternalLastAccessedTime(System.currentTimeMillis());
        _session.setVersion(version);
        _session.setDirty(false);
        _session.setReplicated(false);

        return _session;
    }
    
    /**
     * Saves the given SipSession to this store.
     *
     * If a SipSession with the same id already exists in this session
     * store, it will be replaced.
     *
     * @param session The SipSession to be saved
     *
     * @exception IOException
     */
    public void save(SipSessionDialogImpl session) throws IOException { 
        HASipSession haSession = (HASipSession)session;
        long previousVersion = haSession.getVersion();
        haSession.incrementVersion();
        haSession.setInternalLastAccessedTime(System.currentTimeMillis());
        try {
            if( haSession.isReplicated() && !haSession.isDirty() ) {
                this.updateContainerExtraParam(haSession);
            } else {
                haSession.setIsBeingReplicated(true);
                this.doSave(haSession);
                haSession.setReplicated(true);
            }
            haSession.setDirty(false);
        } catch (IOException ex) {
            haSession.setVersion(previousVersion);
            throw ex;
        } finally {
            haSession.setIsBeingReplicated(false);
        }
    }    

    /**
     * Saves the given SipSession to this store.
     *
     * If a SipSession with the same id already exists in this session
     * store, it will be replaced.
     *
     * @param session The SipSession to be saved
     *
     * @exception IOException
     */
    public void doSave(HASipSession haSession) throws IOException {
        byte[] sessionState = ReplicationUtil.getByteArray(haSession);
        BackingStore replicator = this.getBackingStore();
        SimpleMetadata simpleMetadata =        
            new SimpleMetadata(
                haSession.getVersion(),
                haSession.getInternalLastAccessedTime(),
                0L, //maxinactiveinterval
                sessionState,
                haSession.getExtraParameters() //containerExtraParam
            );
                
        try {        
            replicator.save(haSession.getId(), //id
                    simpleMetadata);  //SimpleMetadata 
        } catch (BackingStoreException ex) {
            IOException ex1 = 
                (IOException) new IOException("Error during save: " + ex.getMessage()).initCause(ex);
            throw ex1;
        } 
    }
    
    /**
     * Removes the SipSession with the given id from this store.
     *
     * @param id The id of the SipSession to be removed
     *
     * @exception IOException
     */
    public void remove(String id) throws IOException {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("SipSessionStoreImpl>>remove" + " id: " + id);                       
        } 
        BackingStore replicator = this.getBackingStore();
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("SipSessionStoreImpl>>remove: replicator: " + replicator);                       
        }        
        try {
            replicator.remove(id);
        } catch (BackingStoreException ex) {
            IOException ex1 = 
                (IOException) new IOException("Error during remove: " + ex.getMessage()).initCause(ex);
            throw ex1;       
        }
    } 
    
    /**
     * update the lastaccess time of the specified SipSession into this Store.
     *
     * @param sipSession SipSessionDialogImpl to be saved
     *
     * @exception IOException if an input/output error occurs
     */    
    public void updateContainerExtraParam(HASipSession sipSession)
            throws IOException {
        BackingStore replicator = this.getBackingStore();
        JxtaBackingStoreImpl jxtaReplicator = null;
        if(replicator instanceof JxtaBackingStoreImpl) {
            jxtaReplicator = (JxtaBackingStoreImpl)replicator;
        }
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("SipSessionStoreImpl>>updateContainerExtraParam: replicator: " + replicator);                       
        }         
        try {
            replicator.updateContainerExtraParam(sipSession.getId(), //id
                    sipSession.getInternalLastAccessedTime(), //internallastaccesstime
                    sipSession.getVersion(), //version
                    sipSession.getExtraParameters()); //containerExtraParams
        } catch (BackingStoreException ex) {
            IOException ex1 = 
                (IOException) new IOException("Error during updateContainerExtraParam: " + ex.getMessage()).initCause(ex);
            throw ex1;
        }
    }        

    /**
     * Removes all SipSessions from this store.
     */
    public void clear() throws IOException {
        //FIXME
    }

    /**
     * clean up resources
     */
    public void cleanup() {
        //FIXME        
    }

    /**
     * Removes any expired SipSessions from this store.
     *
     * This method is invoked by the background reaper thread.
     */
    public void processExpires() {
        SipTransactionPersistentManager replicationMgr =
            (SipTransactionPersistentManager) this.getSipSessionManager();
        if(!(replicationMgr.isThirdPartyBackingStoreInUse())) {
            replicationMgr.processExpiredSipSessionReplicas();
        } else {
            replicationMgr.processExpiredSipSessionReplicasThirdPartySPI();
            //removeExpiredSessions();
        }
    }
    
    /** This method deletes all the sip  
     * sessions corresponding to the "appId" that should be expired
     * @return number of removed sessions
     */    
    public int removeExpiredSessions() {        
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("IN SipSessionStoreImpl>>removeExpiredSessions");
        }
        int result = 0;
        BackingStore replicator = getBackingStore();
        try {
            result = replicator.removeExpired();
        } catch (BackingStoreException ex) {
            _logger.log(Level.WARNING,
                "unable to remove expired sip session replicas", ex);
        }
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("SipSessionStoreImpl>>removeExpiredSessions():number of expired sessions = " + result);
        }
        return result;

    }   
}
