/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package com.ericsson.ssa.container;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

// inserted by hockey (automatic)
import java.util.logging.Logger;
import org.jvnet.glassfish.comms.util.LogUtil;


/**
 * This class represents a pool of connections to a specific host and port.
 * A client that needs a connection may obtain one by calling
 * getConnection(). If no connection is available this method will block
 * until a connection is released by another tread by calling releaseConnection().
 * Obtained connections MUST be released by calling releaseConnection().
 *
 * @author epiesan
 * @since Apr 16, 2006
 */
public class HttpConnectionPool {
    private static Logger log = LogUtil.SIP_LOGGER.getLogger();
    private final static int DEFAULT_NUM_CONNECTIONS = 2;
    private HttpHost host;
    private List<HttpConnection> allConnections;
    private BlockingQueue<HttpConnection> freeConnections;

    /**
     * Constructor for HttpConnectionPool.
     * Creates a pool with default values for max connections
     * and idle connection timeout.
     *
     * @param host The host this pool manages connections for.
     */
    public HttpConnectionPool(final HttpHost host) {
        this(host, DEFAULT_NUM_CONNECTIONS);
    }

    /**
     * Constructor for HttpConnectionPool.
     *
     * @param host The host this pool manages connections for.
     * @param numConnections The number of connections in this pool.
     */
    public HttpConnectionPool(final HttpHost host, int numConnections) {
        this(host, numConnections, -1);
    }

    /**
     * Constructor for HttpConnectionPool.
     *
     * @param host The host this pool manages connections for.
     * @param numConnections The number of connections in this pool.
     * @param socketTimeout The socket timeout value for the connections
     *   in this pool.
     */
    public HttpConnectionPool(final HttpHost host, int numConnections,
        int socketTimeout) {
        this.host = host;
        this.allConnections = new ArrayList<HttpConnection>();
        this.freeConnections = new LinkedBlockingQueue<HttpConnection>();

        // Create all connections initially.
        createConnections(host, numConnections, socketTimeout);
    }

    /**
     * Obtains a connection from the pool.
     * If no connection is currently available this method will block
     * until a connection is available (released by another thread).
     *
     * @return A free connection from the pool.
     */
    public HttpConnection getConnection() {
        return getConnection(0);
    }

    /**
     * Obtains a connection from the pool.
     * If no connection is currently available this method will block
     * the specified amount of time or forever if zero is given
     * as the timeout value.
     *
     * @param The maximum time to block. Zero means block until
     *   a connection is available.
     * @return A free connection from the pool or null
     */
    public HttpConnection getConnection(int timeout) {
        while (true) {
            try {
                if (log.isLoggable(Level.FINE)) {
                    if (freeConnections.peek() == null) {
                        log.log(Level.FINE,
                            "No connection is currently available. " +
                            "The current thread will block until another " +
                            "thread releases a connection.");
                    }
                }

                if (timeout > 0) {
                    // Wait only a limited time before giving up.
                    return freeConnections.poll(timeout, TimeUnit.MILLISECONDS);
                } else {
                    // Wait forever if necessary.
                    return freeConnections.take();
                }
            } catch (InterruptedException e) {
                // Thread was interrupted. Try again!
            }
        }
    }

    /**
     * Release a connection that was previously obtained from this pool.
     *
     * @param connection The connection to release.
     * @throws IllegalArgumentException Thrown if the passed connection
     *   is not managed by this pool.
     * @throws IllegalStateException Thrown if the passed connection has
     *   already been released.
     */
    public void releaseConnection(HttpConnection connection)
        throws IllegalArgumentException, IllegalStateException {
        if (!allConnections.contains(connection)) {
            throw new IllegalArgumentException("The connection cannot be" +
                " releases since it is not managed by this pool.");
        }

        if (freeConnections.contains(connection)) {
            throw new IllegalStateException(
                "The connection has already been released.");
        }

        // Add the connection to the list of free connections.
        freeConnections.offer(connection);
    }

    /**
     * Shuts down all connections in the pool.
     * Still not released connections will also be closed.
     *
     * @throws IllegalStateException If there still are active connections.
     */
    public void shutdown() throws IllegalStateException {
        try {
            if (freeConnections.size() < allConnections.size()) {
                throw new IllegalStateException("Some connections to host '" +
                    host + "' are still active but will be closed anyway.");
            }
        } finally {
            for (HttpConnection connection : allConnections) {
                connection.close();
            }

            allConnections.clear();
            freeConnections.clear();
        }
    }

    /**
     * Create the initial HttpConnection instances.
     *
     * @param host The host this pool manages connections for.
     * @param numConnections The number of connections in this pool.
     * @param socketTimeout The socket timeout value for the
     *   connections in this pool.
     */
    private void createConnections(final HttpHost host, int numConnections,
        int socketTimeout) {
        for (int i = 0; i < numConnections; i++) {
            HttpConnection connection;

            if (socketTimeout < 0) {
                connection = new HttpConnection(host, this);
            } else {
                connection = new HttpConnection(host, socketTimeout, this);
            }

            this.allConnections.add(connection);
            this.freeConnections.add(connection);
        }
    }
}
