/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package com.ericsson.ssa.container;

import com.ericsson.ssa.container.startup.SipMonitoring;
import com.ericsson.ssa.sip.Header;
import com.ericsson.ssa.sip.Layer;
import com.ericsson.ssa.sip.LayerHelper;
import com.ericsson.ssa.sip.SipParser;
import com.ericsson.ssa.sip.SipServletMessageImpl;

//import com.ericsson.ssa.sip.SipServletMessageImpl.SipMessageType;
import com.ericsson.ssa.sip.SipServletRequestImpl;
import com.ericsson.ssa.sip.SipServletResponseImpl;
import com.ericsson.ssa.sip.dns.SipTransports;
import com.ericsson.ssa.sip.dns.TargetResolver;
import com.ericsson.ssa.sip.dns.TargetTuple;

import com.sun.grizzly.Context;
import com.sun.grizzly.Controller.Protocol;
import com.sun.grizzly.ProtocolFilter;
import com.sun.grizzly.SelectorHandler;
import com.sun.grizzly.filter.ReadFilter;
import com.sun.grizzly.util.ByteBufferInputStream;
import com.sun.grizzly.util.WorkerThread;

import java.io.IOException;

import java.net.DatagramSocket;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketAddress;

import java.nio.ByteBuffer;
import java.nio.channels.DatagramChannel;
import java.nio.channels.SelectionKey;
import java.nio.channels.SocketChannel;

import java.security.cert.X509Certificate;

import java.util.ArrayList;
import java.util.concurrent.Callable;
import java.util.logging.Level;
import org.jvnet.glassfish.comms.util.LogUtil;
import java.util.logging.Logger;


/**
 *
 * @author ekrigro
 */
public class MessageProcessorFilter implements ProtocolFilter, Callable {
    private final static Logger logger = LogUtil.SIP_LOGGER.getLogger();
    private final Layer _networkHandler;
    
    /**
     * Should use a attribute from request-processing rather
     * than a system property. default is false.
     */
    private static final boolean finalRequestOnContainerThreadPool = 
        Boolean.getBoolean("org.jvnet.glassfish.comms.finalRequestOnContainerThreadPool");
    /** Creates a new instance of MessageProcessorFilter */
    public MessageProcessorFilter(Layer layer) {
        _networkHandler = layer;
    }

    public boolean execute(final Context ctx) throws IOException {
        SipServletMessageImpl parsedMessage = null;
        Socket s = null;
        InetSocketAddress remoteAddress = null;
        SipServletMessageImpl _message = null;
        boolean invokeNextFilter = true;
        
        final WorkerThread workerThread = ((WorkerThread) Thread.currentThread());

        ByteBuffer buffer = workerThread.getByteBuffer();
        SipParser _parser = SipParser.getInstance();

        Protocol prot = ctx.getProtocol();
        final SelectionKey key = ctx.getSelectionKey();
        final SelectorHandler handler = ctx.getSelectorHandler();

        TargetTuple remote = null;
        InetSocketAddress local = null;
        X509Certificate[] x509Certs = null;

        switch (prot) {
        case TCP:
            s = ((SocketChannel) key.channel()).socket();
            remoteAddress = (InetSocketAddress) s.getRemoteSocketAddress();
            remote = new TargetTuple(SipTransports.TCP_PROT, remoteAddress);
            local = (InetSocketAddress) s.getLocalSocketAddress();

            break;

        case UDP:
            ctx.setKeyRegistrationState(Context.KeyRegistrationState.NONE);
            handler.register(key, SelectionKey.OP_READ);

            DatagramSocket d = ((DatagramChannel) key.channel()).socket();                        
            remoteAddress = (InetSocketAddress) ctx.getAttribute(ReadFilter.UDP_SOCKETADDRESS);
            remote = new TargetTuple(SipTransports.UDP_PROT, remoteAddress);
            local = (InetSocketAddress) d.getLocalSocketAddress();
       
            break;

        case TLS:
            s = ((SocketChannel) key.channel()).socket();
            remoteAddress = (InetSocketAddress) s.getRemoteSocketAddress();
            remote = new TargetTuple(SipTransports.TLS_PROT, remoteAddress);
            local = (InetSocketAddress) s.getLocalSocketAddress();

            Object[] certs = (Object[]) ctx.removeAttribute(GrizzlyNetworkManager.SIP_CERTS);

            if ((certs != null) && (certs.length > 0)) {
                ArrayList<X509Certificate> al = new ArrayList<X509Certificate>();

                for (int i = 0; i < certs.length; i++) {
                    if (certs[i] instanceof X509Certificate) {
                        al.add((X509Certificate) certs[i]);
                    } else {
                        logger.log(
                            Level.WARNING, "Certificate provided by TLS implementation is not of type X509. " +
                            certs[i].getClass());
                    }
                }

                x509Certs = al.toArray(new X509Certificate[al.size()]);
            }

            break;
        }

        try {
            int initialSize = 0;
            buffer.flip();

            int remaining = buffer.remaining();
            
            while (((remaining > 0) && (initialSize != remaining)) ||
                    (parsedMessage == null)) {
                initialSize = remaining;
		if (_message == null){
                    skipNewLines(buffer);
                }
                if (!buffer.hasRemaining()) return invokeNextFilter;
                
                parsedMessage = _parser.parseMessage(_message, buffer, local,remote, null);
                remaining = buffer.remaining();
                
                if ((parsedMessage != null) &&
                        parsedMessage.isMessageComplete() && remaining > 0) {
                    final SipServletMessageImpl msg = parsedMessage;                  
                    parsedMessage = null;
                    _message = null;
                    final InetSocketAddress _remoteAddress = remoteAddress;
                    msg.setCertificate(x509Certs);
                    SipContainerThreadPool.getInstance().execute(new Callable() {
                        public Object call() throws Exception {
                            processMessage(msg, key, handler, _remoteAddress);
                            return null;
                        }
                    });
                    continue;
                }
                
                if ((parsedMessage == null) || !parsedMessage.isMessageComplete()) {
                    // UDP packet are *always* read using a single read, hence
                    // no need to try another read using a temporary Selector.
                    if (prot != Protocol.UDP) {
                        if (logger.isLoggable(Level.FINE)) {
                            logger.log(Level.FINE, "The parsed request is not complete " +
                                "- reading more! (" + buffer.position() + ')');
                        }
						initialSize = remaining;
                        if (buffer.hasRemaining()){
                            buffer.compact();
                        } else {
                            buffer.clear();
                        }
                                       
                        // The thread might block when reading more bytes using 
                        // a temporary Selector.
                        ByteBufferInputStream inputStream = new ByteBufferInputStream();
                        inputStream.setSecure((prot == Protocol.TLS));
                        inputStream.setSelectionKey(ctx.getSelectionKey());
                        inputStream.setReadTimeout(15000);

                        int nRead = inputStream.read(buffer);
                        if (nRead <= 0) {
                            logger.log(Level.SEVERE, "Unable to read more bytes after " +
                                    "30 seconds pause. Closing the connection.");
                            // Do not invoke the next ProtocolFilter, if any.
                            return true;
                        }
                            
                        remaining = buffer.remaining();
                        _message = parsedMessage;
                    } else {
                        return true;
                    }
                }
            }
            if (parsedMessage != null) {
                parsedMessage.setCertificate(x509Certs);                
                if (prot == Protocol.UDP) {
                    processMessage(parsedMessage, key, handler, remoteAddress);
				} else {
					if (!finalRequestOnContainerThreadPool){
                    /**
                     * The key has to be registered back here for TCP, because
                     * the request will be processesd in the same worker thread 
                     * and further requests cannot be read until this request
                     * is processed by the app. 
                     */
                    ctx.setKeyRegistrationState(Context.KeyRegistrationState.NONE);
                    handler.register(key, SelectionKey.OP_READ);
                    processMessage(parsedMessage, key, handler, remoteAddress);
                } else {
                    /**
                     * This is necessary if we have to ensure (mitigate
					 * ) that the messages are processed in same order.                      * TODO : To register the key, the buffer 
                     * has to be copied into another buffer because this buffer
                     * will be reused when this worker thread is used for processing.
                     */ 
                    final SipServletMessageImpl finalProcessedMessage = parsedMessage;
                    final InetSocketAddress finalRemoteAddress = remoteAddress;
                     SipContainerThreadPool.getInstance().execute(new Callable() {
                        public Object call() throws Exception {
                            processMessage(finalProcessedMessage, 
                                    key, handler, finalRemoteAddress);
                            return null;
                        }
                    });
                }
			}
           }
        } catch (Throwable t) {
            invokeNextFilter = false;
            if (t instanceof Exception) {
                if (logger.isLoggable(Level.WARNING)) {
                    logger.log(Level.WARNING,
                        "Exception processing request " + t.getMessage(), t);
                }
            } else {
                // FIXME alarm for e.g. OutOfMemoryError
                logger.log(Level.SEVERE, "Caught Throwable: ", t);
            }

            if (SipMonitoring.isEnabled(SipMonitoring.NETWORK_MANAGER)) {
                ((NetworkManager)_networkHandler).incrEasInvalidSipMessages();
            }
        } finally {
            buffer.clear();
            _message = null;
        }

        return invokeNextFilter;
    }

    public boolean postExecute(Context context) throws IOException {
        return true;
    }

    protected void processMessage(final SipServletMessageImpl message,
        SelectionKey key, SelectorHandler handler, SocketAddress remoteAdress) {
        if (message.getMessageType() == SipServletMessageImpl.SipMessageType.SipRequest) {
            SipServletRequestImpl req = (SipServletRequestImpl) message;

            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Network IN request " + req.getMethod() + " --> \r\n" +
                    req.toString());
            }

            if (hasBodyWithoutContentType(message)) {
                return;
            }
            
            StreamResponseDispatcher rd = new StreamResponseDispatcher(handler,
                    key, _networkHandler);
            req.pushTransactionDispatcher(rd); //Could be the same channel            
            req.pushApplicationDispatcher(_networkHandler);          
            //_networkHandler.next(req); //Delegate the
            LayerHelper.next(req, _networkHandler, _networkHandler);
            
            if (SipMonitoring.isEnabled(SipMonitoring.NETWORK_MANAGER)) {
                ((NetworkManager)_networkHandler).incrEasReceivedSipRequests();
            }
  
  
        } else {
            SipServletResponseImpl resp = (SipServletResponseImpl) message;
            Header cseq = resp.getRawHeader(Header.CSEQ);
             /*
	          * this should never be null
	          */
	        if ((cseq == null) || (cseq.getValue() == null)) {
	              logger.log(Level.SEVERE, "CSeq null in response, dropping it");
	              if (SipMonitoring.isEnabled(SipMonitoring.NETWORK_MANAGER)) {
	                  ((NetworkManager)_networkHandler).incrEasInvalidSipMessages();
	               }
	               return;     
	        }

            String c = cseq.getValue();
            int index = c.indexOf(' ');
            resp.setMethod(c.substring(index + 1));

            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE, "Network IN response " + resp.getStatus() + " " +
                    resp.getMethod() + " --> \r\n" + resp.toString());
            }

            //_networkHandler.next(resp);
            LayerHelper.next(resp, _networkHandler, _networkHandler);
            
            
           if (SipMonitoring.isEnabled(SipMonitoring.NETWORK_MANAGER)) {
                ((NetworkManager)_networkHandler).incrEasReceivedSipResponses();
            }
            
        }
    }

    public boolean hasBodyWithoutContentType(SipServletMessageImpl message) {
        // The SIP-stack should refuse all request with body and without
        // Content-Type header field.
        if (message.hasBody() && (message.getContentType() == null)) {
            try {
                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE,
                        "Missing Content-Type header field, the request has a body.");
                }

                SipServletRequestImpl req = (SipServletRequestImpl) message;
                String phraze = "Missing Content-Type header field";
                SipServletResponseImpl resp = req.createTerminatingResponse(400,
                        phraze);

                while (resp.popDispatcher() != null) {
                } // After this layer it shouldn't go anywhere

                TargetResolver tr = TargetResolver.getInstance();
                TargetTuple tt = tr.resolveResponse(resp);

                if (tt != null) {
                    resp.setRemote(tt);
                    _networkHandler.dispatch(resp);
                } else if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE,
                        "failed to find out where to send error response");
                }
            } catch (Exception ignore) {
                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE, "unexpected exception: " + ignore);
                }
            }

            return true;
        }

        return false;
    }

    // TODO
    public Object call() throws Exception {
        return null;
    }
    
    private void skipNewLines(ByteBuffer buffer) {
        int position = buffer.position();
        while (buffer.remaining() == 2) {
            if (buffer.get(position) == '\r' && buffer.get(position + 1) == '\n') {
                position += 2;
                buffer.position(position);
            } else {
                break;
            }
        }
    }
}
