/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package com.ericsson.ssa.container;

import com.ericsson.ssa.config.ConfigFactory;
import com.ericsson.ssa.config.annotations.Configuration;
import com.ericsson.ssa.config.annotations.UpdatePolicy;
import com.ericsson.ssa.config.annotations.UsagePolicy;
import com.ericsson.ssa.sip.Header;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import com.ericsson.ssa.sip.Layer;
import com.ericsson.ssa.sip.SipServletRequestImpl;
import com.ericsson.ssa.sip.SipServletResponseImpl;
import java.util.concurrent.atomic.AtomicLong;
import com.ericsson.ssa.container.callflow.Reporter;
import com.ericsson.ssa.container.callflow.CallflowResolver;
import java.util.logging.Level;
import org.jvnet.glassfish.comms.util.LogUtil;
import java.util.logging.Logger;


public abstract class NetworkManager implements Layer {
    private static NetworkManager instance = null;

    private static final String GRIZZLY_ENABLE_PROPERTY = "sip.network.grizzly";
    
    private AtomicLong _EasInvalidSipMessages = new AtomicLong();
    private AtomicLong _EasReceivedSipRequests = new AtomicLong();
    private AtomicLong _EasReceivedSipResponses = new AtomicLong();
    private AtomicLong _EasSentSipRequests = new AtomicLong();
    private AtomicLong _EasSentSipResponses = new AtomicLong();
    
    private Reporter _reporter;
    
    public void setReporters(String reporters){
        _reporter = CallflowResolver.getInstance().getReporter(reporters);
    }
    
    public Reporter getReporter(){
        return _reporter;
    }
    
    private Logger log = LogUtil.SIP_LOGGER.getLogger();
    private int _SipLinkTimeout = 10; // ms
    private int _SipLinkTimeoutRetries = 25; // iterations
    private int _SipLinkMaxQueueLen = 50;
    private int _SipLinkWaitLockTimeout = 5000;
    private long _SipLinkAliveTimeout = 120;
    private boolean _errorResponseEnabled = false;   

    private Set<String> additionalHeaders = Collections.emptySet();
    public abstract void next(SipServletRequestImpl req);

    public abstract void next(SipServletResponseImpl resp);

    public abstract void registerNext(Layer layer);

    public abstract void dispatch(SipServletRequestImpl req);

    public abstract void dispatch(SipServletResponseImpl resp);

    public synchronized static NetworkManager getInstance() {
        if (instance == null) {
            //TODO remove in final version so only one Network manager is present
            String grizzlyEnabledPropertyValue = System.getProperty(GRIZZLY_ENABLE_PROPERTY);
            if (grizzlyEnabledPropertyValue != null &&
                    !Boolean.parseBoolean(grizzlyEnabledPropertyValue)) {
                instance = new OLDNetworkManager();                
            } else {
                instance = new GrizzlyNetworkManager();
            }
        }

        return instance;
    }

    public boolean isRunning() {
        return true;
    }
    
        // package access
    void incrEasInvalidSipMessages() {
        _EasInvalidSipMessages.incrementAndGet();
    }

    void incrEasReceivedSipRequests() {
        _EasReceivedSipRequests.incrementAndGet();
    }

    void incrEasReceivedSipResponses() {
        _EasReceivedSipResponses.incrementAndGet();
    }

    void incrEasSentSipRequests() {
        _EasSentSipRequests.incrementAndGet();
    }

    void incrEasSentSipResponses() {
        _EasSentSipResponses.incrementAndGet();
    }

    public long getEasInvalidSipMessages() {
        return _EasInvalidSipMessages.longValue();
    }

    public long getEasReceivedSipRequests() {
        return _EasReceivedSipRequests.longValue();
    }

    public long getEasReceivedSipResponses() {
        return _EasReceivedSipResponses.longValue();
    }

    public long getEasSentSipRequests() {
        return _EasSentSipRequests.longValue();
    }

    public long getEasSentSipResponses() {
        return _EasSentSipResponses.longValue();
    }
    
    
    /**
     * Defines timeout value in milliseconds for a single write operation of a
     * sip link.
     * </p>
     * Range: 1-50ms
     *
     * @param timeout
     */
    @Configuration (key="WriteTimeoutInMillis", node="/SipService/SipProtocol/SipLink")
    public void setSipLinkTimeout(int timeout) {
        if ((timeout > 0) && (timeout <= 50)) {
            _SipLinkTimeout = timeout;
        } else {
            if (log.isLoggable(Level.WARNING)) {
                log.log(Level.WARNING, "sip.stack.network.bad_sip_link_timeout", new Object[] { timeout });
            }   Integer i;             
        }
    }

    /**
     * Defines timeout value in milliseconds for a single write operation of a
     * sip link.
     *
     * @return timeout for write operation of a sip link
     */
    public int getSipLinkTimeout() {
        return _SipLinkTimeout;
    }

    /**
     * Defines timeout value in seconds for keeping an inactive link. If no
     * traffic has been generated on that link for that time period the link is
     * closed.
     *
     * @return timeout value in seconds for keeping an inactive link
     */
    public long getSipLinkAliveTimeout() {
        return _SipLinkAliveTimeout;
    }

       // this code should be refactored; the code should be moved to the Header class
    // and the header class should get a configuration hook (e.g., register itself for configuration).
    // Also the way it is used here the ExtraMultiLineHeaderNames is stuck to being a property and will not 
    // be resolved if ever changed to a attribute. This might be OK.
    @Configuration (usage=UsagePolicy.IGNORE, key="ExtraMultiLineHeaderNames", node="/SipService/SipProtocol")
    public void addHeaders(String configuredHeaders) {
        HeaderSet additionalHeaderSet = new HeaderSet(configuredHeaders);
        Set<String> tempAdditionalHeaders = additionalHeaderSet.getSet();
        
        Set<String> newHeaders = new HashSet<String>(tempAdditionalHeaders);
        newHeaders.removeAll(this.additionalHeaders);
        
        Set<String> removedHeaders = new HashSet<String>(this.additionalHeaders);
        removedHeaders.removeAll(tempAdditionalHeaders);
        
        for (String header : removedHeaders) {
            Header.MULTI_LINE_HEADER_SET.remove(header);            
        }

        for (String header : newHeaders) {
            Header.MULTI_LINE_HEADER_SET.add(header);
        }
        
        this.additionalHeaders = tempAdditionalHeaders;
        
        if (log.isLoggable(Level.FINE)) {
            StringBuffer additionalHeadersString = new StringBuffer();
            additionalHeadersString.append("extra Multiline Headers: <");
            boolean first = true;
            for (String header : tempAdditionalHeaders) {
                if (first) {                    
                    first = false;
                } else {
                    additionalHeadersString.append(",");
                }
                additionalHeadersString.append(header);                
            }
            additionalHeadersString.append(">");
            log.fine(additionalHeadersString.toString());
        }
    }
    
    /**
     * Inner class to enable automatic conversion of
     * a comma seperated string of headernames to a set
     * of properly formatted names.
     *
     */
    public class HeaderSet {
        HashSet<String> headerSet = new HashSet<String>();
        public HeaderSet(String aCommaSeparatedHeaderSet) {
            String[] headers = aCommaSeparatedHeaderSet.split(",");
            for (int i = 0; i < headers.length; i++) {
                String header = headers[i].trim();
                if (!"".equals(header)) {
                    headerSet.add(Header.format(header));
                }
            }
        }
        
        public Set<String> getSet() {
            return headerSet;
        }
        
        
    }

    /**
     * Defines timeout value in seconds for keeping an inactive link. If no
     * traffic has been generated on that link for that time period the link is
     * closed.
     *
     * @param timeout
     *           value in seconds for keeping an inactive link
     */
    @Configuration (key="ConnectionAliveTimeoutInSeconds", node="/SipService/SipProtocol/SipLink")
    public void setSipLinkAliveTimeout(long timeout) {
        _SipLinkAliveTimeout = timeout;
    }

    /**
     * Defines the number of retries to perform of a single write operation of a
     * sip link.
     * </p>
     * Range: 1-25
     * </p>
     * The maximum time a link will try for success of a write operation is
     * defined by SipLinkTimeout * SipLinkTimeoutRetries milliseconds.
     *
     * @param retries
     */
    @Configuration (key="WriteTimeoutRetries", node="/SipService/SipProtocol/SipLink")
    public void setSipLinkTimeoutRetries(int retries) {
        if ((retries > 0) && (retries <= 25)) {
            _SipLinkTimeoutRetries = retries;
        } else {
            if (log.isLoggable(Level.WARNING)) {
                log.log(Level.WARNING, "sip.stack.network.bad_sip_link_timeout_retries", new Object[] { retries });
            }
        }

        _SipLinkTimeoutRetries = retries;
    }

    /**
     * Defines the number of retries to perform of a single write operation of a
     * sip link.
     * </p>
     * The maximum time a link will try for success of a write operation is
     * defined by SipLinkTimeout * SipLinkTimeoutRetries milliseconds.
     *
     * @return the number of retries to perform of a single write operation of a
     *         sip link.
     */
    public int getSipLinkTimeoutRetries() {
        return _SipLinkTimeoutRetries;
    }

    /**
     * Returns the maximum number of connect/write tasks that can be queued in a
     * link.
     *
     * @return the maximum number of connect/write tasks that can be queued in a
     *         link
     */
    public int getSipLinkMaxQueueLength() {
        return _SipLinkMaxQueueLen;
    }

    /**
     * Returns the maximum number of connect/write tasks that can be queued in a
     * link.
     *
     * @param length
     *           the maximum number of connect/write tasks that can be queued in
     *           a link
     */
    @Configuration (key="MaxQueueLength", node="/SipService/SipProtocol/SipLink")
    public void setSipLinkMaxQueueLength(int length) {
        _SipLinkMaxQueueLen = length;
    }

    /**
     * Returns the maximum time a thread will wait to get an exclusive lock for a
     * sip link.
     *
     * @return the maximum time a thread will wait to get an exclusive lock for a
     *         sip link.
     */
    public int getSipLinkWaitLockTimeout() {
        return _SipLinkWaitLockTimeout;
    }

    /**
     * Sets the maximum time a thread will wait to get an exclusive lock for a
     * sip link.
     *
     * @param timeout
     */
    @Configuration (key="SipLinkWaitLockTimeout", node="/SipService/SipProtocol/SipLink", 
        usage=UsagePolicy.IGNORE, update=UpdatePolicy.STARTUP) //This was never before configurable!
    public void setSipLinkWaitLockTimeout(int timeout) {
        _SipLinkWaitLockTimeout = timeout;
    }
    
    /**
     * Returns whether a malformed request should be handled by sending a proper
     * informational error response.
     *
     * @return
     */
    public Boolean isErrorResponseEnabled() {
        return _errorResponseEnabled;
    }

    /**
     * Setter of the <code>ErrorResponseEnabled</code> property.
     *
     * @param responseEnabled
     */
    @Configuration (key="ErrorResponseEnabled", node="/SipService/SipProtocol")
    public void setErrorResponseEnabled(Boolean responseEnabled) {
        _errorResponseEnabled = responseEnabled;
    }    
}
