/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package com.ericsson.ssa.container;

import com.ericsson.ssa.config.Config;
import com.ericsson.ssa.config.ConfigFactory;
import com.ericsson.ssa.sip.dns.SipTransports;
import com.ericsson.ssa.sip.dns.TargetTuple;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * A SipBindingCtx is a snapshot of sip-listener from config.
 * Or the SipBindingCtx is artificially devised to reflect public bindings or
 * old style configuration.
 * @author ELNELBO
 */
public class SipBindingCtx {
    private static final String SSL ="/Ssl";
    private SipBindingResolver resolver;
    private String context; 
    private Config config = ConfigFactory.getConfig();
    boolean sipListener;
    
    SipBindingCtx(SipBindingResolver aResolver, String aContext, 
            TargetTuple[] aTargetTuples) {
        this(aResolver, aContext, aTargetTuples, true);
    }   
    
    SipBindingCtx(SipBindingResolver aResolver, String aContext, 
            TargetTuple[] aTargetTuples, boolean isSipListener) {
        resolver = aResolver;
        context = aContext;
        sipListener = isSipListener;
        snapshot(aTargetTuples);
    }   
    
    public String getContext() {
        return context;
    }
    
    public String getId() {
        return id;
    }
    
    public TargetTuple[] getTargetTuples() {
        return targetTuples;
    }
    
    public boolean isEnabled() {
        return enabled;        
    }
    
    public String getSSLAttribute(String anAttribute) {
        return sslAttributes.get(anAttribute);
    }
    
    /**
     * Checks whether the SipBindingCtx is still up to date.
     * @return true if SipBindingCtx is still false if still up to date.
     */
    public boolean isStale() {
        return resolver.isStale(this);
    }
    
    public TargetTuple getTargetTupleForProtocol(SipTransports protocol){
        TargetTuple ttOfProto = null;
        
        if (targetTuples!=null) {
            for (TargetTuple tt : targetTuples)  {
                if (tt.getProtocol().ordinal()==protocol.ordinal()) {
                    ttOfProto = tt;
                    break;
                }
            }
        }
        return ttOfProto;
    }
    
    //Update support
    public enum UpdateHint {
        UP_TO_DATE(0x00),
        ENABLE_DISABLED(0x01),      //00001
        SSL_UPDATE(0x02),           //00010 
        TARGET_TUPLE_UPDATE(0x04),  //00100
        MULTIPLE(0x08),             //01000
        NOT_SAME(0x10);             //10000
        
        private int hint;
        private int mask;

        UpdateHint(int aHint) {
            hint = aHint;
            mask=hint;
        }

        public int mask() {
            return this==MULTIPLE ? mask : hint;
        }
        
        public boolean contains(UpdateHint aHint) {
            return this==MULTIPLE ? (mask&aHint.hint)>0 : false;
        }
        
        int mask(UpdateHint aHint) {
            mask ^= aHint.hint;
            return this==MULTIPLE ? mask : hint;
        }
    };
    
    public UpdateHint getUpdateHints(SipBindingCtx update) {
        List<UpdateHint> hints = new ArrayList<UpdateHint>(1);
        if (isSame(update)) {
            if (enabled!=update.enabled) {
                hints.add(SipBindingCtx.UpdateHint.ENABLE_DISABLED);
            }
            
            if (targetTuples.length ==
                    update.targetTuples.length) {
                for (TargetTuple tt : targetTuples) {
                    TargetTuple updateTT = 
                            update.getTargetTupleForProtocol(tt.getProtocol());
                    if (updateTT==null || !tt.getSocketAddress().equals(
                            updateTT.getSocketAddress())) {
                        hints.add(SipBindingCtx.UpdateHint.TARGET_TUPLE_UPDATE);
                        break;
                    }
                } 
            } else {
                hints.add(SipBindingCtx.UpdateHint.TARGET_TUPLE_UPDATE);
            }
            
            if (!sslAttributes.equals(update.sslAttributes)) {
                hints.add(SipBindingCtx.UpdateHint.SSL_UPDATE);                
            }
            
        } else {
            hints.add(UpdateHint.NOT_SAME);
        }

        UpdateHint hint = null;

        if (hints.isEmpty()) {
          hint = UpdateHint.UP_TO_DATE;  
        } else if (hints.size()==1) {
            hint = hints.get(0);
        } else {
            hint = UpdateHint.MULTIPLE;
        }
        return hint;
    }
    
    public boolean isSame(SipBindingCtx other) {
        return other!=null && context.equals(other.context) 
                && id.equals(other.id);
    }
    
    //Snapshot
    private String id;
    private TargetTuple[] targetTuples;
    private boolean enabled;
    private Map<String, String> sslAttributes = new HashMap<String, String>();
    
    private void snapshot(TargetTuple[] aTargetTuples) {
        targetTuples = aTargetTuples;
        id = sipListener ? config.get(context, "Id") : context;
        enabled = sipListener ? Boolean.parseBoolean(config.get(context, "Enabled")) : true;
        
        if(sipListener) {
            String attr = null;
            
            attr = config.get(context+SSL, "CertNickname");
            if (attr!=null) {
                sslAttributes.put("CertNickname", attr);
            }
            attr = config.get(context+SSL, "Ssl2Enabled");
            if (attr!=null) {
                sslAttributes.put("Ssl2Enabled", attr);
            }
            attr = config.get(context+SSL, "Ssl2Ciphers");
            if (attr!=null) {
                sslAttributes.put("Ssl2Ciphers", attr);
            }
            attr = config.get(context+SSL, "Ssl3Enabled");
            if (attr!=null) {
                sslAttributes.put("Ssl3Enabled", attr);
            }
            attr = config.get(context+SSL, "Ssl3TlsCiphers");
            if (attr!=null) {
                sslAttributes.put("Ssl3TlsCiphers", attr);
            }
            attr = config.get(context+SSL, "TlsEnabled");
            if (attr!=null) {
                sslAttributes.put("TlsEnabled", attr);
            }
            attr = config.get(context+SSL, "TlsRollbackEnabled");
            if (attr!=null) {
                sslAttributes.put("TlsRollbackEnabled", attr);
            }
            attr = config.get(context+SSL, "ClientAuthEnabled");
            if (attr!=null) {
                sslAttributes.put("ClientAuthEnabled", attr);
            }
        }
    }
}
