/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package com.ericsson.ssa.sip;

import com.ericsson.ssa.sip.PathNode.Type;

import java.io.Serializable;

import java.util.logging.Level;

import javax.servlet.Servlet;
import javax.servlet.sip.Proxy;


/**
 * Implementation of a ProxyInterface for a confirmed dialog. This class has only
 * a subset of the state that ProxyImpl and most important: it is Serializable.
 *
 * It merely implements the dispatch()-methods which does not need to be aware of
 * earlier responses.
 *
 * @author qmaghes
 *
 */
public class ConfirmedProxyImpl extends AbstractProxyImpl
    implements Serializable {
    private static final long serialVersionUID = -7258322652405154081L;

    private transient ReinviteProxyTracers reinviteTracers = new ReinviteProxyTracers(); 
    
    public ConfirmedProxyImpl() {
        super();
    }

    /**
     * For deserialization only
     *
     */
    public ConfirmedProxyImpl(String applicationSessionName) {
        super(applicationSessionName);
    }

    public void dispatch(SipServletRequestImpl req, ProxyContext pc) {
        if (_log.isLoggable(Level.FINE)) {
            _log.log(Level.FINE, req.toDebugString());
        }

        // update max forwards
        int maxForwards = req.getMaxForwards();

        if (maxForwards == 0) {
            if (_log.isLoggable(Level.INFO)) {
                _log.log(Level.INFO, "Too Many Hops for request = " + req);
            }

            SipServletResponseImpl response = req.createTerminatingResponse(483);

            if (response != null) {
                response.popDispatcher().dispatch(response);
            }

            return;
        } else if (maxForwards == -1) {
            maxForwards = 70;
        } else {
            maxForwards--;
        }

        req.setMaxForwards(maxForwards);
        

        
        if (req.getMethod().equals("CANCEL")) {
            // FIXME investigate handing of CANCEL on a confirmed dialog.
            // For now we keep the code moved from ProxyImpl.

            // lets answer cancel with 200 OK...
            SipServletResponseImpl resp = req.createTerminatingResponse(200);
            resp.setRemote(req.getRemote());
            resp.popDispatcher().dispatch(resp);

            try {
                // notify the servlet about the CANCEL...
                setDerivedOrOriginalSession(req, pc);

                Servlet s = getServlet(pc.getSipSession().getHandler());

                if (s != null) {
                    req.setProxyContext(pc);
                    req.getSessionImpl().updateSipSessionState(req, Type.Proxy);
                    s.service(req, null);
                    // For CANCEL in dialog we have no branches to cancel
                    // Send it on in the dialog if we can deduce the correct branchId
                    reinviteTracers.handleRequest(req);
                } else {
                    if (_log.isLoggable(Level.INFO)) {
                        _log.log(Level.INFO,
                            "Could not find servlet name: " +
                            req.getSessionImpl().getHandler() +
                            " in application: " +
                            req.getSessionImpl().getApplicationSessionImpl()
                               .getName());
                    }
                }
            } catch (Exception e) {
                // problem in servlet, FIXME what to do?
               if (_log.isLoggable(Level.FINE)) {
                  _log.log(Level.FINE, "Problem in servlet ", e);
              }
            }
        } else { // not CANCEL

            if (getRecordRoute()) {
                try {
                    // support for dialog creational NOTIFY
                    setDerivedOrOriginalSession(req, pc);

                    // if record route is enabled invoke the servlet
                    Servlet s = getServlet(pc.getSipSession().getHandler());

                    if (s != null) {
                        req.setProxyContext(pc);
                        req.getSessionImpl().updateSipSessionState(req, Type.Proxy);
                        s.service(req, null);

                        if (req.getSupervised()) {
                            // only if the response will visit this proxy
                            // the request need to be cloned
                            SipServletRequestImpl clone = (SipServletRequestImpl) req.clone();
                            clone.pushTransactionDispatcher(pc);
                            clone.setTransactionRequest(req);
                            reinviteTracers.handleRequest(clone);
                            clone.popDispatcher().dispatch(clone);
                        } else {
                           reinviteTracers.handleRequest(req);
                           req.popDispatcher().dispatch(req);
                        }
                    } else {
                        if (_log.isLoggable(Level.INFO)) {
                            _log.log(Level.INFO,
                                "Could not find servlet name: " +
                                req.getSessionImpl().getHandler() +
                                " in application: " +
                                req.getSessionImpl().getApplicationSessionImpl()
                                   .getName());
                        }
                    }
                } catch (Exception e) {
                    if (_log.isLoggable(Level.FINE)) {
                        _log.log(Level.FINE, "Problem in servlet ", e);
                    }

                    // problem in servlet, // FIXME
                }
            } else {
               reinviteTracers.handleRequest(req);
                req.popDispatcher().dispatch(req);
            }
        }
    }

    public void dispatch(SipServletResponseImpl resp, ProxyContext pc) {
        if (_log.isLoggable(Level.FINE)) {
            _log.log(Level.FINE, resp.toDebugString());
        }

        reinviteTracers.handleResponse(resp);
        invokeServletAndForward(resp, pc);
    }

    /**
     * Getting a Proxy is not applicable for a confirmed dialog.
     *
     */
    public Proxy getFacade(SipServletRequestImpl request) {
        throw new IllegalStateException(
            "getFacade not applicable for Confirmed Proxy");
    }

    /**
     * Only applicable before dialog is confirmed
     *
     */
    public void setVirtualProxyBranchRequest(SipServletResponseImpl resp) {
        throw new IllegalStateException(
            "setVirtualProxyBranchRequest not applicable for Confirmed Proxy");
    }

    /**
     * Only applicable before dialog is confirmed
     *
     */
    public void startVirtualProxyBranch(SipServletResponseImpl resp) {
        throw new IllegalStateException(
            "startVirtualProxyBranch not applicable for Confirmed Proxy");
    }
}
