/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package com.ericsson.ssa.sip;

import java.util.Iterator;
import java.util.Map;
import java.util.Queue;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.servlet.sip.Address;

import org.jvnet.glassfish.comms.util.LogUtil;

import com.ericsson.ssa.sip.transaction.TransactionManager;
import com.sun.enterprise.server.ApplicationServer;


/**
 * TODO Add comments (class description)
 *
 * @author ehsroha
 * @since 2005-maj-26
 */
public class DialogSet{
    private static final long serialVersionUID = -6352106202257349396L;
    private static final Logger logger = LogUtil.SIP_LOGGER.getLogger();
    private static final int INITIAL_CAPACITY = 1000;
    private static final long DS_LIFETIME;
    static {
        {
            long dlt = 64 * TransactionManager.getInstance().getTimerT1() + 10000;
            String dltStr = System.getProperty("sipContainer.ds.lifeTime", "" + dlt);
            try {
                dlt = Long.parseLong(dltStr);
            } catch (Throwable t) {
                // Ignore. Default will be used.
            }
            DS_LIFETIME = dlt;
        }
    }
    
    private static final ConcurrentHashMap<String, DialogSet> m_EarlyDialogs = new ConcurrentHashMap<String, DialogSet>(INITIAL_CAPACITY);
    private final Ascii7String m_CallId;
    private final Address m_From;
    private final Queue<DialogFragment> m_Dialogs = new ConcurrentLinkedQueue<DialogFragment>();
    private final AtomicInteger m_DialogSetFragmentId = new AtomicInteger(0);
    private final DialogFragment m_InitialDialogFragment;
    private boolean isDialogCreational;
    private long expirationTime;

    /**
     * creates a DialogSet including the initial DialogFragment
     *
     */
    public DialogSet(String callId, Address from, boolean isDialogCreational) {
        this.isDialogCreational = isDialogCreational;
        expirationTime = System.currentTimeMillis() + DS_LIFETIME;
        m_CallId = new Ascii7String(callId);
        m_From = from;
        
        m_InitialDialogFragment = DialogFragmentManager.getInstance().createDialogFragment(this);
        addDialog(m_InitialDialogFragment);
    }

    /**
     * Indicates that this dialog fragment was created from a dialog creational request.
     * @return
     */
    public boolean isDialogCreational() {
        return isDialogCreational;
    }


    public DialogFragment getInitialDialogFragment(){
    	return m_InitialDialogFragment;
    }
    
    public DialogFragment createAdditionalDialogFragment() {
    	DialogFragment df = DialogFragmentManager.getInstance().createDialogFragment(this);
        addDialog(df);
        return df;
    }

    public static String createKey(String callId, String fromTag) {
        return DialogFragment.createKey(callId, fromTag, null, null);
    }

    public String getCallId() {
        return m_CallId.toString();
    }
    
    public Ascii7String getCallIdAscii7String() {
        return m_CallId;
    }

    public Address getFrom() {
        return m_From;
    }

    /**
     * @return the from tag of this Dialog.
     */
    public String getFromTag() {
        return getFrom().getParameter(AddressImpl.TAG_PARAM);
    }

    private void addDialog(DialogFragment d) {
        m_Dialogs.add(d);
    }

    public void removeDialog(DialogFragment d) {
        m_Dialogs.remove(d);
        remove();
    }

    public DialogFragment getDialog(String fragmentId) {
    	if (fragmentId.equals(DialogFragment.DEFAULT_FRAGMENT_ID)) {
    		return  m_InitialDialogFragment;
    	}
        for (DialogFragment d : m_Dialogs) {
            if (fragmentId.equals(d.getFragmentId())) {
                // match, lets return it...
                return d;
            }
        }

        // no match...
        return null;
    }

    public Iterator<DialogFragment> getDialogs() {
        return m_Dialogs.iterator();
    }

    public DialogFragment searchForDialog(String toTag, String fragmentId)  {
    	if (fragmentId.equals(DialogFragment.DEFAULT_FRAGMENT_ID)) {
    		throw new IllegalStateException("This method should not be used to search via the DEFAULT_FRAGMENT_ID");
    	}
    	// TODO Maybe we should iterate over m_Dialogs
    	// instead of via DialogFragmentManager
        String key = DialogFragment.createKey(m_CallId.toString(),
                getFromTag(), toTag, fragmentId);

        DialogFragment result = null;
        try {
            result = DialogFragmentManager.getInstance().findDialogFragment(key);
        } catch (RemoteLockException e) {
            throw new RemoteLockRuntimeException(e);
        }
        return result;
    }

    /**
     * Needed for creational NOTIFY
     *
     * @param fragmentId
     * @return a cloned DialogFragment matching the fragmentId or null otherwise
     */
    public DialogFragment cloneDialog(String fragmentId) {
    	if (fragmentId.equals(DialogFragment.DEFAULT_FRAGMENT_ID)) {
    		return (DialogFragment) m_InitialDialogFragment.clone();
    	}
        for (DialogFragment d : m_Dialogs) {
            if (d.getFragmentId().equals(fragmentId)) {
                // match, lets return a clone...
                return (DialogFragment) d.clone();
            }
        }

        // no match...
        return null;
    }
    
    public String nextFragmentId() {
    	String instanceId = ApplicationServer.getServerContext().getInstanceName();
    	return instanceId+"_"+m_DialogSetFragmentId.incrementAndGet();	
    }

    /**
     * Removes this dialog set.
     */
    public void remove() {
        for (DialogFragment df : m_Dialogs) {
            df.clearDialogSet(false);
        }
    
        if (!m_EarlyDialogs.isEmpty()) {
            String id = createKey(getCallId(), getFromTag());

            if (m_EarlyDialogs.remove(id) != null && logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "Removed early dialog with key = " + id + ", DialogSet = " + this);
            }
        }
    }    
    

    /**
     * Gets the early dialog identified by the id (CallId + From-tag)
     * @param id
     * @return
     */
    public static DialogSet getEarlyDialog(String id) {
        return m_EarlyDialogs.get(id);
    }
    
    /**
     * Register the early dialog.
     *
     * @param req the request
     */
    public static void registerEarlyDialog(SipServletRequestImpl req) {
        if (SipFactoryImpl.isDialogCreational(req.getMethod()) && req.isInitial()) {
            DialogFragment d = req.getDialog();

            if (d != null) {
                DialogSet ds = d.getDialogSet();
                String id = DialogSet.createKey(ds.getCallId(), ds.getFromTag());
                m_EarlyDialogs.putIfAbsent(id, ds);
            }
        }
    }

    public static boolean earlyDialogsExists() {
        return !m_EarlyDialogs.isEmpty();
    }
    
    public static void doScavenge() {
    	Iterator<Entry<String, DialogSet>> scavengeIterator = m_EarlyDialogs.entrySet().iterator();
    	
        int numRemoved = 0;
        if (logger.isLoggable(Level.FINE)) {
           logger.log(Level.FINE, "EarlyDialogs contains #entries:"+m_EarlyDialogs.size());
        }
        long now = System.currentTimeMillis();
        while (scavengeIterator.hasNext()) {
            Map.Entry<java.lang.String, com.ericsson.ssa.sip.DialogSet> entry = scavengeIterator.next();
            DialogSet ds = entry.getValue();
            if (ds != null && now > ds.expirationTime) {
                numRemoved++;
            	ds.remove();
                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE, "The dialog set: "+ds+", is due for removal ");
                }
            }
        }
        if (logger.isLoggable(Level.FINE)) {
           logger.log(Level.FINE, "Number of EarlyDialogs removed:"+numRemoved);
        }
    }

    /**
     * Prolong the lifetime of this dialog set by the specified amount.
     * @param duration
     */
    public void prolongLifetime(long duration) {
        expirationTime = System.currentTimeMillis() + duration;
    }
}
