/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package com.ericsson.ssa.sip;

import java.util.logging.Level;
import java.util.logging.Logger;

import org.jvnet.glassfish.comms.util.LogUtil;

import java.io.IOException;

/**
 * @author jluehe
 */
public abstract class PersistentDialogFragmentManager extends DialogFragmentManager {
    private static final Logger log = LogUtil.SIP_LOGGER.getLogger();
    private static final long REAPER_THREAD_SLEEP_TIME = 60 * 1000L; // 1 minute

    protected boolean isRepairDuringFailure = true;

    // The persistent store for dialog fragments
    private DialogFragmentStore store = null;

    /**
     * The background reaper thread.
     */
    private Thread reaperThread = null;
    private boolean reaperThreadDone = false;

    protected PersistentDialogFragmentManager() {
        super();
        reaperThread = new Thread(new ReaperThread(), "DialogFragmentReaper");
        reaperThread.setDaemon(true);
    }

    /**
     * Gets the DialogFragment with the given id.
     *
     * This method first checks the active cache for a DialogFragment with the
     * given id. If not found, it queries the persistent store.
     *
     * @return The DialogFragment with the given id, or null if not found
     *
     * @throws RemoteLockException
     */
    public DialogFragment findDialogFragment(String id)
            throws RemoteLockException {
        
        DialogFragment df = super.findDialogFragment(id);
        if (df != null || store == null) {
            return df;
        }

        try {
            df = swapInDialogFragment(id);
        } catch (IOException ioe) {
            log.log(Level.SEVERE, "sipstack.swap_in_dialog_fragment_error", id);
            log.log(Level.SEVERE, ioe.getMessage(), ioe);
        }

        return df;
    }

    /**
     * Removes the given DialogFragment.
     *
     * @param d The DialogFragment to be removed
     * @param hasTimedOut true if the DialogFragment is removed because it has
     * expired, false otherwise
     */
    public void removeDialogFragment(DialogFragment df, boolean hasTimedOut) {

        super.removeDialogFragment(df, hasTimedOut);

        if (store != null && df != null && df.isReplicable()) {
            try {
                store.remove(df.getFragmentId());
            } catch (IOException ioe) {
                log.log(Level.SEVERE, "sipstack.remove_dialog_fragment_from_store_error", df.getFragmentId());
                log.log(Level.SEVERE, ioe.getMessage(), ioe);
            }
        }
    }    

    /**
     * Persists the given DialogFragment.
     *
     * @param df The DialogFragment to be persisted
     */
    public void saveDialogFragment(DialogFragment df)
            throws IOException {
        if (store != null && df != null && df.isReplicable()) {
            store.save(df);
        }
    }

    /**
     * Swaps in the DialogFragment with the given id from the persistent
     * store of this dialog manager.
     *
     * @return The DialogFragment with the given id, or null if not found
     *
     * @throws IOException
     * @throws RemoteLockException
     */
    protected DialogFragment swapInDialogFragment(String id)
            throws IOException, RemoteLockException {

        if (store == null || id == null) {
            return null;
        }
        
        DialogFragment df = store.load(id);
        if (df == null) {
            return null;
        }

        registerDialogFragment(df);

        return df;
    }

    /**
     * Gets the DialogFragment with the given id 
     * from the active cache only
     *
     * @return The DialogFragment with the given id, or null if not
     * found
     */
    public DialogFragment findDialogFragmentFromCacheOnly(String id) {
        DialogFragment result = null;
        try {
            result = super.findDialogFragment(id);
        } catch (RemoteLockException ex) {
            // Could never happen, since we are fetching from the local active cache!!
            if (log.isLoggable(Level.WARNING)) {
                log.log(Level.WARNING, "sip.stack.remote_lock_in_local_cache", new Object[]{"DialogFragment", id});
            }
        }
        return result;
    }

    /**
     * Removes the given DialogFragment from only the active cache
     *
     * @param df The DialogFragment to be removed
     * @param hasTimedOut true if the DialogFragment is removed because it has
     * expired, false otherwise
     */    
    protected void removeDialogFragmentFromCache(String id,
                                                 boolean hasTimedOut) {
        DialogFragment removeIt = findDialogFragmentFromCacheOnly(id);
        if (removeIt != null) {
            super.removeDialogFragment(removeIt, hasTimedOut);
        }
    }     


    /**
     * Removes the given DialogFragment from only the active cache
     *
     * @param df The DialogFragment to be removed
     * @param hasTimedOut true if the DialogFragment is removed because it has
     * expired, false otherwise
     */    
    protected void removeDialogFragmentFromCache(DialogFragment df,
                                                 boolean hasTimedOut) {
        super.removeDialogFragment(df, hasTimedOut);
    }     

    /**
     * Sets the repairDuringFailure property of this session manager.
     *
     * @param isRepairDuringFailure The value of the repairDuringFailure
     * property
     */
    public void setRepairDuringFailure(boolean isRepairDuringFailure) {
        this.isRepairDuringFailure = isRepairDuringFailure;
    }

    /**
     * @return True if this session manager should participate in a repair
     * during failure, and false otherwise
     */
    public boolean isRepairDuringFailure() {
        return isRepairDuringFailure;
    }

    public DialogFragmentStore getDialogFragmentStore() {
        return store;
    }
    
    public void setDialogFragmentStore(DialogFragmentStore store) {
        this.store = store;
    }
    
    @Override 
    protected void init() {
       super.init();
       // perform init for this class here
    }
    
    @Override
    protected void doStart() {
        super.doStart(); 
        startReaperThread();
        // perform start for this class here
    }
    
    @Override
    protected void doStop() {
        super.doStop();
        stopReaperThread();
        // perform stop for this class here
    }

    /**
     * Starts the background reaper thread.
     */
    private void startReaperThread() {
        reaperThreadDone = false;
        reaperThread.start();
    }

    /**
     * Stops the background reaper thread.
     */
    private void stopReaperThread() {
        reaperThreadDone = true;
        reaperThread.interrupt();
        try {
            reaperThread.join();
        } catch (InterruptedException e) {
            ;
        }
    }

    /**
     * Reaper thread responsible for purging expired dialog fragments
     */
    private class ReaperThread implements Runnable {

        public void run() {
            while (!reaperThreadDone) {
                try {
                    Thread.sleep(REAPER_THREAD_SLEEP_TIME);
                } catch (InterruptedException e) {
                    reaperThreadDone = true;
                }
                if (!reaperThreadDone) {
                    if (store != null) {
                        store.processExpires();
                    }
                }
            }
        }
    }

}
