/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 */
package com.ericsson.ssa.sip;

import java.util.HashMap;
import java.util.Map;


/**
 * Class that represent all the <code>ReinviteProxyTracer</code>s for the (confirmed) proxy.
 *
 * The UAC might send several reINVITE requests. They are distinguished by the CSeq.
 *
 *
 * @author Magnus Hessel, HiQ Stockholm AB
 * @since 2008 apr 30
 *
 */
public class ReinviteProxyTracers {
    /*
     *     mapping CSeq,direction => ReinviteProxyTracer
     *
     *     Note! This map can grow infinitely even if ACKs are not received. However the whole objects is removed when dialog
     *     terminates.
     *
     *     Warning! Using a HashMap to save very little memory (compared with Collections.synchronizedMap(HashMap))
     *     Make sure all public methods are synchronized.
     *
     */
    private Map<Key, ReinviteProxyTracer> proxyTracerMap = new HashMap<Key, ReinviteProxyTracer>(1);

    /**
     * Handle a dispatched request.
     *
     * @param request
     */
    public synchronized void handleRequest(SipServletRequestImpl request) {
        if ("INVITE".equals(request.getMethod())) {
            registerReinviteRequest(request);
        }
        else if ("ACK".equals(request.getMethod())) {
            deregisterTracer(request);
        }
        else if ("CANCEL".equals(request.getMethod())) {
            cancel(request);
        }
    }

    private void cancel(SipServletRequestImpl request) {
        ReinviteProxyTracer tracer = proxyTracerMap.get(new Key(request));

        if (tracer != null) {
            tracer.cancel(request);
        }
    }

    /**
      * Handle a dispatched response.
      */
    public synchronized void handleResponse(SipServletResponseImpl response) {
        if ((response.getStatus() / 100) == 1) {
            registerProvisionalResponse(response);
        }
    }

    /**
     *
     * Deregister tracer associated with the request.
     *
     */
    private void deregisterTracer(SipServletRequestImpl request) {
        proxyTracerMap.remove(new Key(request));
    }

    /**
     * Register a tracer and associate it with the request.
     *
     * @param reinviteRequest
     */
    private void registerReinviteRequest(SipServletRequestImpl reinviteRequest) {
        proxyTracerMap.put(new Key(reinviteRequest),
            new ReinviteProxyTracer(reinviteRequest));
    }

    /**
     * Register a provisional response with the associated ReinviteProxyTracer
     *
     * @param provisionalResponse
     */
    private void registerProvisionalResponse(
        SipServletResponseImpl provisionalResponse) {
        ReinviteProxyTracer tracer = proxyTracerMap.get(new Key(
                    provisionalResponse));

        if (tracer != null) {
            tracer.registerProvisionalResponse(provisionalResponse);
        }
    }


    /**
     * The Key class represents a key for Map lookup.
     * The key consists of two things; the sequence number and the direction.
     *
     * Since to-tag and from from-tag should differ, we can use either as
     * indicator for direction. However some implementations send (erraneously?)
     * "100 Trying" responses for reINVITE requests WITHOUT to-tag so we better use
     * the from-tag as direction marker.
     *
     * Note! There is a DialogFragment.isMessageFromCaller() that theoretically
     * could be used for direction, but fetching the DialogFragment and calling
     * that method needs error handling that we want to avoid. Better to use the
     * to-tag which must be present since we are in a dialog.
     *
     * @author qmaghes
     */
    public class Key {
        private String fromTag;
        private Integer cSeq;

        /**
         * Construction of Key using the parts the key consists of
         *
         * @param toTag
         * @param seq
         */
        public Key(String fromTag, Integer cSeq) {
            this.fromTag = fromTag;
            this.cSeq = cSeq;
        }

        /**
         * Construction of Key by using a SipServletMessageImpl and extracting the parts the
         * message consists of.
         *
         *
         * @param message
         * @throws NotEqualDialogException
         * @throws NumberFormatException
         */
        public Key(SipServletMessageImpl message) {
            this(message.getFrom().getParameter(AddressImpl.TAG_PARAM),
                message.getCSeqNumber());
        }

        /**
         * Generated by eclipse. do not edit.
         * Regenerate if needed.
         */
        @Override
        public int hashCode() {
            final int PRIME = 31;
            int result = 1;
            result = (PRIME * result) + ((cSeq == null) ? 0 : cSeq.hashCode());
            result = (PRIME * result) +
                ((fromTag == null) ? 0 : fromTag.hashCode());

            return result;
        }

        /**
         * Generated by eclipse. do not edit.
         * Regenerate if needed.
         */
        @Override
        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }

            if (obj == null) {
                return false;
            }

            if (getClass() != obj.getClass()) {
                return false;
            }

            final Key other = (Key) obj;

            if (cSeq == null) {
                if (other.cSeq != null) {
                    return false;
                }
            } else if (!cSeq.equals(other.cSeq)) {
                return false;
            }

            if (fromTag == null) {
                if (other.fromTag != null) {
                    return false;
                }
            } else if (!fromTag.equals(other.fromTag)) {
                return false;
            }

            return true;
        }
    }
}
