/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package com.ericsson.ssa.sip;

import java.io.UnsupportedEncodingException;

import java.util.logging.Level;


/**
 * This class will decode SIP URIs. All escaped character received will be
 * unescaped for the applications. Unscaping of the character is according to
 * RFC 3261.
 *
 * @author eraayaz
 * @since 2005-nov-01
 */
public class SipURIDecoder {
    protected String m_encoding = SipFactoryImpl.SIP_CHARSET;

    public SipURIDecoder() {
    }

    /**
     * The default encoding (UTF-8) is used to determine what characters are
     * represented by any consecutive sequences of the form "%xy".
     *
     * @param s
     *        the String to decode
     * @return the newly decoded String
     * @throws UnsupportedEncodingException
     *         If the default character encoding not supported
     */
    public String decode(String s) throws UnsupportedEncodingException {
        if (s == null) {
            return null;
        }

        boolean needToChange = false;
        int numChars = s.length();
        StringBuilder sb = new StringBuilder((numChars > 500) ? (numChars / 2)
                                                              : numChars);
        int i = 0;
        char c;
        byte[] bytes = null;

        while (i < numChars) {
            c = s.charAt(i);

            if (c == '%') {
                try {
                    // (numChars-i)/3 is an upper bound for the number
                    // of remaining bytes
                    if (bytes == null) {
                        bytes = new byte[(numChars - i) / 3];
                    }

                    int pos = 0;

                    while (((i + 2) < numChars) && (c == '%')) {
                        bytes[pos++] = (byte) Integer.parseInt(s.substring(i +
                                    1, i + 3), 16);
                        i += 3;

                        if (i < numChars) {
                            c = s.charAt(i);
                        }
                    }

                    if ((i < numChars) && (c == '%')) {
                        throw new IllegalArgumentException(
                            "URLDecoder: Incomplete trailing escape (%) pattern");
                    }

                    String string = new String(bytes, 0, pos, m_encoding);
                    sb.append(string);
                } catch (NumberFormatException e) {
                    throw new IllegalArgumentException(
                        "URLDecoder: Illegal hex characters in escape (%) pattern - " +
                        e.getMessage());
                }

                /*
                 * catch(UnsupportedEncodingException e) {
                 * m_logger.log(Level.SEVERE, "Unsupported encoding "+ m_encoding); needToChange =
                 * false; return s; }
                 */
                needToChange = true;
            } else {
                sb.append(c);
                i++;
            }
        }

        return (needToChange ? sb.toString() : s);
    }

    /**
     * The supplied encoding is used to determine what characters are represented
     * by any consecutive sequences of the form "%xy".
     *
     * @param s
     *        the String to decode
     * @param encoding
     *        The name of a supported character encoding.
     * @return the newly decoded String
     * @throws UnsupportedEncodingException
     *         If character encoding needs to be consulted, but named character
     *         encoding is not supported
     */
    public String decode(String s, String encoding)
        throws UnsupportedEncodingException {
        if (encoding.length() == 0) {
            throw new UnsupportedEncodingException(
                "URLDecoder: empty string enc parameter");
        }

        m_encoding = encoding;

        return decode(s);
    }
}
