/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package com.ericsson.ssa.sip;

import com.ericsson.eas.telurl.TelUrlResolver;
import com.ericsson.eas.telurl.TelUrlResolverException;

import com.ericsson.ssa.sip.dns.DnsResolver;

import java.io.IOException;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.jvnet.glassfish.comms.util.LogUtil;

import javax.servlet.sip.SipFactory;
import javax.servlet.sip.SipServletRequest;
import javax.servlet.sip.SipURI;
import javax.servlet.sip.URI;


/**
 * This class is used to wrap around a SIP request to access the URI regardless
 * of URI implementation (SipURI or TelURL), so that equivalent parts that are
 * not specified in the URI interface can be accessed.
 *
 *
 * @author ejoelbi
 * @since 2006-dec-21
 * @reviewed qmigkra 2007-jan-17
 *
 */
public class UriWrapper {
    private static Logger _log = LogUtil.SIP_LOGGER.getLogger();
    private static SipFactory sf = SipFactoryImpl.getInstance();
    static TelUrlResolver resolver = (TelUrlResolver) DnsResolver.getInstance();
    private URI transformedUri;
    private SipServletRequest req;
    private boolean hasDoneTransform;

    /**
     * Constructor.
     *
     * @param req
     *           the message containing the URI to be wrapped
     * @throws IllegalArgumentException
     *            if the message cintains an URI that is not a TelURLImpl nor a
     *            SipURI
     */
    public UriWrapper(SipServletRequest req) {
        this.req = req;

        URI uriToBeWrapped = req.getRequestURI();

        if (!(uriToBeWrapped instanceof SipURI ||
                uriToBeWrapped instanceof TelURLImpl)) {
            throw new IllegalArgumentException("The URI class: " +
                uriToBeWrapped.getClass() + " is not supported");
        }
    }

    /**
     * Gets the transformed URI
     *
     * @return the transformed URI
     */
    public URI getTransformedUri() {
        if (!hasDoneTransform) {
            URI uri = req.getRequestURI();
            transformedUri = transformURI(uri);

            if (SipFactoryImpl.TEL_URI_PROTOCOL.equals(uri.getScheme()) &&
                    (transformedUri == null)) {
                transformedUri = uri;
            }

            hasDoneTransform = true;
        }

        return transformedUri;
    }

    /**
     * Transform a URI to a SipURI. If the URI is a TelURL then perform an
     * ENUM lookup of the SipURI.
     *
     * @param uri
     * @return SipURI or null
     */
    private SipURI transformURI(URI uri) {
        SipURI sipUri = null;

        if (_log.isLoggable(Level.FINE)) {
            _log.log(Level.FINE, "transformURI uri: " + uri);
        }

        if (uri.isSipURI()) {
            sipUri = (SipURI) uri;

            String user = sipUri.getParameter("user");

            //           if(_log.isLoggable(Level.FINE))
            //           {
            //              _log.log(Level.FINE, "transformURI user: "+user);
            //           }
            if ((user != null) && user.equals("phone")) {
                //Do the ENUM first
                sipUri = resolve(uri);
            }
        } else if (resolver.isTelephoneNumber(uri)) {
            sipUri = resolve(uri);
        }

        return sipUri;
    }

    /**
     * Resolve a TelURL to the users SipURI
     * @param uri
     * @return
     */
    private SipURI resolve(URI uri) {
        SipURI sipUri = null;

        if (_log.isLoggable(Level.FINE)) {
            _log.log(Level.FINE, "resolve uri: " + uri);
        }

        //      if (uri.isPhoneNumber()) {
        if (resolver.isTelephoneNumber(uri)) {
            try {
                sipUri = resolver.lookupSipURI(uri);

                if (_log.isLoggable(Level.FINE)) {
                    _log.log(Level.FINE, "uri: " + uri);
                    _log.log(Level.FINE, "resolves to: " + sipUri);
                }
            } catch (IOException ioe) {
                if (_log.isLoggable(Level.FINE)) {
                    _log.log(Level.FINE, "failed to resolve: " + ioe);
                }
            } catch (TelUrlResolverException telExc) {
                if (_log.isLoggable(Level.FINE)) {
                    _log.log(Level.FINE,
                        "TelUrlResolverException, failed to resolve: " + uri);
                }
            }
        }

        return sipUri;
    }

    /**
     * Sets the specified parameter of the URI in the request to the specified
     * value.
     *
     * @param name
     *           parameter name
     * @param value
     *           parameter value
     */
    public void setParameter(String name, String value) {
        UriUtil.setParameter(req.getRequestURI(), name, value);
    }

    /**
     * Remove the specified parameter of the URI in the request.
     *
     * @param name
     *           the parameter name
     * @return the value of the removed parameter (or null if parameter does not
     *         exist)
     */
    public String removeParameter(String name) {
        return UriUtil.removeParameter(req.getRequestURI(), name);
    }

    /**
     * Gets the value of the specified parameter of the URI in the request.
     *
     * @param name
     *           parameter name
     */
    public String getParameter(String name) {
        return UriUtil.getParameter(req.getRequestURI(), name);
    }

    @Override
    public String toString() {
        return "Request-URI: " + req.getRequestURI() +
        "; Transformed Request-URI: " + getTransformedUri();
    }
}
