/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package com.ericsson.ssa.sip.timer;

import java.util.concurrent.Callable;
import java.util.concurrent.ScheduledFuture;


/**
 * Common base class for Timer implementations.
 *
 *
 *
 * @author qmaghes
 * @since 2007 nov 21
 *
 */
public abstract class GeneralTimerBase implements GeneralTimer, Callable {
    /**
      * Whether this timer has been successfully cancelled. Used for debugging.
      */
    @SuppressWarnings("unused")
    protected Boolean iscanceled = null;

    /**
      * Lock that prevents simultaneous execution. Note! Do NOT call methods in
      * the associated sip session within synchronization of this lock since there
      * may be a dead lock.
      */
    protected final Object SIPP_APPLICATION_LOCK = new Object();
    
    public Object call() throws Exception {
        try {
            timeout();
        } finally {
            if (getPFieldIsRepeatingTimer()) {
                estimateNextExecution();
            } else {
                // this non-repeating timer is now "ready"
                // and should not be included in the list of active timers
                // The application may already have canceled() the timer though
                cancel(); // dont bother about return value....
            }
        }
        return null;
    }

    protected abstract void timeout();

    /**
      * Setter to set what should be executed.
      *
      * @param f
      */
    public void setFuture(ScheduledFuture<?> f) {
        synchronized (SIPP_APPLICATION_LOCK) {
            setFieldFuture(f);
        }
    }

    /**
      * Helper to calculate when next execution time is.
      *
      */
    private void estimateNextExecution() {
        synchronized (SIPP_APPLICATION_LOCK) {
            if (getPFieldFixedDelay()) {
                setPFieldScheduledExecutionTime(getPFieldPeriod() +
                    System.currentTimeMillis());
            } else {
                if (getPFieldFirstExecution() == 0) {
                    // save timestamp of first execution
                    setPFieldFirstExecution(getPFieldScheduledExecutionTime());
                }

                setPFieldScheduledExecutionTime(getPFieldFirstExecution() +
                    (incrementAndGetPFieldNumInvocations() * getPFieldPeriod()));
            }
        }
    }

    /**
      * Getter for delay property.
      *
      * @return
      */
    public long getDelay() {
        return getPFieldDelay();
    }

    /**
      * Getter for period property
      *
      * @return
      */
    public long getPeriod() {
        return getPFieldPeriod();
    }

    public long scheduledExecutionTime() {
        synchronized (SIPP_APPLICATION_LOCK) {
            return getPFieldScheduledExecutionTime();
        }
    }

    public void cancel() {
        cancel(false);
    }

    /**
      *
      * Cancel this timer, possibly by also interrupting the thread (from the
      * thread pool) running the task. Note that interupting the thread may have
      * undesired consequences.
      *
      * @param mayInterruptIfRunning
      */
    public void cancel(boolean mayInterruptIfRunning) {
        synchronized (SIPP_APPLICATION_LOCK) {
            ScheduledFuture<?> future = getPFieldFuture();

            if (future != null) {
                // need to force cancel to get rid of
                // the task which is currently scheduled
                boolean res = future.cancel(mayInterruptIfRunning);
                // used for debugging/optimizeIt purpose
                // kan be kept in production code since object should
                // be due for gc anyway....
                iscanceled = new Boolean(res);
                setFieldFuture(null);
            }
        }
    }

    protected abstract ScheduledFuture<?> getPFieldFuture();

    protected abstract void setFieldFuture(ScheduledFuture<?> future);

    protected abstract boolean getPFieldIsRepeatingTimer();

    protected abstract long getPFieldFirstExecution();

    protected abstract void setPFieldFirstExecution(long firstExecutionTime);

    protected abstract boolean getPFieldFixedDelay();

    protected abstract long getPFieldPeriod();

    protected abstract long getPFieldScheduledExecutionTime();

    protected abstract void setPFieldScheduledExecutionTime(
        long scheduledExecutionTime);

    protected abstract long incrementAndGetPFieldNumInvocations();

    protected abstract long getPFieldDelay();

    /**
      * Predicate for isRepeating property.
      *
      * @return
      */
    protected boolean isRepeatingTimer() {
        return getPFieldIsRepeatingTimer();
    }

    /**
      * Predicate for fixed delay property
      *
      * @return
      */
    protected boolean isFixedDelay() {
        return getPFieldFixedDelay();
    }
}
