/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package com.ericsson.ssa.sip.timer;

import com.ericsson.ssa.config.ConvergedContext;
import org.jvnet.glassfish.comms.deployment.backend.SipApplicationListeners;
import com.ericsson.ssa.sip.LifeCycle;
import com.ericsson.ssa.sip.PrivilegedGetSipSessionManagerRegistry;
import com.ericsson.ssa.sip.RemoteLockException;
import com.ericsson.ssa.sip.RemoteLockRuntimeException;
import com.ericsson.ssa.sip.SipApplicationSessionImpl;
import com.ericsson.ssa.sip.SipSessionManager;
import com.ericsson.ssa.sip.SipSessionManagerRegistry;

import org.apache.catalina.Globals;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;

import java.security.AccessController;

import java.util.UUID;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import org.jvnet.glassfish.comms.util.LogUtil;
import java.util.logging.Logger;

import javax.servlet.sip.TimerListener;
import javax.servlet.sip.ServletTimer;

public class ServletTimerImpl extends ServletTimerImplBase
    implements Serializable, LifeCycle {
    private static Logger logger = LogUtil.SIP_LOGGER.getLogger();

    /**
     * Class identifier for serialization
     */
    private static final long serialVersionUID = 8123619219185134043L;
    
    /** The serialized format versioning. 1 = first version. */
    private static short serializedFormVersion  = 1;


    /**
     * A future delayed scheduled action to be run
     */
    private ScheduledFuture<?> future;

    /**
     * Information about what this timer is.
     */
    private Serializable info;

    /**
     * The id of the associated SipApplicationSession
     */
    protected String sasId = null;

    /**
     * The application id of the associated context
     */
    private String appId = null;

    /**
     * The associated session manager
     */
    private SipSessionManager sipSessionManager;

    /**
     * Absolute time in milliseconds for next execution.
     */
    private long scheduledExecutionTime = 0;

    /**
     * Delay from creation of timer to execution
     */
    private long delay = 0;

    /**
     * Period between executions. Only applicable for repeating timers.
     */
    private long period = 0;

    /**
     * Number of times execution has happened.
     */
    private long numInvocations = 0;

    /**
     * Absolute time for first execution.
     */
    private long firstExecution = 0;

    /**
     * Whether executions should be scheduled with fixed delay.
     *
     * @see java.util.Timer for semantics.
     */
    private boolean fixedDelay = false;

    /**
     * Whether this timer is persistent.
     */
    private boolean persistent = false;

    /**
     * Registered listener that will get a timeout event when executed.
     */
    private TimerListener listener;
    private transient AtomicBoolean listenerLookedUp = null;

    /**
     * Whether execution should be repeated.
     */
    private boolean isRepeatingTimer = true;

    /**
     * The unique id of this timer.
     */
    private String id = null;
    protected String fullyQualifiedId;

    /**
     * Constructor for non-repeating timer.
     *
     * @param manager The associated session manager
     * @param sas The SipApplicationSession with which this ServletTimer has
     * been associated
     * @param info Information about the timer
     * @param delay Delay until execution
     * @param listener Listener that will get timeout events.
     */
    public ServletTimerImpl(SipSessionManager manager,
        SipApplicationSessionImpl sas, Serializable info, long delay,
        TimerListener listener) {
        this(manager, sas, info, delay, false, 0, listener);
        isRepeatingTimer = false;
    }

    /**
     * Constructor for repeating times
     *
     * @param manager The associated session manager
     * @param sas The SipApplicationSession with which this ServletTimer has
     * been associated
     * @param info Information about the timer
     * @param delay Delay until first execution
     * @param fixedDelay Whether fixed delay mode should be used
     * @param period Period between execution
     * @param listener Listener that will get timeout events.
     */
    public ServletTimerImpl(SipSessionManager manager,
        SipApplicationSessionImpl sas, Serializable info, long delay,
        boolean fixedDelay, long period, TimerListener listener) {
        this.sipSessionManager = manager;
        this.appId = manager.getApplicationId();
        this.sasId = sas.getId();
        this.info = info;
        this.delay = delay;
        scheduledExecutionTime = delay + System.currentTimeMillis();
        this.fixedDelay = fixedDelay;
        this.period = period;
        this.listener = listener;
        this.listenerLookedUp = new AtomicBoolean(true);

        id = UUID.randomUUID().toString();
        sas.addServletTimer(this);
        fullyQualifiedId = "ServletTimer with id " + appId + ":" + id;
    }

    /**
     * @serialData See serialized form version 1 in #readObject(ObjectInputStream in)
     *  
     * @param out the stream to write the object members
     * @throws IOException
     */
    private void writeObject(ObjectOutputStream out) throws IOException {
        out.writeShort(serializedFormVersion);
        
        out.writeUTF(id);
        out.writeObject(info);

        out.writeUTF(appId);
        out.writeUTF(sasId);

        out.writeLong(scheduledExecutionTime);
        out.writeLong(delay);
        out.writeLong(period);
        out.writeLong(numInvocations);
        out.writeLong(firstExecution);
        out.writeBoolean(fixedDelay);
        out.writeBoolean(isRepeatingTimer);
        out.writeBoolean(persistent);
    }

    /**
     * @serialData first field is an short and represents the serializedFormVersion.<br>
     * <h3>Data layout for serializedFormVersion = 1 follows</h3>
     * 
     * <li>field is a <b>String</b> and represents timerId field</li>
     * <li>field is a <b>Serializable</b> and represents attached info object for this timer</li>
     * <li>field is a <b>String</b> and represents applicationId field</li>
     * <li>field is a <b>String</b> and represents sip applicationSessionId field</li>
     * <li>field is a <b>Long</b> and represents the scheduledExecutionTime field as absolute time</li>
     * <li>field is a <b>Long</b> and represents the delay field in milliseconds</li>
     * <li>field is a <b>Long</b> and represents the period field in milliseconds</li>
     * <li>field is a <b>Long</b> and represents the numInvocations field</li>
     * <li>field is a <b>Long</b> and represents the firstExecution field</li>
     * <li>field is a <b>Boolean</b> and represents the fixedDelay field</li>
     * <li>field is a <b>Boolean</b> and represents the isRepeatingTimer field</li>
     * <li>field is a <b>Boolean</b> and represents the persistent field</li>
     *  
     * @param in the stream to read the object members
     * @throws IOException is thrown when unsupported version is detected
     * @throws ClassNotFoundException
     */
    private void readObject(ObjectInputStream in)
        throws IOException, ClassNotFoundException {
        short readSerializedFormVersion = in.readShort();
        switch(readSerializedFormVersion) {
        case 1:
            id = in.readUTF();
            info = (Serializable) in.readObject();
    
            // Read the appid, and use it to get the corresponding
            // SipSessionManager and associated context.
            appId = in.readUTF();
            sasId = in.readUTF();
    
            fullyQualifiedId = "ServletTimer with id " + appId + ":" + id;
    
            scheduledExecutionTime = in.readLong();
            delay = in.readLong();
            period = in.readLong();
            numInvocations = in.readLong();
            firstExecution = in.readLong();
            fixedDelay = in.readBoolean();
            isRepeatingTimer = in.readBoolean();
            persistent = in.readBoolean();
            listenerLookedUp = new AtomicBoolean(false); // Need to explicitely instantiate.
            break;
        default:
            throw new IOException("Unable to deserialize into "
                    + this.getClass().getName()
                    + " due to unknown serializedFormVersion of "
                    + readSerializedFormVersion);
        }
    }
    
    /**
     * @see com.ericsson.ssa.sip.timer.ServletTimerImpl#getAppId()
     */
    public String getAppId() {
        return appId;
    }    

    // --- PField accessors ---
    @Override
    protected String getPFieldId() {
        return id;
    }

    @Override
    protected ScheduledFuture<?> getPFieldFuture() {
        return future;
    }

    @Override
    protected void setFieldFuture(ScheduledFuture<?> future) {
        this.future = future;
    }

    @Override
    protected Serializable getPFieldInfo() {
        return info;
    }

    @Override
    protected synchronized SipApplicationSessionImpl getPFieldAppSession() {
        SipApplicationSessionImpl sas = null;
        try {
            sas = getSipSessionManager().findSipApplicationSession(sasId);
        } catch (RemoteLockException e) {
            throw new RemoteLockRuntimeException(e); 
        }

        if (sas == null) {
            logger.log(Level.WARNING, "SipApplication Session " + sasId +
            " not found");
        }
        return sas;
    }

    @Override
    protected long getPFieldFirstExecution() {
        return firstExecution;
    }

    @Override
    protected long getPFieldPeriod() {
        return period;
    }

    @Override
    protected long getPFieldScheduledExecutionTime() {
        return scheduledExecutionTime;
    }

    @Override
    protected long incrementAndGetPFieldNumInvocations() {
        return ++numInvocations;
    }

    @Override
    protected void setPFieldFirstExecution(long firstExecution) {
        this.firstExecution = firstExecution;
    }

    @Override
    protected void setPFieldScheduledExecutionTime(long scheduledExecutionTime) {
        this.scheduledExecutionTime = scheduledExecutionTime;
    }

    @Override
    protected long getPFieldDelay() {
        return delay;
    }

    @Override
    protected boolean getPFieldFixedDelay() {
        return fixedDelay;
    }

    @Override
    protected boolean getPFieldPersistent() {
        return persistent;
    }

    @Override
    protected void setPFieldPersistent(boolean p) {
        this.persistent = p;
    }

    @Override
    protected boolean getPFieldIsRepeatingTimer() {
        return isRepeatingTimer;
    }

    @Override
    protected synchronized TimerListener getPFieldListener() {
        if (!listenerLookedUp.getAndSet(true)) {
            SipSessionManager sipSessionManager = getSipSessionManager();

            if (sipSessionManager != null) {
                ConvergedContext ctx = sipSessionManager.getContext();

                if (ctx != null) {
                    SipApplicationListeners appListeners = ctx.getSipApplicationListeners();

                    if (appListeners != null) {
                        listener = appListeners.getTimerListener();
                    }
                }
            }
        }

        return listener;
    }

    protected synchronized SipSessionManager getSipSessionManager() {
        if (sipSessionManager == null) {
            SipSessionManagerRegistry reg = null;

            if (Globals.IS_SECURITY_ENABLED) {
                reg = (SipSessionManagerRegistry) AccessController.doPrivileged(new PrivilegedGetSipSessionManagerRegistry());
            } else {
                reg = SipSessionManagerRegistry.getInstance();
            }

            if (reg != null) {
                sipSessionManager = reg.get(appId);
            }

            if (sipSessionManager == null) {
                logger.log(Level.WARNING, "SipSessionManager " + appId + " not found");
            }
        }

        return sipSessionManager;
    }

    protected String getFullyQualifiedId() {
        return fullyQualifiedId;
    }

    private void restartTimer() {
        // Rescedule timer
        TimerServiceImpl.getInstance().rescheduleTimer(this);
    }

    /*
     * Cancels this timer.
     */
    @Override
    public void cancel(boolean mayInterruptIfRunning) {
        super.cancel(mayInterruptIfRunning);
        getSipSessionManager().removeServletTimer(this);
    }

    /*
     * (non-Javadoc)
     * @see com.ericsson.ssa.sip.LifeCycle#activate()
     */
    public boolean activate() {

        // Add to active cache
        SipSessionManager mgr = getSipSessionManager();

        if (mgr != null) {
            ServletTimer st = mgr.addServletTimer(this);
            if (st != null) {
                // While this thread has been in the process of loading the
                // requested ServletTimer, and is now trying to activate it,
                // another thread has also loaded the same ServletTimer, and
                // has already added it to the active cache. Abort activation.
                return true;
            }
        }

        // Restart timer if not already started
        if (future == null) {
            restartTimer();
        }

        return true;
    }

    /*
     * (non-Javadoc)
     * @see com.ericsson.ssa.sip.LifeCycle#passivate()
     */
    public void passivate() {

        // Remove from active cache
        SipSessionManager mgr = getSipSessionManager();
        if (mgr != null) {
            mgr.removeServletTimer(this);
        }

        stop(true);
    }

    /*
     * (non-Javadoc)
     *
     * @see java.lang.Object#toString()
     */
    public String toString() {
        StringBuilder sb = new StringBuilder();
        String fullyQualifiedId = getFullyQualifiedId();

        if (fullyQualifiedId != null) {
            sb.append(fullyQualifiedId).append('\n');
        }

        sb.append("Info = ").append(getPFieldInfo()).append('\n');
        sb.append("Scheduled execution time = ")
          .append(getPFieldScheduledExecutionTime()).append('\n');
        sb.append("Time now = ").append(System.currentTimeMillis()).append('\n');
        sb.append("SipApplicationSession = ").append(sasId)
          .append('\n');
        sb.append("ScheduledFuture = ").append(getPFieldFuture()).append('\n');
        sb.append("Delay = ").append(getPFieldDelay()).append('\n');

        return sb.toString();
    }

}
