/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package com.ericsson.ssa.sip.timer;

import com.ericsson.ssa.sip.SipApplicationSessionImpl;

import org.apache.catalina.session.SessionLock;

import java.io.Serializable;

import java.util.concurrent.ScheduledFuture;

import javax.servlet.sip.ServletTimer;
import javax.servlet.sip.SipApplicationSession;
import javax.servlet.sip.TimerListener;


/**
 * Implementation of SSA interface <code>javax.servlet.sip.ServletTimer</code>.
 */
public abstract class ServletTimerImplBase extends GeneralTimerBase
    implements ServletTimer {
    private final SessionLock timerLock = new SessionLock();

    // Empty constructor for serialization
    public ServletTimerImplBase() {
    }

    public void timeout() {
        getPFieldListener().timeout(this);
    }

    /**
     * Cancels this timer, possibly by also interrupting the thread (from the
     * thread pool) running the task. Note that interupting the thread may have
     * undesired consequences.
     *
     * As part of its cancellation, this timer is also removed from its parent
     * SipApplicationSession.
     * 
     * @param mayInterruptIfRunning
     */
    @Override
    public void cancel(boolean mayInterruptIfRunning) {
        super.cancel(mayInterruptIfRunning);
        SipApplicationSessionImpl sas = getPFieldAppSession();
        if (sas != null) {
            sas.cancelServletTimer(this);
        }
    }

    /**
     * Stops this timer, possibly by also interrupting the thread (from the
     * thread pool) running the task. Note that interupting the thread may have
     * undesired consequences.
     *
     * In contrast to cancel, this method does not remove this timer from its
     * parent SipApplicationSession.
     *
     * @param mayInterruptIfRunning
     */
    protected void stop(boolean mayInterruptIfRunning) {
        super.cancel(mayInterruptIfRunning);
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.servlet.sip.ServletTimer#getApplicationSession()
     */
    public SipApplicationSession getApplicationSession() {
        return getPFieldAppSession();
    }

    /*
     * (non-Javadoc)
     *
     * @see javax.servlet.sip.ServletTimer#getInfo()
     */
    public Serializable getInfo() {
        return getPFieldInfo();
    }

    /**
     * Gets the unique id of this timer.
     *
     * @return The unique id of this timer
     */
    public String getId() {
        return getPFieldId();
    }

    /**
     * Predicate for persistence property
     *
     * @return
     */
    public boolean isPersistent() {
        return getPFieldPersistent();
    }

    /**
     * Setter for persistence property
     *
     * @param p
     */
    public void setPersistent(boolean p) {
        setPFieldPersistent(p);

        SipApplicationSessionImpl appSession = getPFieldAppSession();

        if (p && (appSession != null)) {
            appSession.setServletTimerShouldBePersisted(this);
        }
    }

    /* To be implemented */
    public String getID() {
        throw new RuntimeException("Not yet implemented");
    }

    /**
     * Getter for listener object
     *
     * @return
     */
    TimerListener getListener() {
        return getPFieldListener();
    }

    //begin lock related code

    /**
     * get this timer locked for foreground if the timer is found to be
     * presently background locked; retry logic in a time-decay polling loop
     * waits for background lock to clear after 6 attempts (12.6 seconds) it
     * unlocks the timer and acquires the foreground lock
     */
    public boolean lockForegroundWithRetry() {
        boolean result = false;

        long pollTime = 200L;
        int tryNumber = 0;
        int maxNumberOfRetries = 7;
        boolean keepTrying = true;

        // try to lock up to numTries (i.e. 7) times
        // poll and wait starting with 200 ms
        while (keepTrying) {
            boolean lockResult = lockForeground();
            if (lockResult) {
                keepTrying = false;
                result = true;
                break;
            }

            tryNumber++;

            if (tryNumber < maxNumberOfRetries) {
                pollTime = pollTime * 2L;
                try {
                    Thread.sleep(pollTime);
                } catch (InterruptedException e) {
                    ;
                }
            } else {
                // unlock the background so we can take over
                // FIXME: need to log warning for this situation
                unlockBackground();
            }
        }

        return result;
    }

    /**
     * return whether this timer is currently foreground locked
     */
    public synchronized boolean isForegroundLocked() {
        return timerLock.isForegroundLocked();
    }

    /**
     * lock the timer for foreground returns true if successful; false if
     * unsuccessful
     */
    public synchronized boolean lockBackground() {
        return timerLock.lockBackground();
    }

    /**
     * lock the timer for background returns true if successful; false if
     * unsuccessful
     */
    public synchronized boolean lockForeground() {
        return timerLock.lockForeground();
    }

    /**
     * unlock the timer completely irregardless of whether it was foreground
     * or background locked
     */
    public synchronized void unlockForegroundCompletely() {
        timerLock.unlockForegroundCompletely();
    }

    /**
     * unlock the timer from foreground
     */
    public synchronized void unlockForeground() {
        timerLock.unlockForeground();
    }

    /**
     * unlock the timer from background
     */
    public synchronized void unlockBackground() {
        timerLock.unlockBackground();
    }

    /**
     * return the timer lock
     */
    public SessionLock getTimerLock() {
        return timerLock;
    }

    //end lock related code

    // --- Persistable fields
    protected abstract String getPFieldId();

    protected abstract Serializable getPFieldInfo();

    protected abstract SipApplicationSessionImpl getPFieldAppSession();

    protected abstract boolean getPFieldPersistent();

    protected abstract void setPFieldPersistent(boolean p);

    protected abstract String getFullyQualifiedId();

    protected abstract TimerListener getPFieldListener();
}
