    /*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 *
 * Contributor(s):
 *
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.jvnet.glassfish.comms.admin.gui.extensions.handlers;

import com.sun.appserv.management.config.AMXConfig;
import com.sun.enterprise.tools.admingui.handlers.ApplicationHandlers;
import com.sun.enterprise.tools.admingui.util.AMXUtil;
import com.sun.enterprise.tools.admingui.util.GuiUtil;
import com.sun.enterprise.tools.admingui.util.JMXUtil;
import com.sun.enterprise.tools.admingui.util.TargetUtil;
import com.sun.jsftemplating.annotation.Handler;
import com.sun.jsftemplating.annotation.HandlerInput;
import com.sun.jsftemplating.annotation.HandlerOutput;
import com.sun.jsftemplating.layout.descriptors.handler.HandlerContext;
import com.sun.webui.jsf.model.Option;
import com.sun.webui.jsf.model.UploadedFile;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Random;
import java.util.Set;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import org.jvnet.glassfish.comms.admin.gui.extensions.util.CLBHelper;
import org.jvnet.glassfish.comms.admin.gui.extensions.util.SipTargetUtil;
import org.jvnet.glassfish.comms.admin.gui.extensions.util.SipUtil;


/**
 *
 * @author irfan@sun.com
 */

public class CLBHandlers {
    
    /** Creates a new instance of CLBHandlers */
    public CLBHandlers() {
    }
    
    /** The function returns the load balancers specified in the configs for instances/clusters
     */
    @Handler(id="getCLBList",
    output={
        @HandlerOutput(name="Result", type=List.class)
    })    
    public static void getCLBList(HandlerContext handlerContext) {
        List result = new ArrayList();
        try {
            ArrayList<String> clbs = CLBHelper.getConvergedLoadBalancers();
            for (String name : clbs) {
                HashMap row = new HashMap();
                row.put("name", name);
                row.put("selected", false);
                row.put("isSelfLB", CLBHelper.isSelfLB(name));                
                result.add(row);
            }
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }
        handlerContext.setOutputValue("Result", result);
    }
    
    
    /** The handler deletes the CLB based on the CLB Name. The backend will figure out which
     *  target the CLB is on as their is 1:1 mapping between a CLB and a target.
     */
    @Handler(id="deleteCLB",
    input={
        @HandlerInput(name="selectedRows", type=List.class, required=true)
    })
    public static void deleteCLB(HandlerContext handlerContext) {
        try {
            List<Map> selectedRows = (List) handlerContext.getInputValue("selectedRows");
            for(Map row: selectedRows) {
                String clbName = (String) row.get("name");
                CLBHelper.deleteCLB(clbName);
            }
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }
    }
    
    
    /**
     * This handler creats the converged LB and an associated converged LB Config
     */
    @Handler(id="createConvergedLoadBalancer",
    input={
        @HandlerInput(name="lbName", type=String.class, required=true),
        @HandlerInput(name="autoApply", type=Boolean.class),
        @HandlerInput(name="configFileName", type=String.class),
        @HandlerInput(name="targets", type=String[].class),
	@HandlerInput(name="selfLB", type=Boolean.class)
    })
    public static void createConvergedLoadBalancer(HandlerContext handlerContext) {
        String clbName = (String) handlerContext.getInputValue("lbName");
        try {
            CLBHelper.createCLB(clbName, handlerContext);
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }
                
    }
    
    /** The function returns the targets that do not have a CLB associated with it
     */
    @Handler(id="getTargetsForNewCLB",
    output={
        @HandlerOutput(name="availableTargets", type=Option[].class),
        @HandlerOutput(name="selectedTargets", type=String[].class)
    })
    public static void getTargetsForNewCLB(HandlerContext handlerContext) {        
        ArrayList targets = CLBHelper.getTargetsWithoutCLB();        
        Option[] availableOptions = null;
        if(targets != null) {
            availableOptions = new Option[targets.size()];
            List<String> strList = GuiUtil.convertListOfStrings(targets);
            availableOptions = GuiUtil.getSunOptions(strList);
        }
        handlerContext.setOutputValue("availableTargets", availableOptions);
        handlerContext.setOutputValue("selectedTargets", new String[]{""});
    }

    
    @Handler(id="getCLBInfo",
    input={
	@HandlerInput(name="lbName", type=String.class, required=false),
	@HandlerInput(name="targetName", type=String.class, required=false),
	@HandlerInput(name="instanceName", type=String.class)
    },
    output={
        @HandlerOutput(name="clbInfoMap", type=Map.class),
        @HandlerOutput(name="isCluster", type=Boolean.class),
        @HandlerOutput(name="isInstance", type=Boolean.class),
        @HandlerOutput(name = "Properties", type = Map.class)
    })
    public static void getCLBInfo(HandlerContext handlerContext) {
        String clbName = (String) handlerContext.getInputValue("lbName");
	String targetName = (String) handlerContext.getInputValue("targetName");
	if ((targetName == null) || (targetName.trim().length() == 0)) {
	    // Instance name is used in case of standalone instances
	    targetName = (String) handlerContext.getInputValue("instanceName");
	    if (targetName == null) {
		targetName = CLBHelper.getCLBHostingTarget(clbName);
	    }
	}
	
        HashMap clbInfoMap = new HashMap();
        Boolean isCluster = TargetUtil.isCluster(targetName);
        try {
            clbInfoMap = CLBHelper.getCLBInfo(targetName);
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }
        handlerContext.setOutputValue("isCluster", isCluster);
        handlerContext.setOutputValue("isInstance", !isCluster);
        handlerContext.setOutputValue("clbInfoMap", clbInfoMap);
    }
    
    
    @Handler(id="getCLBRefInfo",
    input={
        @HandlerInput(name="lbName", type=String.class, required=true),
        @HandlerInput(name="targetName", type=String.class, required=true)
    },
    output={
        @HandlerOutput(name="clbRefInfoMap", type=Map.class),
        @HandlerOutput(name="isCluster", type=Boolean.class),
        @HandlerOutput(name="isInstance", type=Boolean.class),
        @HandlerOutput(name = "Properties", type = Map.class)
    })
    public static void getCLBRefInfo(HandlerContext handlerContext) {
        String clbName = (String) handlerContext.getInputValue("lbName");
        String targetName = (String) handlerContext.getInputValue("targetName");
        HashMap clbRefInfoMap = new HashMap();
        Boolean isCluster = TargetUtil.isCluster(targetName);
        try {
            clbRefInfoMap = CLBHelper.getCLBRefInfo(clbName, targetName);
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }
        handlerContext.setOutputValue("isCluster", isCluster);
        handlerContext.setOutputValue("isInstance", !isCluster);
        handlerContext.setOutputValue("clbRefInfoMap", clbRefInfoMap);
    }
    
    
    @Handler(id="saveCLBInfo",
    input={
        @HandlerInput(name="lbName", type=String.class, required=true),
        @HandlerInput(name="targetName", type=String.class, required=true),
        @HandlerInput(name="clbInfoMap", type=Map.class, required=true),
        @HandlerInput(name="AddProps", type=Map.class),
        @HandlerInput(name="RemoveProps", type=List.class)
    })
    public static void saveCLBInfo(HandlerContext handlerContext) {
        String targetName = (String) handlerContext.getInputValue("targetName");
        HashMap clbInfoMap = (HashMap) handlerContext.getInputValue("clbInfoMap");
        try {
            ObjectName clb = CLBHelper.getCLB(targetName);
            if (clb == null) {
                GuiUtil.handleError(handlerContext, "This Converged load balancer does not exist.");
                return;
            }
            HashMap attrs = new HashMap();
            attrs.put(CLBHelper.CLB_KEYS.CONFIG_FILE, clbInfoMap.get("configFile"));
            attrs.put(CLBHelper.CLB_KEYS.AUTO_COMMIT, clbInfoMap.get("autoApply"));
            SipUtil.setAttributeValues(clb, attrs);
            
            //NEED TO SAVE PROXY
            attrs.clear();
            attrs.put(CLBHelper.CLB_PROXY_KEYS.REQUEST_POOL_SIZE, clbInfoMap.get("reqPoolSize"));
            attrs.put(CLBHelper.CLB_PROXY_KEYS.SEND_RETRY_COUNT, clbInfoMap.get("sendRetryCount"));
            attrs.put(CLBHelper.CLB_PROXY_KEYS.READ_TIMEOUT_IN_MILLIS, clbInfoMap.get("readTimeout"));
            ObjectName proxy = (ObjectName) SipUtil.getChildByFunction(clb, "getProxy");
            if (proxy == null) {
                proxy = SipUtil.createChildByFunction(clb, "createProxy");
            }
            SipUtil.setAttributeValues(proxy, attrs);
            
            HashMap addProps = (HashMap) handlerContext.getInputValue("AddProps");
            ArrayList removeProps = (ArrayList) handlerContext.getInputValue("RemoveProps");
            SipUtil.setProperties(proxy, addProps, removeProps);
            
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }
    }
    
    
    @Handler(id="getDefaultCLBInfo",
    input={
        @HandlerInput(name="lbName", type=String.class, required=true),
        @HandlerInput(name="targetName", type=String.class, required=true)
    },
    output={
        @HandlerOutput(name="clbInfoMap", type=Map.class)
    })
    public static void getDefaultCLBInfo(HandlerContext handlerContext) {
        String lbName = (String) handlerContext.getInputValue("lbName");
        String targetName = (String) handlerContext.getInputValue("targetName");
        HashMap clbInfoMap = SipUtil.getDefaultAttributeValues(CLBHelper.CLB_KEYS.ELEMENT_NAME);
        
        try {
            clbInfoMap.put("autoApply", clbInfoMap.remove(CLBHelper.CLB_KEYS.AUTO_COMMIT));
            clbInfoMap.putAll(SipUtil.getDefaultAttributeValues(CLBHelper.CLB_PROXY_KEYS.ELEMENT_NAME));
            clbInfoMap.put("reqPoolSize", clbInfoMap.remove(CLBHelper.CLB_PROXY_KEYS.REQUEST_POOL_SIZE));
            clbInfoMap.put("sendRetryCount", clbInfoMap.remove(CLBHelper.CLB_PROXY_KEYS.SEND_RETRY_COUNT));
            clbInfoMap.put("readTimeout", clbInfoMap.remove(CLBHelper.CLB_PROXY_KEYS.READ_TIMEOUT_IN_MILLIS));
            
            ObjectName clb = CLBHelper.getCLB(targetName);
            if(clb != null) {
                clbInfoMap.put("configFile", (String)JMXUtil.getAttribute(clb, CLBHelper.CLB_KEYS.CONFIG_FILE));
                ObjectName proxy = (ObjectName) SipUtil.getChildByFunction(clb, "getProxy");
                if (proxy != null) {
                    HashMap props = SipUtil.getProperties(proxy);
                    clbInfoMap.put("props", props);
                }
            }
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }
        handlerContext.setOutputValue("clbInfoMap", clbInfoMap);
    }
    
    
    @Handler(id="getDefaultCLBRefInfo",
    input={
        @HandlerInput(name="lbName", type=String.class, required=true),
        @HandlerInput(name="targetName", type=String.class, required=true)
    },
    output={
        @HandlerOutput(name="clbRefInfoMap", type=Map.class)
    })
    public static void getDefaultCLBRefInfo(HandlerContext handlerContext) {
        String lbName = (String) handlerContext.getInputValue("lbName");
        String targetName = (String) handlerContext.getInputValue("targetName");
        HashMap clbRefInfoMap = new HashMap();
        try {
            if(TargetUtil.isCluster(targetName)) {
                clbRefInfoMap = SipUtil.getDefaultAttributeValues(CLBHelper.CLB_CLUSTER_REF_KEYS.ELEMENT_NAME);
                clbRefInfoMap.put("selfLoadBalancing", clbRefInfoMap.remove(
                        CLBHelper.CLB_CLUSTER_REF_KEYS.SELF_LOADBALANCE));
            } else {
                clbRefInfoMap = SipUtil.getDefaultAttributeValues(CLBHelper.SERVER_REF_KEYS.ELEMENT_NAME);
                clbRefInfoMap.put("disableTimeoutInMinutes", 
                        clbRefInfoMap.remove(CLBHelper.SERVER_REF_KEYS.DISABLE_TIMEOUT_IN_MINUTES));
                clbRefInfoMap.put("enabled", 
                        clbRefInfoMap.remove(CLBHelper.SERVER_REF_KEYS.ENABLED));
                clbRefInfoMap.put("lbEnabled", 
                        clbRefInfoMap.remove(CLBHelper.SERVER_REF_KEYS.LB_ENABLED));
                clbRefInfoMap.putAll(SipUtil.getDefaultAttributeValues(CLBHelper.HEALTH_CHECKER_KEYS.ELEMENT_NAME));
                clbRefInfoMap.put("intervalInSeconds", 
                        clbRefInfoMap.remove(CLBHelper.HEALTH_CHECKER_KEYS.INTERVAL_IN_SECONDS));
                clbRefInfoMap.put("timeoutInSeconds", 
                        clbRefInfoMap.remove(CLBHelper.HEALTH_CHECKER_KEYS.TIMEOUT_IN_SECONDS));
                clbRefInfoMap.put("url", 
                        clbRefInfoMap.remove(CLBHelper.HEALTH_CHECKER_KEYS.URL));
            }
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }
        handlerContext.setOutputValue("clbRefInfoMap", clbRefInfoMap);
    }
    
    
    @Handler(id="getCLBConfigInfo",
    input={
        @HandlerInput(name="lbName", type=String.class, required=true)
    },
    output={
        @HandlerOutput(name="clbConfigInfoMap", type=Map.class),
        @HandlerOutput(name="PolicyTypeOption", type=String.class)
    })
    public static void getCLBConfigInfo(HandlerContext handlerContext) {
        String clbName = (String) handlerContext.getInputValue("lbName");
        HashMap clbConfigInfoMap = new HashMap();
        try {
            clbConfigInfoMap = CLBHelper.getCLBConfigInfoMap(clbName);
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }
        handlerContext.setOutputValue("clbConfigInfoMap", clbConfigInfoMap);
        handlerContext.setOutputValue("PolicyTypeOption", clbConfigInfoMap.get("policyTypeOption"));
    }
    
    
    @Handler(id="saveCLBConfig",
    input={
        @HandlerInput(name="lbName", type=String.class, required=true),
        @HandlerInput(name="AddProps", type=Map.class),
        @HandlerInput(name="RemoveProps", type=List.class),
        @HandlerInput(name="PolicyType", type=String.class),
        @HandlerInput(name="clbConfigInfoMap", type=Map.class),
        @HandlerInput(name="UploadedFilePath", type=String.class)
    })
    public static void saveCLBConfigInfo(HandlerContext handlerContext) {
        String clbName = (String) handlerContext.getInputValue("lbName");
        HashMap clbConfigInfoMap = (HashMap) handlerContext.getInputValue("clbConfigInfoMap");
        try {
            ObjectName clbConfig = CLBHelper.getCLBConfig(clbName);
            if(clbConfig == null) {
                GuiUtil.handleError(handlerContext, "The associated config with the CLB " + clbName + 
                        "is not available!!");
                return;
            }
            ObjectName clbPolicy = (ObjectName) SipUtil.getChildByFunction(clbConfig, "getConvergedLbPolicy");
            if (clbPolicy == null) {
                clbPolicy = SipUtil.createChildByFunction(clbConfig, "createConvergedLbPlocy");
            }
            HashMap attrs = new HashMap();
            String policyType = (String) handlerContext.getInputValue("PolicyType");
            if (policyType.equals("httpsip")) {
                attrs.put(CLBHelper.CLB_POLICY_KEYS.HTTP, clbConfigInfoMap.get("httpPolicy"));
                String[] selectedSipPolicies = (String[]) clbConfigInfoMap.get("selectedSipPolicies");
                attrs.put(CLBHelper.CLB_POLICY_KEYS.SIP, GuiUtil.arrayToString(selectedSipPolicies, ","));
                // remove DCR policy if set.
                CLBHelper.unsetDCRFile(clbName);
                SipUtil.setAttributeValues(clbPolicy, attrs);
            } else {
                String filePath = (String)handlerContext.getInputValue("UploadedFilePath");
                String clbConfigName = clbConfig.getKeyProperty("name");
                CLBHelper.setDCRFile(filePath, clbConfigName, null);
            }
            HashMap addProps = (HashMap) handlerContext.getInputValue("AddProps");
            ArrayList removeProps = (ArrayList) handlerContext.getInputValue("RemoveProps");
            SipUtil.setProperties(clbPolicy, addProps, removeProps);
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }
    }
    
    
    @Handler(id="getCLBHostingTargets",
    input={
        @HandlerInput(name="lbName", type=String.class, required=true)
    },
    output={
        @HandlerOutput(name="isCluster", type=Boolean.class),
        @HandlerOutput(name="isInstance", type=Boolean.class),
        @HandlerOutput(name="result", type=List.class)
    })
    public static void getCLBHostingTargets(HandlerContext handlerContext) {
        String clbName = (String) handlerContext.getInputValue("lbName");
        ArrayList<String> targets = CLBHelper.getCLBHostingTargets(clbName);
        ArrayList result = new ArrayList();
        
        //Either all are clusters or none are clusters
        Boolean isCluster = TargetUtil.isCluster(targets.get(0));
        for (String targetName : targets) {
            HashMap oneRow = new HashMap();
            oneRow.put("targetName", targetName);
            oneRow.put("selected", false);
            if (isCluster) {
                oneRow.put("targetLink", "/cluster/clusterGeneral.jsf?clusterName="+targetName);
            } else {
                oneRow.put("targetLink", "/standalone/standaloneInstanceGeneral.jsf?instanceName="+targetName);
                oneRow.put("lbEnabled", false);
            }
            result.add(oneRow);
        }
        handlerContext.setOutputValue("result", result);
        handlerContext.setOutputValue("isCluster", isCluster);
        handlerContext.setOutputValue("isInstance", !isCluster);
    }
    
    @Handler(id="getCLBRefTargets",
    input={
        @HandlerInput(name="lbName", type=String.class, required=true)
    },
    output={
        @HandlerOutput(name="isCluster", type=Boolean.class),
        @HandlerOutput(name="isInstance", type=Boolean.class),
        @HandlerOutput(name="showCreateTargetLink", type=Boolean.class),
        @HandlerOutput(name="result", type=List.class)
    })
    public static void getCLBRefTargets(HandlerContext handlerContext) {
        String clbName = (String) handlerContext.getInputValue("lbName");
        ArrayList result = new ArrayList();
        Boolean isCluster = true;
        Boolean showCreateTargetLink = false;
        try {
            ArrayList<String> targets = CLBHelper.getCLBRefTargets(clbName);
            for (String targetName : targets) {
                isCluster = TargetUtil.isCluster(targetName);
                HashMap oneRow = new HashMap();
                oneRow.put("targetName", targetName);
                oneRow.put("selected", false);
                //oneRow.put("appLink", ApplicationHandlers.getNumberLBAppsByTarget(targetName));
                if (isCluster) {
                    oneRow.put("targetLink", "/cluster/clusterGeneral.jsf?clusterName="+targetName);
                    //oneRow.put("clusterInstanceLink", TargetUtil.getNumberLBInstancesByTarget(targetName));
                } else {
                    oneRow.put("targetLink", "/standalone/standaloneInstanceGeneral.jsf?instanceName="+targetName);
                    oneRow.put("lbEnabled", false);
                }
                result.add(oneRow);
            }
            showCreateTargetLink = (targets.size() == 0);
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }        
        handlerContext.setOutputValue("result", result);
        handlerContext.setOutputValue("isCluster", isCluster);
        handlerContext.setOutputValue("isInstance", !isCluster);
        handlerContext.setOutputValue("showCreateTargetLink", showCreateTargetLink);
    }
    
    
    @Handler(id="getManagebleHostingTargets",
    input={
        @HandlerInput(name="lbName", type=String.class, required=true)
    },
    output={
        @HandlerOutput(name="selectedTargets", type=String[].class),
        @HandlerOutput(name="availableTargets", type=Option[].class)
    })
    public static void getManagebleHostingTargets(HandlerContext handlerContext) {
        String clbName = (String) handlerContext.getInputValue("lbName");
        ArrayList availableTargets = CLBHelper.getTargetsWithoutCLB();
        ArrayList<String> targets = CLBHelper.getCLBHostingTargets(clbName);
        availableTargets.addAll(targets);
        
        Option[] availableOptions = null;
        if(availableTargets != null) {
            availableOptions = new Option[availableTargets.size()];
            List<String> strList = GuiUtil.convertListOfStrings(availableTargets);
            availableOptions = GuiUtil.getSunOptions(strList);
        }
        
        String selectedOptions[] = targets.toArray(new String[targets.size()]);
        
        handlerContext.setOutputValue("availableTargets", availableOptions);
        handlerContext.setOutputValue("selectedTargets", selectedOptions);
    }
    
    /** The handler is used to get a list of cluster/standalone instances that can be used as lb-refs in a 
     *	CLB configuration. The criteria for inclusion in valid list is that the instance should not have a 
     *	CLB on it.
     * @param handlerContext
     */
    @Handler(id="getManagebleRefTargets",
    input={
        @HandlerInput(name="lbName", type=String.class, required=true)
    },
    output={
        @HandlerOutput(name="selectedTargets", type=String[].class),
        @HandlerOutput(name="availableTargets", type=Option[].class),
        @HandlerOutput(name="clbRefTargetInfoMap", type=Map.class)
    })
    public static void getManagebleRefTargets(HandlerContext handlerContext) {
        String clbName = (String) handlerContext.getInputValue("lbName");
        Option[] availableOptions = null;
        String[] selectedOptions = null;
        HashMap clbRefTargetInfoMap = new HashMap();
        try {
            ArrayList<String> availableTargets = CLBHelper.getTargetsWithoutCLB();
            ArrayList<String> selectedTargets = CLBHelper.getCLBRefTargets(clbName);
            if(selectedTargets.size() > 0) {
                clbRefTargetInfoMap = CLBHelper.getCLBRefInfo(clbName, (String) selectedTargets.get(0));
            }
	    if (availableTargets.size() == 0) {
		clbRefTargetInfoMap.put("noAvailableLBTargets", true);
	    }
	    
	    // Add the selectedTarget into the list as the availableTargets will not have these if this is
	    // self load balanced CLB
	    for (String targetName : selectedTargets) {
		if (!availableTargets.contains(targetName)) {
		    availableTargets.add(targetName);
		}
	    }
	    
	    //Get the targets as these will not be listed by getTargetsWithoutCLB
	    ArrayList<String> targets = CLBHelper.getCLBHostingTargets(clbName);
	    String targetString = "";
	    for (String targetName : targets) {
		if (!availableTargets.contains(targetName)) {
		    availableTargets.add(targetName);
		}
		targetString += targetName + ",";
	    }
	    targetString = targetString.substring(0, targetString.length()-1);
	    clbRefTargetInfoMap.put("targets", targetString);
	    
            if(availableTargets.size() > 0) {
                availableOptions = new Option[availableTargets.size()];
                List<String> strList = GuiUtil.convertListOfStrings(availableTargets);
                availableOptions = GuiUtil.getSunOptions(strList);
            }

            selectedOptions = selectedTargets.toArray(new String[selectedTargets.size()]);
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }
        
        handlerContext.setOutputValue("availableTargets", availableOptions);
        handlerContext.setOutputValue("selectedTargets", selectedOptions);
        handlerContext.setOutputValue("clbRefTargetInfoMap", clbRefTargetInfoMap);
    }
    
    
    @Handler(id="createCLBHostingTargets",
    input={
        @HandlerInput(name="lbName", type=String.class, required=true),
        @HandlerInput(name="selectedTargets", type=String[].class)
    },
    output={
        @HandlerOutput(name="nextPage", type=String.class)
    })
    public static void createCLBHostingTargets(HandlerContext handlerContext) {
        String clbName = (String) handlerContext.getInputValue("lbName");
        String nextPage = "/sip/clb/clbHostingTargets.jsf";
        
        try {
            ArrayList<String> associatedTargets = CLBHelper.getCLBHostingTargets(clbName);
            List<String> selectedTargets = Arrays.asList((String[])
                handlerContext.getInputValue("selectedTargets"));
            
            HashMap clbInfo = CLBHelper.getCLBInfo(associatedTargets.get(0));
            //remove the targets which are no longer selected
	    // This will remove all targets if none are selected which means the CLB itself would be deleted
	    // along with the associted CLB configs.
            for (String targetName: associatedTargets) {
                if(!selectedTargets.contains(targetName)) {
                    CLBHelper.deleteCLB(targetName);
                }
            }
            
            for (String targetName: selectedTargets) {
                if(!associatedTargets.contains(targetName)) {
                    // create a basic LB
                    CLBHelper.copyCLB(clbInfo, targetName);
                }
            }
            
            if (selectedTargets.size() == 0) {
                // The user has selected to delete all the targets. This means that the CLB it self would 
                // be deleted.
                nextPage = "/sip/clb/clbs.jsf";
            }
            
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }
        handlerContext.setOutputValue("nextPage", nextPage);
    }
    
    
    @Handler(id="createCLBRefTargets",
    input={
        @HandlerInput(name="lbName", type=String.class, required=true),
        @HandlerInput(name="selectedTargets", type=String[].class),
        @HandlerInput(name="clbRefTargetInfoMap", type=Map.class)
    })
    public static void createCLBRefTargets(HandlerContext handlerContext) {
        String clbName = (String) handlerContext.getInputValue("lbName");
        try {
            ArrayList<String> associatedTargets = CLBHelper.getCLBRefTargets(clbName);
            List<String> selectedTargets = Arrays.asList((String[]) 
                    handlerContext.getInputValue("selectedTargets"));
            Boolean isCluster = true;
            
            //Make sure the selected targets are the same type
            if (selectedTargets.size()>0){
                isCluster = TargetUtil.isCluster(selectedTargets.get(0));
                for(String sel: selectedTargets){
                    AMXConfig sc = null;
                    if(isCluster)
                        sc = AMXUtil.getDomainConfig().getClusterConfigMap().get(sel);
                    else
                        sc = AMXUtil.getDomainConfig().getStandaloneServerConfigMap().get(sel);
                    if (sc == null){
                        GuiUtil.handleError(handlerContext, GuiUtil.getMessage("loadBalancer.targetHelp"));
                        return;
                    }
                }
            }
            
            // Check for Self LB targets
            HashMap clbRefTargetInfoMap = (HashMap) handlerContext.getInputValue("clbRefTargetInfoMap");
            if(isCluster) {
                Boolean selfLB = (Boolean) clbRefTargetInfoMap.get("selfLoadBalancing");
                if (selfLB == null) {
                    selfLB = false;
                }
                if(selfLB && (selectedTargets.size() > 1)) {
                    String message = "Self Load Balancing CLB can only have a single cluster target. " + 
                            "Please select a single cluster target only or make sure Self Load Balancing" + 
                            " is unchecked.";
                    GuiUtil.handleError(handlerContext, message);
                    return;
                }
            }
            
            if (associatedTargets.size() > 0) {
                // all targets have to be same
                isCluster = TargetUtil.isCluster(associatedTargets.get(0));
            }
            
            if (selectedTargets.size() == 0) {
                // remove all target refs from the config
                for(String targetName : associatedTargets) {
                    CLBHelper.deleteCLBRef(clbName, targetName);
                }
            } else {
                // delete the targets that have been removed
                for (String targetName: associatedTargets) {
                    if(!selectedTargets.contains(targetName)) {
                        CLBHelper.deleteCLBRef(clbName, targetName);
                    }
                }
                
                //Create the targets that have been added new
                if (clbRefTargetInfoMap == null) {
                    clbRefTargetInfoMap = new HashMap();
                }
                
                for (String targetName: selectedTargets) {
                    if(!associatedTargets.contains(targetName)) {
                        // create a basic LB
                        CLBHelper.createCLBRef(clbName, targetName, clbRefTargetInfoMap);
                    }
                }
            }
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }
        
    }
        
        
    @Handler(id="saveCLBRefTargetDetails",
    input={
        @HandlerInput(name="lbName", type=String.class, required=true),
        @HandlerInput(name="targetName", type=String.class, required=true),
        @HandlerInput(name="clbRefInfoMap", type=Map.class)
    })
    public static void saveCLBRefTargetDetails(HandlerContext handlerContext) {
        String clbName = (String) handlerContext.getInputValue("lbName");
        String targetName = (String) handlerContext.getInputValue("targetName");
        try {
            HashMap clbRefInfoMap = (HashMap) handlerContext.getInputValue("clbRefInfoMap");
            boolean isCluster = TargetUtil.isCluster(targetName);
            ObjectName clbConfig = CLBHelper.getCLBConfig(clbName);
            if(isCluster) {
                ObjectName clusterRef = (ObjectName) SipUtil.getChildByFunction(clbConfig, 
                        "getConvergedLbClusterRefByRef", new Object[]{targetName},
                        new String[]{"java.lang.String"});
                HashMap props = new HashMap();
		Object selfLB = clbRefInfoMap.get("selfLoadBalancing");
		if (selfLB != null) {
		    props.put(CLBHelper.CLB_CLUSTER_REF_KEYS.SELF_LOADBALANCE, Boolean.parseBoolean(
			    selfLB.toString()));
		} else {
		    props.put(CLBHelper.CLB_CLUSTER_REF_KEYS.SELF_LOADBALANCE, false);
		}
                
                SipUtil.setAttributeValues(clusterRef, props);
            } else {
                ObjectName serverRef = (ObjectName) SipUtil.getChildByFunction(clbConfig, 
                        "getServerRefByRef", new Object[]{targetName},
                        new String[]{"java.lang.String"});
                HashMap props = new HashMap();
                props.put(CLBHelper.SERVER_REF_KEYS.DISABLE_TIMEOUT_IN_MINUTES,
                        clbRefInfoMap.get("disableTimeoutInMinutes"));
                /*props.put("enabled", clbRefInfoMap.get("enabled"));
                props.put("lb-enabled", clbRefInfoMap.get("lbEnabled"));*/
                SipUtil.setAttributeValues(serverRef, props);
                
                ObjectName healthChecker = (ObjectName) SipUtil.getChildByFunction(serverRef, "getHealthChecker");
                if (healthChecker == null) {
                    healthChecker = SipUtil.createChildByFunction(serverRef, "createHealthChecker");
                }
                props.clear();
                props.put(CLBHelper.HEALTH_CHECKER_KEYS.INTERVAL_IN_SECONDS, 
                        clbRefInfoMap.get("intervalInSeconds"));
                props.put(CLBHelper.HEALTH_CHECKER_KEYS.TIMEOUT_IN_SECONDS, 
                        clbRefInfoMap.get("timeoutInSeconds"));
                props.put(CLBHelper.HEALTH_CHECKER_KEYS.URL, clbRefInfoMap.get("url"));
                SipUtil.setAttributeValues(healthChecker, props);
            }
        } catch (Exception ex) {
            GuiUtil.handleException(handlerContext, ex);
        }
    }
    
    @Handler(id="uploadFile",
    input={
        @HandlerInput(name="file", type=UploadedFile.class, required=true)
    },
    output={
        @HandlerOutput(name="origPath", type=String.class),
        @HandlerOutput(name="uploadedTempFile", type=String.class)
    })
    public static void uploadFile(HandlerContext handlerCtx) {
        UploadedFile uploadedFile = (UploadedFile)handlerCtx.getInputValue("file");
        String uploadTmpFile = "";
        
        if(uploadedFile != null) {
            String name = uploadedFile.getOriginalName();
            //see bug# 6498910, for IE, getOriginalName() returns the full path, including the drive.
            //for any other browser, it just returns the file name.
            int lastIndex = name.lastIndexOf("\\");
            if (lastIndex != -1){
                name = name.substring(lastIndex+1, name.length());
            }
            handlerCtx.setOutputValue("origPath", name);
            File target = new File(TargetUtil.getDomainRoot() + File.separator + "config" + File.separator + name);
            try {
                uploadedFile.write(target);
                uploadTmpFile = target.getCanonicalPath();
            } catch (IOException ioex) {
                try {
                    uploadTmpFile = target.getAbsolutePath();
                } catch (Exception ex) {
                    //Handle AbsolutePathException here
                }
            } catch (Exception ex) {
                GuiUtil.handleException(handlerCtx, ex);
            }
        }
        handlerCtx.setOutputValue("uploadedTempFile", uploadTmpFile);
    }
}
