/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.jvnet.glassfish.comms.clb.core;

import java.util.logging.Logger;
import org.jvnet.glassfish.comms.clb.core.common.chr.ConsistentHashRouter;
import org.jvnet.glassfish.comms.clb.core.common.chr.StickyConsistentHashRouter;
import org.jvnet.glassfish.comms.util.LogUtil;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;

import java.util.HashMap;
import java.util.List;
import java.util.logging.Level;


/**
 * Factory class for creating router. The router are created based on load-balancing policy
 * as well as clusters part of this router.
 *
 * @author kshitiz
 */
public class RouterFactory {
    private static final Logger _logger = LogUtil.CLB_LOGGER.getLogger();

    //consistent hash load-balancing policy name
    public static final String LBPolicy_CONSISTENT_HASH = "consistent-hash";

    //consistent hash load-balancing policy name
    public static final String LBPolicy_STICKY_CONSISTENT_HASH = "sticky-consistent-hash";

    //round robin load-balancing policy name
    public static final String LBPolicy_ROUND_ROBIN = "round-robin";

    //round robin load-balancing policy name
    public static final String LBPolicy_STICKY_ROUND_ROBIN = "sticky-round-robin";

    //map maintaining a router for each load balancing policy
    private static final HashMap<String, String> lbPolicyRouterMap = new HashMap<String, String>();

    static {
        //adding consistent hash routers to the list of router
        lbPolicyRouterMap.put(LBPolicy_CONSISTENT_HASH,
            ConsistentHashRouter.class.getName());
        lbPolicyRouterMap.put(LBPolicy_STICKY_CONSISTENT_HASH,
            StickyConsistentHashRouter.class.getName());

        //adding round-robin routers to the list of router
        lbPolicyRouterMap.put(LBPolicy_ROUND_ROBIN,
            RoundRobinRouter.class.getName());
        lbPolicyRouterMap.put(LBPolicy_STICKY_ROUND_ROBIN,
            StickyRoundRobinRouter.class.getName());
    }

    /**
     * Creates the router based on load-balancing policy and list of cluster provided. It can
     * create a new router at every call, or it can share a router created with given load-balancing
     * policy and list of clusters. Also there is a flag to indicate whether router need to route
     * to only active instances
     *
     * In general when router is created for RequestGroup, it need to cater to failover scenarios as well,
     * thus this flag is false.
     * However for a ServerCluster, it need to only route request to active instances within cluster. Thus
     * this flag is true in this case.
     *
     * @param lbpolicy the load-balancing policy
     * @param clusters a list of clusters
     * @param activeRouting flag to indicate whether router should only route to active instances only
     *
     * @return the router
     */
    public static Router createRouter(String lbpolicy,
        List<ServerCluster> clusters, boolean activeRouting)
        throws CLBRuntimeException {
        String className = lbPolicyRouterMap.get(lbpolicy);

        if (className == null) {
            //Log severe error message
            return null;
        }

        try {
            Class routerClass = Class.forName(className);
            Constructor routerConstructor = routerClass.getConstructor(List.class,
                    Boolean.TYPE);
            Router router = (Router) routerConstructor.newInstance(clusters,
                    activeRouting);

            return router;
        } catch (IllegalArgumentException ex) {
            String message = _logger.getResourceBundle().
                    getString("clb.unable_to_instantiate_router") + ex.getMessage();
            _logger.log(Level.SEVERE, message);
            throw new CLBRuntimeException(message, ex);
        } catch (InvocationTargetException ex) {
            String message = _logger.getResourceBundle().
                    getString("clb.unable_to_instantiate_router") + ex.getMessage();
            _logger.log(Level.SEVERE, message);
            throw new CLBRuntimeException(message, ex);
        } catch (IllegalAccessException ex) {
            String message = _logger.getResourceBundle().
                    getString("clb.unable_to_instantiate_router") + ex.getMessage();
            _logger.log(Level.SEVERE, message);
            throw new CLBRuntimeException(message, ex);
        } catch (InstantiationException ex) {
            String message = _logger.getResourceBundle().
                    getString("clb.unable_to_instantiate_router") + ex.getMessage();
            _logger.log(Level.SEVERE, message);
            throw new CLBRuntimeException(message, ex);
        } catch (SecurityException ex) {
            String message = _logger.getResourceBundle().
                    getString("clb.unable_to_instantiate_router") + ex.getMessage();
            _logger.log(Level.SEVERE, message);
            throw new CLBRuntimeException(message, ex);
        } catch (NoSuchMethodException ex) {
            String message = _logger.getResourceBundle().
                    getString("clb.unable_to_instantiate_router") + ex.getMessage();
            _logger.log(Level.SEVERE, message);
            throw new CLBRuntimeException(message, ex);
        } catch (ClassNotFoundException ex) {
            String message = _logger.getResourceBundle().
                    getString("clb.unable_to_instantiate_router") + ex.getMessage();
            _logger.log(Level.SEVERE, message);
            throw new CLBRuntimeException(message, ex);
        }
    }
}
