/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.jvnet.glassfish.comms.clb.core;

import java.util.logging.Logger;
import org.jvnet.glassfish.comms.util.LogUtil;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.logging.Level;


/**
 * Implementation of interface ServerCluster
 * @author kshitiz
 */
public class ServerCluster {
    private static final Logger _logger = LogUtil.CLB_LOGGER.getLogger();

    /* name of the cluster */
    private String name;

    /* map of server instance to server name */
    private HashMap<String, ServerInstance> instanceMap;

    /* router associated with this cluster */
    private Router router;

    /* flag indicating whether this is a self load-balancing cluster */
    private boolean selfLoadbalance;

    /** Creates a new instance of ServerClusterImple */
    public ServerCluster(String clusterName, boolean selfLoadbalance) {
        this.name = clusterName;
        this.selfLoadbalance = selfLoadbalance;
        instanceMap = new HashMap<String, ServerInstance>();
    }

    public boolean enableCluster() {
        throw new UnsupportedOperationException();
    }

    public boolean disableCluster() {
        throw new UnsupportedOperationException();
    }

    public boolean addInstance(ServerInstance instance) {
        instanceMap.put(instance.getName(), instance);

        return true;
    }

    public boolean deleteInstance(ServerInstance instance) {
        return (instanceMap.remove(instance) != null);
    }

    public boolean enableInstance(ServerInstance instance) {
        return instance.enableInstance();
    }

    public boolean disableInstance(ServerInstance instance) {
        return instance.disableInstance();
    }

    public Router getClusterRouter() {
        return router;
    }

    public void initializeClusterRouter(String lbpolicy, Controller controller)
        throws CLBRuntimeException {
        if(_logger.isLoggable(Level.FINE))
            _logger.log(Level.FINE,
                    "clb.initializing_cluster_router",
                    new Object[]{name});

        ArrayList<ServerCluster> clusterList = new ArrayList<ServerCluster>();
        clusterList.add(this);

        if (controller.getLBType() == CLBConstants.HTTP_CLB) {
            router = RouterFactory.createRouter(lbpolicy, clusterList, true);
            router.setController(controller);
        } else {
            router = RouterFactory.createRouter(RouterFactory.LBPolicy_CONSISTENT_HASH,
                    clusterList, true);
            router.setController(controller);
        }

        router.initialize();
        if(_logger.isLoggable(Level.FINE))
            _logger.log(Level.FINE,
                    "clb.initialized_cluster_router",
                    new Object[]{router, lbpolicy, name});
    }

    public void initialize(int lbType) throws CLBRuntimeException {
        if(_logger.isLoggable(Level.FINE))
            _logger.log(Level.FINE,
                    "clb.initializing_the_cluster",
                    new Object[]{name});

        for (ServerInstance instance : getAllInstances())
            instance.initialize(lbType);

        if(_logger.isLoggable(Level.FINE))
            _logger.log(Level.FINE,
                    "clb.initialized_the_cluster",
                    new Object[]{name});
    }

    public void cleanup() {
        if(_logger.isLoggable(Level.FINE))
            _logger.log(Level.FINE,
                    "clb.cleaning_up_the_cluster",
                    new Object[]{name});

        for (ServerInstance instance : getAllInstances())
            instance.cleanup();

        instanceMap = null;
        router = null;
        if(_logger.isLoggable(Level.FINE))
            _logger.log(Level.FINE,
                    "clb.cleaned_up_the_cluster",
                    new Object[]{name});
    }

    public void reinitializeClusterRouter() {
        throw new UnsupportedOperationException();
    }

    public String getName() {
        return name;
    }

    public void addInstance(String instanceName, boolean enabled,
        int timeoutInMinutes, String[] listeners) throws CLBRuntimeException {
        if(_logger.isLoggable(Level.FINE))
            _logger.log(Level.FINE,
                    "clb.adding_instance_to_the_cluster",
                    new Object[]{instanceName, name});

        if (getInstance(instanceName) != null) {
            String message = _logger.getResourceBundle().getString(
                    "clb.cannot_add_duplicate_instance_to_cluster") + instanceName;
            throw new CLBRuntimeException(message);
        }

        ServerInstance instance = new ServerInstance(this, instanceName,
                enabled, timeoutInMinutes);
        instance.addEndPoints(listeners);
        addInstance(instance);
    }

    public ServerInstance getInstance(String instanceName) {
        return instanceMap.get(instanceName);
    }

    public Collection<ServerInstance> getAllInstances() {
        return instanceMap.values();
    }

    public boolean isSelfLoadbalancing() {
        return selfLoadbalance;
    }

    public void markInstancesHealthy(List<String> instanceList) {
        for (String instanceName : instanceList) {
            ServerInstance instance = instanceMap.get(instanceName);
            if (instance == null) {
                _logger.log(Level.SEVERE,
                    "clb.healthy_instance_not_found_in_cluster_list",
                    new Object[]{instanceName, name});
                continue;
            }
            if(_logger.isLoggable(Level.FINE))
                _logger.log(Level.FINE,
                        "clb.marking_instance_in_cluster_as_healthy",
                        new Object[]{instanceName, name});
            instance.markAsHealthy();
        }
    }

    public void reconfig(ServerCluster oldCluster) {
        if(_logger.isLoggable(Level.FINE))
            _logger.log(Level.FINE,
                    "clb.reconfiguring_the_cluster",
                    new Object[]{name});

        Collection<ServerInstance> oldInstances = oldCluster.getAllInstances();

        for (ServerInstance oldInstance : oldInstances) {
            ServerInstance newInstance = getInstance(oldInstance.getName());

            if (newInstance != null) {
                newInstance.reconfig(oldInstance);
            }
        }

        if(_logger.isLoggable(Level.FINE))
            _logger.log(Level.FINE,
                    "clb.reconfigured_the_cluster",
                    new Object[]{name});
    }
}
