/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.jvnet.glassfish.comms.clb.core;

import java.util.logging.Logger;
import org.jvnet.glassfish.comms.clb.admin.CLBConfigurator;
import org.jvnet.glassfish.comms.clb.core.util.IDGenerator;
import org.jvnet.glassfish.comms.util.LogUtil;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.logging.Level;


/**
 * Implementation of interface ServerInstance
 * @author kshitiz
 */
public class ServerInstance {
    private static final Logger _logger = LogUtil.CLB_LOGGER.getLogger();

    /* name of the instance*/
    private String name;

    /* unique identity of this instance*/
    private String id;

    /* cluster with which this instance is associated with */
    private ServerCluster cluster;

    /* field indicating whether this instance is enabled or not */
    private boolean enabled;

    /* field indicating whether this instance is healthy or not */
    private boolean healthy;

    /* timeout in minutes */
    private int timeoutInMinutes;

    /* Map maintaining endpoints of the instance per protocol */
    private HashMap<String, ArrayList<EndPoint>> endPointProtocolMap;

    /* Index of endpoint returned on last call to getEndPoint(..)  */
    private HashMap<String, IndexCount> protocolEndPointIndex;

    /* Routers listening to event on this instance */
    private ArrayList<Router> routers;

    /* flag indicating whether instance is quiesced or not */
    private boolean quiesce;

    /* thread to take care of quiescing */
    private QuiesceThread quiesceThread;

    /* thread to take care of quiesce thread lock */
    private Object quiesceThreadLock;

    /* flag to mark that quiesce thread need to be started at the time of initialization */
    private boolean startQuiesceThread;

    /** Creates a new instance of ServerInstanceImpl */
    public ServerInstance(ServerCluster cluster, String instanceName,
        boolean enabled, int timeoutInMinutes) {
        this.cluster = cluster;
        this.name = instanceName;
        this.enabled = enabled;
        this.timeoutInMinutes = timeoutInMinutes;
        healthy = false;
        endPointProtocolMap = new HashMap<String, ArrayList<EndPoint>>();
        routers = new ArrayList<Router>();
        quiesce = !enabled;
    }

    public synchronized boolean enableInstance() {
        if(enabled)
            return true;
        
        enabled = true;
        stopQuiesceThread();

        for (Router router : routers)
            router.handleEnableEvent(this);

        return true;
    }

    public synchronized boolean disableInstance() {
        if(!enabled)
            return true;
        
        enabled = false;

        for (Router router : routers)
            router.handleDisableEvent(this);

        startQuiesceThread();

        return true;
    }

    public synchronized void markAsUnhealthy() {
        if(!healthy)
            return;
        
        healthy = false;

        for (Router router : routers) {
            if(_logger.isLoggable(Level.FINE))
                _logger.log(Level.FINE,
                        "clb.sending_instance_failure_event_to_router",
                        new Object[]{name, router});
            router.handleFailureEvent(this);
        }
    }

    public synchronized void markAsHealthy() {
        if(healthy)
            return;
        
        healthy = true;

        for (Router router : routers) {
            if(_logger.isLoggable(Level.FINE))
                _logger.log(Level.FINE,
                        "clb.sending_instance_recovery_event_to_router",
                        new Object[]{name, router});
            router.handleRecoveryEvent(this);
        }
    }

    public boolean addEndPoint(EndPoint endPoint) {
        ArrayList<EndPoint> endPointList = endPointProtocolMap.get(endPoint.getProtocol());

        if (endPointList == null) {
            endPointList = new ArrayList<EndPoint>();
            endPointProtocolMap.put(endPoint.getProtocol(), endPointList);
        }
        
        return endPointList.add(endPoint);
    }

    public boolean deleteEndPoint(EndPoint endPoint) {
        ArrayList<EndPoint> endPointList = endPointProtocolMap.get(endPoint.getProtocol());

        if (endPointList == null) {
            return false;
        }

        return endPointList.remove(endPoint);
    }

    public boolean enableEndPoint(EndPoint endPoint) {
        return endPoint.enableEndPoint();
    }

    public boolean disableEndPoint(EndPoint endPoint) {
        return endPoint.disableEndPoint();
    }

    public ServerCluster getServerCluster() {
        return cluster;
    }

    public EndPoint getEndPoint(String protocol) {
        ArrayList<EndPoint> endPointList = endPointProtocolMap.get(protocol);

        if (endPointList.size() < 1) {
            return null;
        }

        if (endPointList.size() == 1) {
            return endPointList.get(0);
        }

        return endPointList.get(
            protocolEndPointIndex.get(protocol).incrementAndGet());
    }

    public boolean addRouter(Router router) {
        return routers.add(router);
    }

    public String getName() {
        return name;
    }

    public boolean isEnabled() {
        return enabled;
    }

    public boolean isDisabled() {
        return !enabled;
    }

    public boolean isHealthy() {
        return healthy;
    }

    public boolean isQuiesced() {
        return quiesce;
    }

    public void addEndPoints(String[] listeners) throws CLBRuntimeException {
        //flag to indicate whether this is a local instance
        boolean local = false;

        //It can be local instance only in case of self-loadbalancing cluster
        if (CLBConfigurator.getInstance().isSelfLoadbalanced()) {
            local = isLocalInstance();
        }

        for (int i = 0; i < listeners.length; i++) {
            String listener = listeners[i].trim();
            String protocol;

            if (listener.startsWith(CLBConstants.HTTP_PROTOCOL)) {
                protocol = CLBConstants.HTTP_PROTOCOL;
            } else if (listener.startsWith(CLBConstants.HTTPS_PROTOCOL)) {
                protocol = CLBConstants.HTTPS_PROTOCOL;
            } else if (listener.startsWith(CLBConstants.SIP_PROTOCOL)) {
                protocol = CLBConstants.SIP_PROTOCOL;
            } else if (listener.startsWith(CLBConstants.SIPS_PROTOCOL)) {
                protocol = CLBConstants.SIPS_PROTOCOL;
            } else if (listener.indexOf("://") == -1) {
                //protocol is not specified
                //mark it as http listener
                protocol = CLBConstants.HTTP_PROTOCOL;
            } else {
                String message = _logger.getResourceBundle().
                        getString("clb.unknown_listener_protocol") + listener;
                throw new CLBRuntimeException(message);
            }

            if (listener.indexOf(protocol) > -1) {
                listener = listener.substring(protocol.length());
            }

            if (listener.length() == 0) {
                String message = _logger.getResourceBundle().
                        getString("clb.host_name_port_number_not_specified_for_listener") + listener;
                throw new CLBRuntimeException(message);
            }

            String hostName = null;
            int port = -1;
            int index;

            if ((index = listener.indexOf(":")) > -1) {
                hostName = listener.substring(0, index);

                try {
                    port = Integer.valueOf(listener.substring(index + 1));
                } catch (NumberFormatException ex) {
                    String message = _logger.getResourceBundle().
                        getString("clb.specified_port_is_not_numeric") + listener.substring(index + 1);
                    throw new CLBRuntimeException(message, ex);
                }
            } else {
                hostName = listener;

                //Since port number is not specified
                //take default port for that protocol
                if (protocol.equals(CLBConstants.HTTP_PROTOCOL)) {
                    port = CLBConstants.DEFAULT_HTTP_PORT;
                } else if (protocol.equals(CLBConstants.HTTPS_PROTOCOL)) {
                    port = CLBConstants.DEFAULT_HTTPS_PORT;
                } else if (protocol.equals(CLBConstants.SIP_PROTOCOL)) {
                    port = CLBConstants.DEFAULT_SIP_PORT;
                } else {
                    port = CLBConstants.DEFAULT_SIPS_PORT;
                }
            }

            EndPoint endPoint = new EndPoint(this, protocol, hostName, port,
                    local);
            addEndPoint(endPoint);
        }
    }

    public String getID() {
        return id;
    }

    private void generateID(int lbType) throws CLBRuntimeException {
        ArrayList<EndPoint> endPointList;
        EndPoint endPoint = null;
        boolean secure = false;

        if (lbType == CLBConstants.SIP_CLB) {
            endPointList = endPointProtocolMap.get(CLBConstants.SIP_PROTOCOL);

            if ((endPointList != null) && (endPointList.size() > 0)) {
                endPoint = endPointList.get(0);
            } else {
                endPointList = endPointProtocolMap.get(CLBConstants.SIPS_PROTOCOL);

                if ((endPointList != null) && (endPointList.size() > 0)) {
                    endPoint = endPointList.get(0);
                    secure = true;
                }
            }
        } else {
            endPointList = endPointProtocolMap.get(CLBConstants.HTTP_PROTOCOL);

            if ((endPointList != null) && (endPointList.size() > 0)) {
                endPoint = endPointList.get(0);
            } else {
                endPointList = endPointProtocolMap.get(CLBConstants.HTTPS_PROTOCOL);

                if ((endPointList != null) && (endPointList.size() > 0)) {
                    endPoint = endPointList.get(0);
                    secure = true;
                }
            }
        }

        if (endPoint == null) {
            String message = _logger.getResourceBundle().
                        getString("clb.no_endpoints_found") + name;
            throw new CLBRuntimeException(message);
        }

        id = IDGenerator.getId(endPoint.getHost(), endPoint.getPort(), secure);
    }

    public void reconfig(ServerInstance oldInstance) {
        if ((this.isDisabled() && oldInstance.isEnabled()) ||
                (oldInstance.isDisabled() && !oldInstance.isQuiesced() &&
                this.isDisabled())) {
            startQuiesceThread = true;
            quiesce = false;
        }
    }

    public void initialize(int lbType) throws CLBRuntimeException {
        generateID(lbType);
        
        Iterator<String> protocolEndPointIndexKeys = 
            endPointProtocolMap.keySet().iterator();
        while(protocolEndPointIndexKeys.hasNext()){
            String protocol = protocolEndPointIndexKeys.next();
            ArrayList<EndPoint> endPointList = endPointProtocolMap.get(protocol);
            if(endPointList.size() > 1){
                //Since there are more than one end point, need to initialize protocolEndPointIndex
                if (protocolEndPointIndex == null) {
                    protocolEndPointIndex = new HashMap<String, IndexCount>();
                }
                protocolEndPointIndex.put(protocol,
                    new IndexCount(0, endPointList.size()));
            }
        }

        if (startQuiesceThread) {
            startQuiesceThread();
        }
    }

    public void cleanup() {
        stopQuiesceThread();
        cluster = null;
        endPointProtocolMap = null;
        protocolEndPointIndex = null;
        routers = null;
        quiesceThreadLock = null;
    }

    private void stopQuiesceThread() {
        if (quiesceThreadLock != null) {
            synchronized (quiesceThreadLock) {
                if (quiesceThread != null) {
                    quiesceThread.killThread();
                    quiesceThread.interrupt();
                    waitForQuiesceThreadCompletion();
                    quiesceThread = null;
                }
            }
        }
    }

    private void startQuiesceThread() {
        if (quiesceThreadLock == null) {
            quiesceThreadLock = new Object();
        }

        synchronized (quiesceThreadLock) {
            quiesceThread = new QuiesceThread(timeoutInMinutes);
            quiesceThread.start();
        }
    }

    private void waitForQuiesceThreadCompletion() {
        synchronized (quiesceThreadLock) {
            while (quiesceThread != null) {
                try {
                    quiesceThreadLock.wait();
                } catch (InterruptedException ex) {
                    if(_logger.isLoggable(Level.FINE))
                        _logger.log(Level.FINE,
                                "clb.caught_an_exception",
                                new Object[]{ex.getMessage()});
                    if(_logger.isLoggable(Level.FINEST))
                        _logger.log(Level.FINEST,
                                "",
                                ex);
                }
            }
        }
    }

    /**
     * Method to check whether it is a local instance or a remote instance
     * Check is based on comparing the instance name
     */
    private boolean isLocalInstance() {
        String instanceName = com.sun.enterprise.server.ondemand.OnDemandServer.getServerContext()
                                                                               .getInstanceName();

        if (instanceName.equals(name)) {
            if(_logger.isLoggable(Level.FINE))
                _logger.log(Level.FINE,
                        "clb.a_local_instance",
                        new Object[]{instanceName, name});
            return true;
        } else {
            if(_logger.isLoggable(Level.FINE))
                _logger.log(Level.FINE,
                        "clb.a_remote_instance",
                        new Object[]{instanceName, name});
            return false;
        }
    }

    class QuiesceThread extends Thread {
        private long timeout;
        private long startTime;
        private boolean killThread;

        QuiesceThread(long timeoutInMinutes) {
            this.timeout = timeoutInMinutes * 60 * 1000;
        }

        public void run() {
            startTime = System.currentTimeMillis();

            long waitTime = timeout;

            while (true) {
                try {
                    Thread.sleep(waitTime);

                    break;
                } catch (InterruptedException ex) {
                    if (killThread) {
                        break;
                    }

                    if(_logger.isLoggable(Level.FINE))
                        _logger.log(Level.FINE,
                                "clb.caught_an_exception",
                                new Object[]{ex.getMessage()});
                    if(_logger.isLoggable(Level.FINEST))
                        _logger.log(Level.FINEST,
                                "",
                                ex);
                    waitTime = timeout -
                        (System.currentTimeMillis() - startTime);

                    if (waitTime <= 0) {
                        break;
                    }
                }
            }

            if (_logger.isLoggable(Level.FINE)) {
                _logger.log(Level.INFO, 
                    "clb.quiescing_instance",
                    new Object[]{timeoutInMinutes, name});
            }
            quiesce = true;
        }

        public long getTimeLeft() {
            return (timeout - (System.currentTimeMillis() - startTime));
        }

        private void killThread() {
            killThread = true;
        }
    }
    
    class IndexCount {
        
        int currentValue;
        int maxValue;
        
        IndexCount(int initValue, int maxValue){
            currentValue = initValue;
            this.maxValue = maxValue;
        }
        
        private synchronized int incrementAndGet(){
            currentValue = (currentValue + 1) % maxValue;
            return currentValue;
        }
    }
}
