/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.jvnet.glassfish.comms.clb.core.common.chr;

import com.ericsson.ssa.sip.SipFactoryImpl;
import com.ericsson.ssa.sip.dns.DnsResolver;

import org.jvnet.glassfish.comms.clb.core.ConsistentHashRequest;
import org.jvnet.glassfish.comms.clb.core.Controller;
import org.jvnet.glassfish.comms.clb.core.DCRFileUpdateEventListener;
import org.jvnet.glassfish.comms.clb.core.Router;
import org.jvnet.glassfish.comms.clb.core.ServerCluster;
import org.jvnet.glassfish.comms.clb.core.ServerInstance;
import org.jvnet.glassfish.comms.clb.core.common.chr.dcr.DcrConfigurableHashKeyExtractor;
import org.jvnet.glassfish.comms.clb.core.common.chr.dcr.DcrRulesException;
import org.jvnet.glassfish.comms.clb.core.common.chr.dcr.DcrUtils;
import org.jvnet.glassfish.comms.clb.core.util.ConsistentHash;
import org.jvnet.glassfish.comms.clb.core.util.LoadbalancerUtil;
import org.jvnet.glassfish.comms.clb.proxy.http.util.HttpRequest;
import org.jvnet.glassfish.comms.util.LogUtil;

import java.io.File;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;


/**
 * This is a router (common for HTTP and SIP) that implements the logic for
 * server instance selection based on DCR. It can be used both for cluster (then
 * it shall be set up by calling {@link #setClusterRouter(boolean)} with the
 * argument true) and for request groups (default).
 * <p>
 * Server selection is performed in several steps depending of configuration:
 *
 * <pre>
 * Extract hash key from request using configurable hash key extractor
 * IF NOT configured as a cluster router:
 *   First select server using a consistent hash which includes all servers in all
 *     clusters (and assumes that they are all healthy): the &quot;Consistent Hash for
 *     the Ideal Configuration&quot;
 *   Check that selected server is healthy, and if it is use it.
 *   Otherwise (if the selected server is faulty), select server using the &quot;Cluster
 *   Router&quot; of the cluster of the failed server
 * ENDIF
 * If no server was found above, select server using a router which contains only
 * the healthy servers of all clusters: the &quot;Consistent hash for the Actual
 * Configuration&quot;
 * Finally, if no servers were found at all return null.
 * </pre>
 */
public class ConsistentHashRouter implements Router, DCRFileUpdateEventListener {
    /** Parameter used to setup consistent hashes */
    private static final int POINTS_PER_NODE = 1024;
    private static Logger logger = LogUtil.CLB_LOGGER.getLogger();

    /** Synchronization lock. */
    private final Object lock;

    /**
     * Consistent hash for the ideal configuration (not updated at server
     * failure). Contains all servers in all clusters.
     */
    protected ConsistentHash<String, ServerWrapper> idealConsistentHash;

    /**
     * Consistent hash for the ideal configuration (updated at server
     * failure/recovery). Contains all healthy servers in all clusters.
     */
    private ConsistentHash<String, ServerWrapper> actualConsistentHash;

    /**
     * All clusters that are associated with this router and that are used for
     * routing at fail-over.
     */
    private List<ServerCluster> associatedClusters = new ArrayList<ServerCluster>();

    /** The composite hash key extractor. */
    private CompositeHashKeyExtractor hashKeyExtractor = new CompositeHashKeyExtractor();

    /** The XML file for DCR configuration. */
    private File dcrFile;
    protected boolean ignoreIdealHash;
    private Controller controller;

    /**
     * @param associatedClusters
     */
    public ConsistentHashRouter(List<ServerCluster> associatedClusters,
        boolean activeRouting) {
        this.associatedClusters = associatedClusters;
        this.ignoreIdealHash = activeRouting;
        lock = new Object(); 
    }

    /**
     * <b>Note</b>, this method assumes that the specified request is an
     * instance of {@link ConsistentHashRequest}.
     * <p>
     * Also note that the provided request will be updated, the hash key will be
     * inserted into it.
     *
     * @see org.jvnet.glassfish.comms.clb.core.Router#selectInstance(Request)
     */
    public ServerInstance selectInstance(ConsistentHashRequest request) {
        // Must be a ConsistentHashRequest
        if (request.getHashKey() == null) {
            // Extract the hash key...
            if (request.isHttp()) {
                request.setHashKey(hashKeyExtractor.getHashKey(
                        request.getHttpRequest()));
            } else {
                request.setHashKey(hashKeyExtractor.getHashKey(
                        request.getSipRequest()));
            }
        }

        // Then do routing based on the hash key...
        return getServerInstance(request);
    }

    /**
     * Gets the server instance for the specified request.
     *
     * @param req the request
     * @return the server instance for the given hash key
     */
    protected ServerInstance getServerInstance(ConsistentHashRequest req) {
        return getActiveInstance(req);
    }

    protected ServerInstance getActiveInstance(ConsistentHashRequest req){
        if (req.getHashKey() == null) {
            return null;
        }
        ServerWrapper serverWrapper;
        synchronized (lock){
             serverWrapper = actualConsistentHash.get(req.getHashKey());
        }
            if (serverWrapper != null) {
                return serverWrapper.getServerInstance();
            } else {
                return null;
            }
    }
    
    /**
     * Initializes this router.
     *
     * @see org.jvnet.glassfish.comms.clb.core.Router#initialize()
     */
    public boolean initialize() {
            setupHashKeyExtractor();

            if (!ignoreIdealHash) {
                // Only create this if it is a request group router
                idealConsistentHash = new ConsistentHash<String, ServerWrapper>(POINTS_PER_NODE, "CLB_IdealHash");
            }

            actualConsistentHash = new ConsistentHash<String, ServerWrapper>(POINTS_PER_NODE,  "CLB_ActualHash");

            for (ServerCluster cluster : associatedClusters) {
                for (ServerInstance server : cluster.getAllInstances()) {
                    server.addRouter(this);

                    if (idealConsistentHash != null) {
                        idealConsistentHash.addNode(new ServerWrapper(server));
                    }

                    if (server.isHealthy() && server.isEnabled()) {
                        actualConsistentHash.addNode(new ServerWrapper(server));
                    }
                }
            }

            if (idealConsistentHash != null) {
                idealConsistentHash.setup();
            }

            actualConsistentHash.setup();
            return true;
    }

    /**
     * Sets the DCR file.
     *
     * @param dcrFile the DCR file
     */
    public void setDcrFile(File dcrFile) {
        this.dcrFile = dcrFile;
    }

    /**
     * Sets up the hash key extractor.
     * <p>
     * Pushes first a default hash key extractor and then a DCR extractor.
     */
    protected void setupHashKeyExtractor() {
        // First push the default hash key extractor...
        DefaultHashKeyExtractor defaultHashKeyExtractor = new DefaultHashKeyExtractor();
        pushHashKeyExtractor(defaultHashKeyExtractor);

        // Then DCR hash key extractor...
        DcrUtils.setup(DnsResolver.getInstance(), SipFactoryImpl.getInstance());

        // Only try to read DCR file if it exists; otherwise fall-back to Call-ID, from-tag (the default extractor)
        if ((dcrFile != null) && dcrFile.exists()) {
            try {
                DcrConfigurableHashKeyExtractor dcrHashKeyExtractor = new DcrConfigurableHashKeyExtractor(dcrFile.getAbsolutePath());
                pushHashKeyExtractor(dcrHashKeyExtractor);
            } catch (DcrRulesException e) {
                logger.severe("Failed to read DCR configuration: " + e +
                    " ; will only use default hash key extraction.");
            }
        }
        
        pushHashKeyExtractor(new StickyHashKeyExtractor());
    }

    /**
     * Pushes the specified hash key extractor on top of the others.
     *
     * @param hke the hash key extractor
     */
    protected void pushHashKeyExtractor(HashKeyExtractor hke) {
        hashKeyExtractor.push(hke);
    }

    public void handleDisableEvent(ServerInstance instance) {
        if (instance.isHealthy()) {
            removeFromActualConsistentHash(instance);
        }
    }

    public void handleEnableEvent(ServerInstance instance) {
        if (instance.isHealthy()) {
            addToActualConsistentHash(instance);
        }
    }

    public void handleFailureEvent(ServerInstance instance) {
        if (instance.isEnabled()) {
            removeFromActualConsistentHash(instance);
        }
    }

    public void handleRecoveryEvent(ServerInstance instance) {
        if(instance.isEnabled()) {
            addToActualConsistentHash(instance);
        }
    }
    
    private void addToActualConsistentHash(final ServerInstance instance) {
        if(logger.isLoggable(Level.FINE))
            logger.log(Level.FINE,
                    "clb.adding_to_active_list",
                    new Object[]{instance.getName()});
        synchronized (lock) {
            actualConsistentHash.addNode(new ServerWrapper(instance));
            actualConsistentHash.setup();
        }
    }
    
    private void removeFromActualConsistentHash(final ServerInstance instance) {
        if(logger.isLoggable(Level.FINE))
            logger.log(Level.FINE,
                    "clb.removing_from_active_list",
                    new Object[]{instance.getName()});
        synchronized (lock) {
            actualConsistentHash.removeNode(new ServerWrapper(instance));
            actualConsistentHash.setup();
        }
    }

    public ServerInstance selectInstance(HttpRequest req) {
        throw new UnsupportedOperationException(
            "Not applicable for Consistent hash router.");
    }

    public void processDCRFileUpdateEvent() {
        setDcrFile(new File(controller.getDCRFileName()));
        setupHashKeyExtractor();
    }

    public void setController(Controller controller) {
        this.controller = controller;
        String dcrFileName = controller.getDCRFileName();
        File dcrFile = null;
        if(dcrFileName != null)
            dcrFile = new File(dcrFileName);
        setDcrFile(dcrFile);
        if (controller != null) {
            controller.getDCRFileUpdateEventNotifier()
                      .addDCRFileUpdateEventListener(this);
        }
    }
    
    protected static class ServerWrapper {
        private ServerInstance serverInstance;
        
        protected ServerWrapper(ServerInstance serverInstance) {
            assert serverInstance != null: "Must wrap a real object!";
            this.serverInstance = serverInstance;
        }

        protected ServerInstance getServerInstance() {
            return serverInstance;
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj)
                return true;
            if (obj == null)
                return false;
            if (obj instanceof ServerWrapper) { 
                return serverInstance.equals(((ServerWrapper)obj).getServerInstance());
            } else if (obj instanceof ServerInstance) { 
                return serverInstance.equals(obj);
            } 
            return false;
        }

        @Override
        public int hashCode() {
            final int prime = 31;
            int result = 1;
            String clusterName = serverInstance.getServerCluster().getName();
            result = prime * result + ((clusterName == null) ? 0 : clusterName.hashCode());
            result = prime * result + ((serverInstance.getName() == null) ? 0 : serverInstance.getName().hashCode());
            return result;
        }

        @Override
        public String toString() {
            return LoadbalancerUtil.getServerString(serverInstance.getServerCluster().getName(), serverInstance.getName());
        }
    }
}
