/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.jvnet.glassfish.comms.clb.core.common.chr;

import java.util.ListIterator;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.servlet.sip.Address;
import javax.servlet.sip.ServletParseException;
import javax.servlet.sip.SipServletRequest;
import javax.servlet.sip.URI;

import org.jvnet.glassfish.comms.clb.core.CLBConstants;
import org.jvnet.glassfish.comms.clb.core.util.LoadbalancerUtil;
import org.jvnet.glassfish.comms.clb.proxy.http.util.HttpRequest;
import org.jvnet.glassfish.comms.util.LogUtil;

import com.ericsson.ssa.sip.Header;
import com.ericsson.ssa.sip.SipServletMessageImpl;
import com.ericsson.ssa.sip.URIImpl;
import com.ericsson.ssa.sip.UriUtil;
import com.sun.grizzly.util.buf.MessageBytes;


/**
 * This hash key extractor extracts hash key encoded in the BEkey cookie of a
 * request.
 */
public class StickyHashKeyExtractor implements HashKeyExtractor {
    private static Logger logger = LogUtil.CLB_LOGGER.getLogger();

    /** The URI parameter name for the hash key to the Back-End. */
    private static final String BE_KEY_PARAM = URIImpl.BEKEY_PARAM;

    /** The header used to transfer BEKey between front-end and back-end. */
    public static final String PROXY_BEKEY_HEADER = Header.PROXY_BEKEY_HEADER;

    /**
     * @see org.jvnet.glassfish.comms.clb.core.common.chr.HashKeyExtractor#getHashKey(javax.servlet.http.HttpServletRequest)
     */
    public String getHashKey(HttpRequest request) {
        String value = null;
        MessageBytes uriMB = request.requestURI();
        String requestURI = uriMB.toString();

        //search URI for encoding
        value = extractBEKeyFromURI(requestURI); 

        if (value != null) {
            if (logger.isLoggable(Level.INFO)) {
                logger.log(Level.INFO, "clb.found_route_encodedURL",
                           new Object[] {CLBConstants.BEKEY_KEY_NAME,
                                         value, requestURI
                                        });
            }
            return value;
        }

        //search the cookies
        String cookieString = request.getHeader(CLBConstants.COOKIE_HEADER);

        if ((cookieString == null) || (cookieString.length() == 0)) {
            return null;
        }

        int index = cookieString.indexOf(CLBConstants.BEKEY_KEY_NAME);

        if (index == -1) {
            return null;
        }

        int valueStartIndex = cookieString.indexOf("=", index + 1);

        if (valueStartIndex == -1) {
            return null;
        }

        int valueEndIndex = cookieString.indexOf(";", valueStartIndex + 1);

        if (valueEndIndex == -1) {
            valueEndIndex = cookieString.indexOf(",", valueStartIndex + 1);
        }

        if (valueEndIndex == -1) {
            return cookieString.substring(valueStartIndex + 1);
        }

        return cookieString.substring(valueStartIndex + 1, valueEndIndex);
    }

    /**
     * @see org.jvnet.glassfish.comms.clb.core.common.chr.HashKeyExtractor#getHashKey(javax.servlet.sip.SipServletRequest)
     */
    public String getHashKey(SipServletRequest req) {
        return extractHashKey(req);
    }

    private String extractHashKey(SipServletRequest request) {
    	
        // First check if it is in proxy-bekey (header used for bekey transfer between F-E and B-E)
        String bekey = request.getHeader(PROXY_BEKEY_HEADER);
        
        if (bekey != null) {
			request.removeHeader(PROXY_BEKEY_HEADER);

			if (logger.isLoggable(Level.FINE)) {
				if (bekey != null) {
					logger.log(Level.FINE, "bekey: " + bekey
							+ " was found in the Proxy-Bekey header");
				}
			}
			return bekey;
		}
        
        // is there a bekey in topmost Route?
        Header routeHeader = ((SipServletMessageImpl) request).getRawHeader(Header.ROUTE);

        if (routeHeader != null) {
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE,
                    "The request contains a Route header: " + routeHeader);
            }

            ListIterator<Address> routeIterator;

            try {
                routeIterator = routeHeader.getAddressValues();
            } catch (ServletParseException e) {
                logger.log(Level.WARNING,
                    StickyHashKeyExtractor.class.getName() +
                    ".address_parse_error", e);

                return null;
            }

            if (routeIterator.hasNext()) {
                Address topRoute = routeIterator.next();
                bekey = decodeHashKeyFromBeKey(topRoute.getURI());
                
                if (bekey == null) {
                	return null;
                }
                else {
                    if (logger.isLoggable(Level.FINE)) {
                        logger.log(Level.FINE,
                            "The bekey: " + bekey +
                            " was found in the topmost Route: " + topRoute);
                    }
                	return bekey;
                }
            }
        }

        
        // is there a bekey in request-URI?
        bekey = decodeHashKeyFromBeKey(request.getRequestURI());

        if (bekey != null) {
            if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE,
                        "bekey: " + bekey + " was found in the request-URI");
            }
            return bekey;
        }

        // last resort: 
        // is there a bekey in to Header? (may only happen on ACK's)
        if ("ACK".equalsIgnoreCase(request.getMethod())) {
        	// (Only) if the request is an ACK there may exist an bekey on the To header 
            // Try to extract BEKey To-header
            Address toHeader;
            try {
                toHeader = request.getAddressHeader(Header.TO);
            } catch (ServletParseException e) {
                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE, "Exception when extracting the To-header", e);
                }
                
                toHeader = null;
            }
            
            if (toHeader != null) {
				bekey = decodeHashKeyFromBeKey(toHeader.getURI());
				if (bekey != null) {
					if (logger.isLoggable(Level.FINE)) {
						logger.log(Level.FINE, "bekey: " + bekey
								+ " was found in the To-header");
					}

					return bekey;
				}
			}
        }

		if (logger.isLoggable(Level.FINE)) {
			logger.log(Level.FINE, "no bekey found, it must be a new request: ");
		}
        // this must be a new request, --> return null --> apply dcr rules
        return null;
    }

    /**
     * Decodes the hashkey from the bekey parameter (if existing).
     * @param uri the URI where the bekey parameter exists
     * @return the decoded hashkey or null (if non-existing or malformed)
     */
    public static String decodeHashKeyFromBeKey(URI uri) {
        return UriUtil.getParameter(uri, BE_KEY_PARAM);
    }

    /**
     * Encode the specified hashkey as a bekey parameter into the specified URI.
     * @param uri the URI
     * @param hashkey the hash key
     */
    public static void encodeHashKeyToBeKey(URI uri, String hashkey) {
        // Replace any escape character (%) in incoming hash key to its escaped value (%25)
        // this is to be able to escape the hash key without affecting any original escaping 
        // when decoding the bekey in decodeHashKeyFromBeKey(URI uri).
        String bekey = hashkey.replaceAll("%", "%25");
        String paramValue = LoadbalancerUtil.encodeParameter(bekey);
        UriUtil.setParameter(uri, StickyHashKeyExtractor.BE_KEY_PARAM, paramValue);
    }

    /**
     * Extract the encoded BEKey from the requestURI.
     * @param requestURI The Http request URI
     *
     * @return BEKey value if found or null
     */
    private String extractBEKeyFromURI(String requestURI) {
        String extractedBEKey = null;

        try {
            int startIndexForBEKey = requestURI.indexOf(
                                              CLBConstants.BEKEY_URI_IDENTIFIER);
            if (startIndexForBEKey != -1) {
                //Found BEKey URL encoded.
                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE,
                               "clb.retrieving_URLencoded_route",
                                new Object[] {CLBConstants.BEKEY_KEY_NAME,
                                              requestURI
                                             });
                }

                int startIndexForBEKeyValue = startIndexForBEKey
                                     + CLBConstants.BEKEY_URI_IDENTIFIER.length();
                int startIndexForNextID = requestURI.indexOf(
                                                CLBConstants.COMMA_DELIMITER, 
                                                startIndexForBEKeyValue);

                if (startIndexForNextID != -1) {
                    //There are more id's encoded; extract BEKey
                    if (logger.isLoggable(Level.FINE)) {
                        logger.log(Level.FINE,
                                   "clb.multiple_URLencoded_IDs",
                                    new Object[]{CLBConstants.BEKEY_KEY_NAME,
                                                 requestURI
                                                });
                    }
                    extractedBEKey = requestURI.substring(
                                                      startIndexForBEKeyValue,
                                                      startIndexForNextID);
                } else {
                    if (logger.isLoggable(Level.FINE)) {
                        logger.log(Level.FINE,
                                   "clb.zero_URLencoded_IDs_after_route",
                                    new Object[] {CLBConstants.BEKEY_KEY_NAME,
                                                  requestURI
                                                 });

                    }
                    extractedBEKey = requestURI.substring(
                                                     startIndexForBEKeyValue);
                }

                return (extractedBEKey.trim());
            }         
        } catch (IndexOutOfBoundsException ex) {
              if (logger.isLoggable(Level.SEVERE)) {
                  logger.log(Level.SEVERE,
                             "clb.error_extract_route_encodedURL",
                             new Object[]{ CLBConstants.BEKEY_KEY_NAME,
                                           requestURI
                                         });
                  logger.log(Level.SEVERE, ex.getMessage(), (Throwable)ex);
              } 
        }
        return extractedBEKey;
    }
    
}
