/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.jvnet.glassfish.comms.clb.core.sip;

import com.ericsson.ssa.sip.dns.SipTransports;

import java.io.IOException;

import java.util.StringTokenizer;


/**
 * Information that specifies a connection between two endpoints.
 */
public class Connection {
    private static final String DELIMITER = "$";
    private static final String UNDEFINED = "UNDEFINED";
    private SipTransports transport;
    private Socket remoteEndpoint;
    private Socket localEndpoint;

    /**
     * Create a connection.
     *
     * @param transport the transport protocol
     * @param localSocket the local end point; set to when creating a new
     *                outgoing connection
     * @param remoteSocket the remote end point
     */
    public Connection(SipTransports transport, Socket localSocket,
        Socket remoteSocket) {
        this.transport = transport;
        this.localEndpoint = localSocket;
        this.remoteEndpoint = remoteSocket;
    }

    /**
     * Create connection from a string encoded using {@link #getEncodedValue()}.
     *
     * @param encodedValue
     * @throws ConnectionParseException
     */
    private Connection(String encodedValue) throws ConnectionParseException {
        try {
            StringTokenizer st = new StringTokenizer(encodedValue, DELIMITER);

            while (st.hasMoreTokens()) {
                transport = SipTransports.getTransport(st.nextToken());

                String hostName = st.nextToken();
                String portStr = st.nextToken();

                if (!portStr.equals(hostName)) {
                    int port = Integer.parseInt(portStr);
                    localEndpoint = new Socket(hostName, port);
                }

                hostName = st.nextToken();
                portStr = st.nextToken();

                if (!portStr.equals(hostName)) {
                    int port = Integer.parseInt(portStr);
                    remoteEndpoint = new Socket(hostName, port);
                }
            }
        } catch (Throwable t) {
            throw new ConnectionParseException(
                "The encoded connection data was incorrectly formatted: " +
                encodedValue, t);
        }
    }

    /**
     * Gets the connection encoded in one string
     *
     * @return the connection encoded in one string
     */
    public String getEncodedValue() {
        String value = '"' + toString() + '"';

        return value.trim();
    }

    /**
     * Creates an instance from an encoded string.
     *
     * @param s the encoded string
     * @return an instance from a Base64 encoded string
     * @throws ConnectionParseException in case connection was malformed
     * @throws IOException in case Base64 decoding failed
     */
    public static Connection getFromEncoded(String s)
        throws ConnectionParseException, IOException {
        String str = s.substring(1, s.length()-1);

        return new Connection(str);
    }

    /**
     * Gets the remote end point.
     *
     * @return the remote end point
     */
    public Socket getRemoteEndpoint() {
        return remoteEndpoint;
    }

    /**
     * Gets the local end point.
     *
     * @return the local end point
     */
    public Socket getLocalEndpoint() {
        return localEndpoint;
    }

    /**
     * @see java.lang.Object#toString()
     */
    @Override
    public String toString() {
        return transport.name() + DELIMITER +
        ((localEndpoint != null) ? localEndpoint.getHostName() : UNDEFINED) +
        DELIMITER +
        ((localEndpoint != null) ? localEndpoint.getPort() : UNDEFINED) +
        DELIMITER +
        ((remoteEndpoint != null) ? remoteEndpoint.getHostName() : UNDEFINED) +
        DELIMITER +
        ((remoteEndpoint != null) ? remoteEndpoint.getPort() : UNDEFINED);
    }

    /**
     * Gets the transport
     *
     * @return the transport
     */
    public SipTransports getTransport() {
        return transport;
    }
}
