/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.jvnet.glassfish.comms.clb.core.sip;

import com.ericsson.ssa.sip.Layer;
import com.ericsson.ssa.sip.LayerHelper;
import com.ericsson.ssa.sip.SipServletRequestImpl;
import com.ericsson.ssa.sip.SipServletResponseImpl;
import com.ericsson.ssa.sip.dns.SipTransports;
import com.ericsson.ssa.sip.dns.TargetTuple;

import com.sun.grizzly.tcp.Response;

import org.jvnet.glassfish.comms.clb.core.CLBRuntimeException;
import org.jvnet.glassfish.comms.clb.core.ConsistentHashRequest;
import org.jvnet.glassfish.comms.clb.core.Controller;
import org.jvnet.glassfish.comms.clb.core.RequestGroup;
import org.jvnet.glassfish.comms.clb.core.Router;
import org.jvnet.glassfish.comms.clb.core.RouterFactory;
import org.jvnet.glassfish.comms.clb.core.ServerCluster;
import org.jvnet.glassfish.comms.clb.core.util.LoadbalancerUtil;
import org.jvnet.glassfish.comms.clb.proxy.http.util.HttpRequest;
import org.jvnet.glassfish.comms.util.LogUtil;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;


/**
 * This is the request group implementation for SIP.
 */
public class SipRequestGroup implements RequestGroup {
    private static Logger logger = LogUtil.CLB_LOGGER.getLogger();
    private SipLoadBalancerIncomingHandler sipLoadBalancer;
    private Layer nextLayer;
    private Layer lbmLayer;
    private List<ServerCluster> clusters = new ArrayList<ServerCluster>();
    private Router stickyRouter;
    private Controller controller;

    /**
     * Create the request group.
     */
    public SipRequestGroup(Controller controller) {
        this.controller = controller;
    }

    /**
     * @see org.jvnet.glassfish.comms.clb.core.RequestGroup#addCluster(org.jvnet.glassfish.comms.clb.core.ServerCluster)
     */
    public void addCluster(ServerCluster cluster) {
        clusters.add(cluster);
    }

    /**
     * @see org.jvnet.glassfish.comms.clb.core.RequestGroup#deleteCluster(org.jvnet.glassfish.comms.clb.core.ServerCluster)
     */
    public void deleteCluster(ServerCluster cluster) {
        clusters.remove(cluster);
    }

    /**
     * @see org.jvnet.glassfish.comms.clb.core.RequestGroup#handleIncomingRequest(com.ericsson.ssa.sip.SipServletRequestImpl)
     */
    public void handleIncomingRequest(SipServletRequestImpl request) {
        Connection connection;
        controller.incrementRequestCount();

        try {
            ConsistentHashRequest chr = new ConsistentHashRequest(request, null);
            connection = sipLoadBalancer.handleIncomingRequest(chr, stickyRouter);

            if (connection != null) {
                // Alter connection and dispatch
                SipLoadBalancerManagerBackEnd.resetNetworkManager(request);
                request.setRemote(new TargetTuple(connection.getTransport(),
                        connection.getRemoteEndpoint().getHostName(),
                        connection.getRemoteEndpoint().getPort()));
                request.popDispatcher().dispatch(request);
            } else {
                // Continue on this server
                request.pushTransactionDispatcher(lbmLayer);
                request.pushApplicationDispatcher(lbmLayer);
                LayerHelper.next(request, lbmLayer, nextLayer);
            }
        } catch (SipRoutingException e) {
            logger.warning("Failed to route incoming request: " + e);

            SipServletResponseImpl errorResponse = request.createTerminatingResponse(503);

            if (errorResponse != null) {
                errorResponse.popDispatcher().dispatch(errorResponse);
            }
        } finally {
            controller.decrementRequestCount();
        }
    }

    /**
     * @see org.jvnet.glassfish.comms.clb.core.RequestGroup#handleIncomingResponse(com.ericsson.ssa.sip.SipServletResponseImpl)
     */
    public void handleIncomingResponse(SipServletResponseImpl response) {
        Connection connection;
        controller.incrementRequestCount();

        try {
            connection = sipLoadBalancer.handleIncomingResponse(response);

            if (connection != null) {
                // Alter connection and dispatch
                SipLoadBalancerManagerBackEnd.resetNetworkManager(response);
                response.setRemote(new TargetTuple(connection.getTransport(),
                        connection.getRemoteEndpoint().getHostName(),
                        connection.getRemoteEndpoint().getPort()));
                response.popDispatcher().dispatch(response);
            } else {
                // Continue on this server
                LayerHelper.next(response, lbmLayer, nextLayer);
            }
        } catch (SipRoutingException e) {
            logger.warning("Failed to route incoming response: " + e);

            // Drop the response
        } finally {
            controller.decrementRequestCount();
        }
    }

    /**
     * @see org.jvnet.glassfish.comms.clb.core.RequestGroup#initializeRouter(java.lang.String)
     */
    public void initializeRouter(String lbpolicy) throws CLBRuntimeException {
        setupServerSelector();
        sipLoadBalancer = new SipLoadBalancerIncomingHandler(LoadbalancerUtil.getLocalSocket(
                    SipTransports.TCP_PROT));
    }

    private void setupServerSelector() throws CLBRuntimeException {
        stickyRouter = RouterFactory.createRouter(RouterFactory.LBPolicy_STICKY_CONSISTENT_HASH,
                clusters, false);
        stickyRouter.setController(controller);
        stickyRouter.initialize();
    }

    /**
     * @see org.jvnet.glassfish.comms.clb.core.RequestGroup#reinitializeRouter()
     */
    public void reinitializeRouter() throws CLBRuntimeException {
        setupServerSelector();
    }

    public void processDCRFileUpdateEvent() throws CLBRuntimeException {
        setupServerSelector();
    }

    /**
     * @see org.jvnet.glassfish.comms.clb.core.RequestGroup#serviceRequest(org.jvnet.glassfish.comms.clb.core.Request,
     *      org.jvnet.glassfish.comms.clb.core.Response)
     */
    public void serviceRequest(HttpRequest req, Response res) {
        throw new UnsupportedOperationException("Not relevant for SIP");
    }

    /**
     * Sets the next layer.
     *
     * @param nextLayer the nextLayer to set
     */
    public void setNextLayer(Layer nextLayer) {
        this.nextLayer = nextLayer;
    }

    /**
     * Sets the LBM layer that this request group belongs to.
     *
     * @param lbmLayer the lbmLayer to set
     */
    public void setLbmLayer(Layer lbmLayer) {
        this.lbmLayer = lbmLayer;
    }

    public void cleanup() {
        clusters = null;
        controller = null;
    }
}
