/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * Portions Copyright Apache Software Foundation.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.jvnet.glassfish.comms.clb.proxy.http.util;

import com.sun.grizzly.http.Constants;
import com.sun.grizzly.tcp.InputBuffer;
import com.sun.grizzly.tcp.Request;
import com.sun.grizzly.util.buf.ByteChunk;
import com.sun.grizzly.util.buf.MessageBytes;
import com.sun.grizzly.util.http.MimeHeaders;

import org.jvnet.glassfish.comms.clb.proxy.config.LoadBalancerProxyConstants;
import org.jvnet.glassfish.comms.clb.proxy.config.ProxyConfig;

import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;

import java.util.logging.Logger;

/**
 * This is a slightly modified implementation of the InternalIn put buffer
 * in grizzly, we cannot extend that because that buf in that implementation is
 * not preserved when parsing the headers, it tampers with the request bytes
 * by removing spaces and converting upper to lower case...
 *
 * Implementation of InputBuffer which provides HTTP request header parsing as
 * well as transfer decoding.
 *
 * @author <a href="mailto:remm@apache.org">Remy Maucherat</a>
 */
public class HttpInputBuffer implements InputBuffer {

    private Logger _logger;
    /**
     * Associated Coyote request.
     */
    protected Request request;
    /**
     * Headers of the associated request.
     */
    protected MimeHeaders headers;
    /**
     * State.
     */
    protected boolean parsingHeader;
    /**
     * Swallow input ? (in the case of an expectation)
     */
    protected boolean swallowInput;
    /**
     * Pointer to the current read buffer.
     */
    protected byte[] buf;
    /**
     * Last valid byte.
     */
    protected int lastValid;
    /**
     * Position in the buffer.
     */
    protected int pos;

    /*
     * Pos of the end of the header in the buffer, which is also the
     * start of the body.
     **/
    protected int headerEnd;
    /**
     * Underlying input stream.
     */
    protected InputStream inputStream;
    /**
     * Underlying input buffer.
     */
    protected InputBuffer inputStreamInputBuffer;

    // START OF SJSAS 6231069
    /**
     * The stage we currently are during parsing the request line and
     * the headers
     */
    private MessageBytes headerValue = null;

    /** Creates a new instance of HttpInputBuffer */
    public HttpInputBuffer() {
    }

    /**
     * Default constructor.
     */
    public HttpInputBuffer(Request request) {
        this(request, LoadBalancerProxyConstants.DEFAULT_HTTP_HEADER_BUFFER_SIZE);
    }

    /**
     * Alternate constructor.
     */
    public HttpInputBuffer(Request request, int headerBufferSize) {
        _logger = ProxyConfig.getInstance().getLogger();

        this.request = request;
        headers = request.getMimeHeaders();

        buf = new byte[headerBufferSize];
        inputStreamInputBuffer = new InputStreamInputBuffer();

        parsingHeader = true;
        swallowInput = true;
    }

    // END OF SJSAS 6231069

    // ------------------------------------------------------------- Properties
    /**
     * Set the underlying socket input stream.
     */
    public void setInputStream(InputStream inputStream) {
        // FIXME: Check for null ?
        this.inputStream = inputStream;
    }

    /**
     * Get the underlying socket input stream.
     */
    public InputStream getInputStream() {
        return inputStream;
    }

    /**
     * Set the swallow input flag.
     */
    public void setSwallowInput(boolean swallowInput) {
        this.swallowInput = swallowInput;
    }

    /**
     * Recycle the input buffer. This should be called when closing the
     * connection.
     */
    public void recycle() {
        // Recycle Request object
        request.recycle();

        inputStream = null;
        lastValid = 0;
        pos = 0;
        parsingHeader = true;
        swallowInput = true;
    }

    /**
     * End processing of current HTTP request.
     * Note: All bytes of the current request should have been already
     * consumed. This method only resets all the pointers so that we are ready
     * to parse the next HTTP request.
     */
    public void nextRequest() {
        // Recycle Request object
        request.recycle();
        /**
         * This is mainly for request pipelining where a single
         * read can contain more than one request. If we support
         * pipelining then we have to copy the next requestto
         * the begining of buffer, for lb its difficult to 
         * support this because this request has to go through 
         * the finder. Cannot support pipelining for now.
         */
        /*
           / Determine the header buffer used for next request					   
        byte[] newHeaderBuf = null;								   
												   
        // Copy leftover bytes to the beginning of the buffer					   
        if ((lastValid - pos) > 0) {								   
            int npos = 0;									   
            int opos = pos;									   
            int opos = pos;							   
												   
            while ((lastValid - opos) > (opos - npos)) {					   
                System.arraycopy(buf, opos, buf, npos, opos - npos);				   
                npos += pos;									   
                opos += pos;									   
            }											   
												   
            System.arraycopy(buf, opos, buf, npos, lastValid - opos);				   
        }											   
												   
        // Reset pointers									   
        lastValid = lastValid - pos;
         */
        lastValid = 0;
        pos = 0;
        parsingHeader = true;
        swallowInput = true;
    }

    /**
     * End request (consumes leftover bytes).
     *
     * @throws IOException an undelying I/O error occured
     */
    public void endRequest() {
        ;
    }

    /**
     * Read the request line. This function is meant to be used during the
     * HTTP request header parsing. Do NOT attempt to read the request body
     * using it.
     *
     * @throws IOException If an exception occurs during the underlying socket
     * read operations, or if the given buffer is not big enough to accomodate
     * the whole line.
     */
    public void parseRequestLine() throws IOException {
        int start = 0;

        // END OF SJSAS 6231069
        byte chr = 0;
        do {
            // Read new bytes if needed
            if (pos >= lastValid) {
                if (!fill()) {
                    throw new EOFException("EOF_error");
                }
            }

            chr = buf[pos++];
        } while ((chr == Constants.CR) || (chr == Constants.LF));

        pos--;

        // Mark the current buffer position
        start = pos;

        //
        // Reading the method name
        // Method name is always US-ASCII
        //
        boolean space = false;

        while (!space) {
            // Read new bytes if needed
            if (pos >= lastValid) {
                if (!fill()) {
                    throw new EOFException("EOF_error");
                }
            }

            if (buf[pos] == Constants.SP) {
                space = true;
                request.method().setBytes(buf, start, pos - start);
            }

            pos++;
        }

        // Mark the current buffer position
        start = pos;

        int end = 0;
        int questionPos = -1;

        //
        // Reading the URI
        //
        space = false;

        boolean eol = false;

        while (!space) {
            // Read new bytes if needed
            if (pos >= lastValid) {
                if (!fill()) {
                    throw new EOFException("EOF_error");
                }
            }

            if (buf[pos] == Constants.SP) {
                space = true;
                end = pos;
            } else if ((buf[pos] == Constants.CR) ||
                    (buf[pos] == Constants.LF)) {
                // HTTP/0.9 style request
                eol = true;
                space = true;
                end = pos;
            } else if ((buf[pos] == Constants.QUESTION) && (questionPos == -1)) {
                questionPos = pos;
            }

            pos++;
        }

        request.unparsedURI().setBytes(buf, start, end - start);

        if (questionPos >= 0) {
            request.queryString().setBytes(buf, questionPos + 1, end - questionPos - 1);
            request.requestURI().setBytes(buf, start, questionPos - start);
        } else {
            request.requestURI().setBytes(buf, start, end - start);
        }

        // Mark the current buffer position
        start = pos;
        end = 0;

        //
        // Reading the protocol
        // Protocol is always US-ASCII
        //
        while (!eol) {
            // Read new bytes if needed
            if (pos >= lastValid) {
                if (!fill()) {
                    throw new EOFException("EOF_error");
                }
            }

            if (buf[pos] == Constants.CR) {
                end = pos;
            } else if (buf[pos] == Constants.LF) {
                if (end == 0) {
                    end = pos;
                }

                eol = true;
            }

            pos++;
        }

        if ((end - start) > 0) {
            request.protocol().setBytes(buf, start, end - start);
        } else {
            request.protocol().setString("");
        }
    }

    /**
     * Parse the HTTP headers.
     */
    public void parseHeaders() throws IOException {
        while (parseHeader()) {
        }

        parsingHeader = false;
        headerEnd = pos;
    }

    /**
     * Should be called only after parse heaser
     */
    public void addHeader(String name, String value) {
        if ((name == null) || (value == null)) {
            return;
        }
        StringBuffer headerstring = new StringBuffer();
        headerstring.append(name);
        headerstring.append(":");
        headerstring.append(value);
        headerstring.append(LoadBalancerProxyConstants.CRLF);
        headers.addValue(name).setString(value);
        addHeader(headerstring.toString().getBytes());
    }

    public void addHeader(byte[] headerBytes) {

        try {
            int len = headerBytes.length;
            if ((len + lastValid) >= buf.length) {
                byte[] tmpbuf = new byte[buf.length + len];
                for (int y = 0; y < lastValid; y++) {
                    tmpbuf[y] = buf[y];
                }
                buf = tmpbuf;
            }

            int count = 0;

            for (int j = (lastValid - 1); j >= (headerEnd - 2); j--) {
                buf[j + len] = buf[j];
            }
            for (int k = (headerEnd - 2); k < (headerEnd + len - 2); k++) {
                buf[k] = headerBytes[count];
                count++;
            }

            headerEnd += len;
            lastValid += len;
            pos += len;
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public byte[] getBytes() {
        return buf;
    }

    /**
     * Parse an HTTP header.
     *
     * @return false after reading a blank line (which indicates that the
     * HTTP header parsing is done
     */
    public boolean parseHeader() throws IOException {
        //
        // Check for blank line
        //
        byte chr = 0;

        while (true) {
            // Read new bytes if needed
            if (pos >= lastValid) {
                if (!fill()) {
                    throw new EOFException("EOF_error");
                }
            }

            chr = buf[pos];

            if ((chr == Constants.CR) || (chr == Constants.LF)) {
                if (chr == Constants.LF) {
                    pos++;

                    return false;
                }
            } else {
                break;
            }

            pos++;
        }

        // Mark the current buffer position
        int start = pos;

        //
        // Reading the header name
        // Header name is always US-ASCII
        //
        boolean colon = false;

        //MessageBytes headerValue = null;
        while (!colon) {
            // Read new bytes if needed
            if (pos >= lastValid) {
                if (!fill()) {
                    throw new EOFException("EOF_error");
                }
            }

            if (buf[pos] == Constants.COLON) {
                colon = true;
                headerValue = headers.addValue(buf, start, pos - start);
            }

            chr = buf[pos];
            /*
            if ((chr >= Constants.A) && (chr <= Constants.Z)) {
            buf[pos] = (byte) (chr - Constants.LC_OFFSET);
            }
             */
            pos++;
        }

        // Mark the current buffer position
        start = pos;

        //
        // Reading the header value (which can be spanned over multiple lines)
        //
        boolean eol = false;
        boolean validLine = true;
        int startheader = pos;
        int endheader = pos;

        while (validLine) {
            boolean space = true;

            // Skipping spaces
            while (space) {
                // Read new bytes if needed
                if (pos >= lastValid) {
                    if (!fill()) {
                        throw new EOFException("EOF_error");
                    }
                }

                if ((buf[pos] == Constants.SP) || (buf[pos] == Constants.HT)) {
                    pos++;
                } else {
                    space = false;
                }
            }

            startheader = pos;
            endheader = pos;

            // Reading bytes until the end of the line
            while (!eol) {
                // Read new bytes if needed
                if (pos >= lastValid) {
                    if (!fill()) {
                        throw new EOFException("EOF_error");
                    }
                }

                if (buf[pos] == Constants.CR) {
                    endheader = pos;
                } else if (buf[pos] == Constants.LF) {
                    eol = true;
                }

                pos++;
            }

            // Checking the first character of the new line. If the character
            // is a LWS, then it's a multiline header
            // Read new bytes if needed
            if (pos >= lastValid) {
                if (!fill()) {
                    throw new EOFException("EOF_error");
                }
            }

            chr = buf[pos];

            if ((chr != Constants.SP) && (chr != Constants.HT)) {
                validLine = false;
            } else {
                eol = false;

            // Copying one extra space in the buffer (since there must
            // be at least one space inserted between the lines)
            }
        }

        // Set the header value
        headerValue.setBytes(buf, startheader, endheader - startheader);

        return true;
    }

    // ---------------------------------------------------- InputBuffer Methods
    /**
     * Read some bytes.
     */
    public int doRead(ByteChunk chunk, Request req) throws IOException {
        return inputStreamInputBuffer.doRead(chunk, req);
    }

    // ------------------------------------------------------ Protected Methods
    /**
     * Fill the internal buffer using data from the undelying input stream.
     *
     * @return false if at end of stream
     */
    protected boolean fill() throws IOException {
        int nRead = 0;
        if (parsingHeader) {
            if (lastValid == buf.length) {
                if ((buf.length*2) > 
                        LoadBalancerProxyConstants.MAX_HEADER_BUFFER_SIZE){
                    return false;
                } 
                byte[] tmpbuf = new byte[buf.length * 2];
                for (int i = 0; i < buf.length; i++) {
                    tmpbuf[i] = buf[i];
                }
                buf = tmpbuf;
            }

            nRead = inputStream.read(buf, pos, buf.length - lastValid);

            if (nRead > 0) {
                lastValid = pos + nRead;
            }
        } else {
            if ((buf.length - headerEnd) < 4500) {
                // In this case, the request header was really large, so we allocate a
                // brand new one; the old one will get GCed when subsequent requests
                // clear all references
                buf = new byte[buf.length];
                headerEnd = 0;
            }

            pos = headerEnd;
            lastValid = pos;
            nRead = inputStream.read(buf, pos, buf.length - lastValid);

            if (nRead > 0) {
                lastValid = pos + nRead;
            }
        }
        return (nRead > 0);
    }

    public int lastValidPos() {
        return lastValid;
    }

    public MimeHeaders getHeaders() {
        return this.headers;
    }

    public int getHeaderLength() {
        return this.headerEnd;
    }

    // ------------------------------------- InputStreamInputBuffer Inner Class
    /**
     * This class is an input buffer which will read its data from an input
     * stream.
     */
    protected class InputStreamInputBuffer implements InputBuffer {

        /**
         * Read bytes into the specified chunk.
         */
        public int doRead(ByteChunk chunk, Request req)
                throws IOException {
            if (pos >= lastValid) {
                if (!fill()) {
                    return -1;
                }
            }

            int length = lastValid - pos;
            chunk.setBytes(buf, pos, length);
            pos = lastValid;

            return (length);
        }
    }
}
