/*
 * The contents of this file are subject to the terms
 * of the Common Development and Distribution License
 * (the License).  You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the license at
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing
 * permissions and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL
 * Header Notice in each file and include the License file
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.
 * If applicable, add the following below the CDDL Header,
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 */
package org.jvnet.glassfish.comms.startup;

import com.ericsson.ssa.config.Config;
import com.ericsson.ssa.config.ConfigFactory;
import com.ericsson.ssa.config.ConfigFactory;
import com.ericsson.ssa.config.LayerHandler;
import com.ericsson.ssa.config.SipFactoryFacade;
import com.ericsson.ssa.config.SipFactoryMap;
import com.ericsson.ssa.container.SipBindingResolver;
import com.ericsson.ssa.container.sim.ApplicationDispatcher;
import com.ericsson.ssa.sip.Layer;
import com.sun.enterprise.Switch;
import com.sun.enterprise.deployment.node.DescriptorFactory;
import com.sun.enterprise.deployment.node.XMLElement;
import com.sun.enterprise.deployment.pluggable.PluggableDeploymentInfo;
import com.sun.enterprise.deployment.pluggable.PluggableNodeInfo;
import com.sun.enterprise.deployment.xml.WebTagNames;
import com.sun.enterprise.management.support.SystemInfoFactory;
import com.sun.enterprise.server.event.ApplicationLoaderEventNotifier;
import com.sun.enterprise.server.event.EarLifecycleListener;
import com.sun.enterprise.web.PluggableContextInfo;
import com.sun.enterprise.web.VirtualServer;
import java.io.IOException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.ejb.TimerService;
import javax.naming.NamingException;
import javax.naming.Reference;
import javax.naming.StringRefAddr;
import javax.servlet.sip.SipFactory;
import javax.servlet.sip.SipSessionsUtil;
import org.apache.catalina.Valve;
import org.jvnet.glassfish.comms.admin.clbadmin.ClbAdmin;
import org.jvnet.glassfish.comms.admin.config.ConfigAdapter;
import org.jvnet.glassfish.comms.admin.monitor.MonitoringAdapter;
import org.jvnet.glassfish.comms.admin.monitor.MonitoringStartup;
import org.jvnet.glassfish.comms.deployment.util.SipTagNames;
import org.jvnet.glassfish.comms.httplayers.HttpLayerHandler;
import org.jvnet.glassfish.comms.startup.stack.ConfigStackConfigProvider;
import org.jvnet.glassfish.comms.startup.stack.ConfigStackConfigProviderForHttp;
import org.jvnet.glassfish.comms.startup.stack.StackConfig;
import org.jvnet.glassfish.comms.startup.stack.StackConfigException;
import org.jvnet.glassfish.comms.util.LogUtil;

/**
 * This class provides a the collection of activities that need to be called
 * during the life cycle of the Sip Container. Typically lifecycle listeners
 * Call this SipContainer at an event to perform the required activities for
 * that event.
 * The current lifecycle is as such:
 * <ul> 
 *   <LI> registerDeployer
 *   <LI> initConfig
 *   <LI> initSystemInfo
 *   <LI> initHttpLayerHandler
 *   <LI> initClbAdmin
 *   <LI> registerRealmAdapter
 *   <LI> resolveSipBindings
 *   <LI> initLayerHandler
 *   <LI> registerConfig
 *   <LI> registerFeatureName
 *   <LI> initMonitoring
 *   <LI> bindSipResources
 *   <LI> registerAtVirtualServer
 *   <LI> startSipStack
 *   <LI> reportReady
 *   <LI> stopSipStack 
 * <ul> 
 * @author elnelbo
 */
public final class SipContainer {
    private static final SipContainer INSTANCE = new SipContainer();
    private static final String GLOBAL_SIP_FACTORY_JNDI_NAME   = "sip/SipFactory";
    private static final String GLOBAL_SESSIONS_UTIL_JNDI_NAME = "sip/SipSessionsUtil";
    private static final String GLOBAL_TIMER_SERVICE_JNDI_NAME = "sip/TimerService";
    
    private Logger log = LogUtil.SIP_LOGGER.getLogger();
    private LayerHandler _layerHandler = LayerHandler.getInstance();
    private ExecutorService es = Executors.newFixedThreadPool(1);
    private HashMap<String, Class> tagDescriptorMap;    
    
    private SipContainer() {
        initMapping();
        initEarLifecycleListener();        
    }
    
    public static SipContainer instance() {
        return INSTANCE;
    }

    /**
     * Register the deployers of the AR and the SIP modules.
     */
    public void registerDeployer() {        
        registerExtensionDeployer();
        registerExtensionNodes();
        registerExtensionDescriptors(); 
    }
    
    /**
     * Initialize the ConfigAdapter. After this all the internal Config model
     * is in sync with the domain config model.
     */
    public void initConfig() {
        ConfigAdapter.instance().startup();
        Config config = ConfigFactory.getConfig();
    }
    
    /**
     * Initialize the system information by establishing the module version.
     */
    public void initSystemInfo() {        
        retrieveSipModuleVersion();        
        
        if (log.isLoggable(Level.FINE)) {
            log.log(Level.FINE, "sip.integration.sip_module_version", 
                    ConfigFactory.getConfig().get("sip.module.version"));
        }
    }
    
    /**
     * Initialize the HttpLayerHandler (and the HttpProxy) used by the CLB and
     * the SipStack from the configured stack configuration or the default 
     * stack configuration.
     * @throws org.jvnet.glassfish.comms.startup.stack.StackConfigException
     */   
    public void initHttpLayerHandler() throws StackConfigException {
        StackConfig.ensureStackConfig();
        ConfigStackConfigProviderForHttp stackConfigProvider =
                new ConfigStackConfigProviderForHttp(
                    SipContainer.class.getClassLoader());
     
        stackConfigProvider.configure(HttpLayerHandler.getInstance());
    }
    
    /**
     * Initialize the CLB by registering an AdminContextEventListener. 
     */
    public void initClbAdmin() {
        ClbAdmin.init();
    }
    
    /**
     * Regsiter the RealmAdapter for the SipArchiveDeployer.
     */
    public void registerRealmAdapter() {// throws ServerLifecycleException {
        String SIP_MODULE_TYPE =
                "org.jvnet.glassfish.comms.deployment.backend.SipArchiveDeployer";
        PluggableContextInfo.registerCustomRealmAdapter(SIP_MODULE_TYPE, "org.jvnet.glassfish.comms.security.RealmAdapter");
    }     
    
    /**
     * Resolve the Sip Binding contexts, from the Configured sip listeners and
     * external address and ports.
     */
    public void resolveSipBindings() {
        SipBindingResolver.instance().init();
    }    

    /**
     * Initialize the Sip container LayerHandler (sip stack) from the configured
     * stack configuration or the default stack configuration.
     * @throws org.jvnet.glassfish.comms.startup.stack.StackConfigException
     */
    public void initLayerHandler()
        throws StackConfigException {
        if (log.isLoggable(Level.FINE)) {
            log.log(Level.FINE, "Initializing the layer handler");
        }

        // Create SIP Stack
        StackConfig.ensureStackConfig();
        ConfigStackConfigProvider stackConfigProvider = 
                new ConfigStackConfigProvider(
                SipServiceListener.class.getClassLoader());
        stackConfigProvider.configure(_layerHandler);
    }
    
    /**
     * Register for reconfiguration event.
     */
    public void registerConfig() {
        ConfigAdapter.instance().register();        
    }
    
    /**
     * Register the Feature name that is needed to establish whether this is an
     * SailFin instance.
     */
    public void registerFeatureName() {
        // set a feature name that can be used via AMX
        // to check if its a SailFin instance
        SystemInfoFactory.getInstance().addFeature("COMMS_APPSERV_FEATURE", true);        
    }
    
    /**
     * Initialize monitoring for the Sip stack layers. Note that registeration
     * for monitoring is done outside the SipContainer object. 
     * @see SipMonitoringManagerImpl.
     */
    public void initMonitoring() {
        if (log.isLoggable(Level.FINE)) {
            log.fine("configuring monitoring");
        }

        MonitoringStartup monitoringStartup = new MonitoringAdapter();
        monitoringStartup.startup();        
    }
    
    /*
     * Bind the global SipFactoryFacade, SipSessionsUtil and TimerService to 
     * JNDI.
     */
    public void bindSipResources() {
        SipFactoryFacade sipFactory = new SipFactoryFacade();

        SipFactoryMap.getInstance()
        .addSipFactoryFacade(GLOBAL_SIP_FACTORY_JNDI_NAME, sipFactory);

    	
        publishResourceReference(SipFactory.class.getName(),
            "org.jvnet.glassfish.comms.deployment.backend.SipFactoryObjectFactory",
            GLOBAL_SIP_FACTORY_JNDI_NAME);

        publishResourceReference(SipSessionsUtil.class.getName(),
            "org.jvnet.glassfish.comms.deployment.backend.SSUObjectFactory",
            GLOBAL_SESSIONS_UTIL_JNDI_NAME);

        publishResourceReference(TimerService.class.getName(),
            "org.jvnet.glassfish.comms.deployment.backend.TimerServiceObjectFactory",
            GLOBAL_TIMER_SERVICE_JNDI_NAME);
    }
    
    /**
     * Register those Sip stack layers that are valves and the 
     * ApplicationDispatcher as ContainerListeners at the virtual server
     * @param host
     * @throws java.lang.Exception
     */
    public void registerAtVirtualServer(VirtualServer host)
        throws Exception {        
        // Reconfigure the host with our own Context config class
        host.setConfigClass(com.ericsson.ssa.config.Constants.CONFIG_CLASS);

        for (Layer layer : _layerHandler.getLayers()) {
            if (layer instanceof Valve) {
                if (layer != ApplicationDispatcher.getInstance()) {
                    host.addValve((Valve) layer); // TODO: check that it is correct to add the valves to 'host' (qbinjoe)
                }
            }
        }

        ApplicationDispatcher appDispatcher = ApplicationDispatcher.getInstance();
        host.addContainerListener(appDispatcher);
    }
    
    /**
     * Starts the SIP stack in a new Thread. Calls start on each layer that has
     * a start method.
     */
    public void startSipStack() {
        Runnable firstRunnableLayer = null;
        Iterator<Layer> i = _layerHandler.getLayers().iterator();

        while (i.hasNext()) {
            Layer l = i.next();

            if ((firstRunnableLayer == null) && l instanceof Runnable) {
                firstRunnableLayer = (Runnable) l;
            }

            try {
                Method m = l.getClass().getMethod("start", new Class[0]);
                m.invoke(l, new Object[0]);
            } catch (NoSuchMethodException ignore) {
                // No start method defined for that layer
            } catch (Exception e) {
                if (log.isLoggable(Level.SEVERE)) {
                    log.log(Level.SEVERE, "sip.integration.sip_stack_initialization_failure_abort", "Failed to start layer: " + l);
                    log.log(Level.SEVERE, e.getMessage(), e);
                }

                throw new RuntimeException(e);
            }
        }

        if (firstRunnableLayer != null) {
            es.execute(firstRunnableLayer);
        } else {
            if (log.isLoggable(Level.SEVERE)) {
                log.log(Level.SEVERE, "sip.integration.sip_stack_initialization_failure_abort", "No startable layer found!");
            }

            stopSipStack();
        }
    }

    /**
     * Report the that the SipStack has started and is ready to server request.
     */
    public void reportReady() {
        if (log.isLoggable(Level.INFO)) {
            log.info("Sip Module started and available.");            
        }
        if (log.isLoggable(Level.FINE)) {
            log.fine("Sip Module uses these bindings: \n"+
                    SipBindingResolver.instance().reportBindings());
        }    
    } 
    
    /**
     * Stops the SIP stack. Calls stop on each layer if it has a stop method.
     */
    public void stopSipStack() {
        try {
            List<Object> objectsToStop = new ArrayList<Object>();

            for (Layer l : _layerHandler.getLayers()) {
                objectsToStop.add(l);
            }

            for (Object o : objectsToStop) {
                try {
                    Method m = o.getClass().getMethod("stop", new Class[0]);
                    m.invoke(o, new Object[0]);
                } catch (NoSuchMethodException ignore) {
                    // No start method defined for that layer
                } catch (Exception e) {
                    if (log.isLoggable(Level.WARNING)) {
                        log.log(Level.WARNING, "sip.integration.sip_stack_teardown_failure_continue", "Failed to stop layer: " + o);
                        log.log(Level.WARNING, e.getMessage(), e);
                    }
                }
            }
        } catch (Exception e) {
            if (log.isLoggable(Level.WARNING)) {
                log.log(Level.WARNING, "sip.integration.sip_stack_teardown_failure_continue", "Reason unknown.");
                log.log(Level.WARNING, e.getMessage(), e);
            }
        } finally {
            if (es != null) {
                es.shutdown();
            }
        }
    }              

    private void publishResourceReference(String resourceInterface,
        String objectFactory, String mappedName) {
        try {
            Reference ref = new Reference(resourceInterface,
                    new StringRefAddr("url", mappedName), objectFactory, null);

            Switch.getSwitch().getNamingManager().publishObject(mappedName, ref, true);
        } catch (NamingException ne) {
            if (log.isLoggable(Level.WARNING)) {
        	log.log(Level.WARNING, "Error Binding Resource " + mappedName);
        	log.log(Level.WARNING, ne.getMessage(), ne);
            }
        }
    }    
    
    private void retrieveSipModuleVersion() {
        // Sipmodule version is retreived from sip-module.version
        try {
            Properties p = new Properties();
            p.load(this.getClass().getClassLoader()
                       .getResourceAsStream("sip-module.version"));
            System.setProperty("sip.module.version",
                p.getProperty("SipModuleVersion"));
        } catch (IOException e) {
            if (log.isLoggable(Level.WARNING)) {
                log.log(Level.WARNING, "sip.integration.sip_module_version_lookup_failure_continue", "default");
                log.log(Level.WARNING, e.getMessage(), e);
            }

            System.setProperty("sip.module.version", "Glassfish_SIP_1.0.0");
        } catch (NullPointerException e) {
            if (log.isLoggable(Level.WARNING)) {
                log.log(Level.WARNING, "sip.integration.sip_module_version_lookup_failure_continue", "default");
                log.log(Level.WARNING, e.getMessage(), e);
            }

            System.setProperty("sip.module.version", "Glassfish_SIP_1.0.0");
        }
    }

    private void registerExtensionDeployer() {        
        //register the SipArchiveDeployer for Sip ARchives
        try {
            String AR_DEPLOYER =
                    "org.jvnet.glassfish.comms.deployment.backend.ArArchiveDeployer";
            String AR_LOADER =
                    "org.jvnet.glassfish.comms.deployment.backend.ArArchiveLoader";

            if (log.isLoggable(Level.FINE)) {
                log.fine("Registering the " + AR_DEPLOYER);
                log.fine("Registering the " + AR_LOADER);
            }
            PluggableDeploymentInfo.registerArchiveDeployerLoader(
                    AR_DEPLOYER, AR_LOADER);

            String SIP_DEPLOYER = "org.jvnet.glassfish.comms.deployment.backend.SipArchiveDeployer";
            String SIP_LOADER = "org.jvnet.glassfish.comms.deployment.backend.SipArchiveLoader";
            PluggableDeploymentInfo.registerArchiveDeployerLoader(SIP_DEPLOYER,
                    SIP_LOADER);
            if (log.isLoggable(Level.FINE)) {
                log.fine("Registering the " + SIP_DEPLOYER);
                log.fine("Registering the " + SIP_LOADER);
            }

//        } catch (ServerLifecycleException sle) {
//            if (log.isLoggable(Level.FINE)) {
//                log.fine("ServerLifecyleException thrown :");
//            }
        } catch (java.lang.Exception e) {
            e.printStackTrace();
        }
    }
    
    private void registerExtensionNodes() {
        String SCAS_RUNTIME_TAG = "sun-sip-app";
        String SCAS_RUNTIME_NODE_CLASS = "org.jvnet.glassfish.comms.deployment.runtime.sip.SipBundleRuntimeNode";
        String SCAS_SIP_BUNDLE_TAG = "sip-app";
        String SCAS_SIP_NODE_CLASS ="org.jvnet.glassfish.comms.deployment.node.sip.SipBundleNode";        
   
       PluggableNodeInfo.registerBundleNode(SCAS_RUNTIME_TAG,
            SCAS_RUNTIME_NODE_CLASS);
        if (log.isLoggable(Level.FINE)) {
            log.fine("Registering the " + SCAS_RUNTIME_TAG);
            log.fine("Registering the " + SCAS_RUNTIME_NODE_CLASS);
        }
    
        // register the node for sip.xml
        PluggableNodeInfo.registerBundleNode(SCAS_SIP_BUNDLE_TAG,
            SCAS_SIP_NODE_CLASS);        
        if (log.isLoggable(Level.FINE)) {
            log.fine("Registering the " + SCAS_SIP_BUNDLE_TAG);
            log.fine("Registering the " + SCAS_SIP_NODE_CLASS);
        }

    }

    private void registerExtensionDescriptors() { 
        // get all the descriptors from the HashMap and register them
        for(Iterator iter = tagDescriptorMap.keySet().iterator(); iter.hasNext();) {
            String xmlPath = (String)iter.next();
            DescriptorFactory.register(
                    new XMLElement(xmlPath), 
                     tagDescriptorMap.get(xmlPath));
        }
    }
    
    private void initMapping() {
        tagDescriptorMap = new HashMap <String,Class>();
        tagDescriptorMap.put(SipTagNames.SIP_BUNDLE+"/"+WebTagNames.ICON,
                org.jvnet.glassfish.comms.deployment.backend.SipApplication.class);        
        tagDescriptorMap.put(SipTagNames.SIP_BUNDLE+"/"+SipTagNames.DISTRIBUTABLE,
                org.jvnet.glassfish.comms.deployment.backend.SipApplication.class);        
        tagDescriptorMap.put(SipTagNames.SIP_BUNDLE+"/"+
                WebTagNames.LISTENER,
                org.jvnet.glassfish.comms.deployment.backend.SipApplication.class); 
        tagDescriptorMap.put(SipTagNames.SIP_BUNDLE+"/"+WebTagNames.LISTENER+"/"+
                WebTagNames.LISTENER_CLASS,
                org.jvnet.glassfish.comms.deployment.backend.SipApplication.class);        
        //servlet
        tagDescriptorMap.put(SipTagNames.SIP_BUNDLE+"/"+SipTagNames.SERVLET,
                org.jvnet.glassfish.comms.deployment.backend.Servlet.class);
        tagDescriptorMap.put(SipTagNames.SIP_BUNDLE+"/"+
                SipTagNames.SERVLET+"/"+WebTagNames.ICON,
                org.jvnet.glassfish.comms.deployment.backend.Servlet.class);
        //servlet-mapping
        tagDescriptorMap.put(SipTagNames.SIP_BUNDLE+"/"+SipTagNames.SERVLET_MAPPING,
                org.jvnet.glassfish.comms.deployment.backend.ServletMapping.class);
        tagDescriptorMap.put(SipTagNames.PATTERN,
                org.jvnet.glassfish.comms.deployment.backend.Pattern.class);
        tagDescriptorMap.put(SipTagNames.AND,
                org.jvnet.glassfish.comms.deployment.backend.AndOperator.class);        
        tagDescriptorMap.put(SipTagNames.OR,
                org.jvnet.glassfish.comms.deployment.backend.OrOperator.class);
        tagDescriptorMap.put(SipTagNames.NOT,
                org.jvnet.glassfish.comms.deployment.backend.NotOperator.class);        
        tagDescriptorMap.put(SipTagNames.EQUALS,
                org.jvnet.glassfish.comms.deployment.backend.EqualOperand.class);
        tagDescriptorMap.put(SipTagNames.EXISTS,
                org.jvnet.glassfish.comms.deployment.backend.ExistsOperand.class);
        tagDescriptorMap.put(SipTagNames.VAR,
                org.jvnet.glassfish.comms.deployment.backend.Operand.class);   
        tagDescriptorMap.put(SipTagNames.VALUE,
                org.jvnet.glassfish.comms.deployment.backend.Operand.class);         
        tagDescriptorMap.put(SipTagNames.CONTAINS,
                org.jvnet.glassfish.comms.deployment.backend.ContainsOperand.class);
        tagDescriptorMap.put(SipTagNames.SUBDOMAINOF,
                org.jvnet.glassfish.comms.deployment.backend.SubDomainOfOperand.class); 
        //proxy-config
        tagDescriptorMap.put(SipTagNames.PROXY_CONFIG,
                org.jvnet.glassfish.comms.deployment.backend.SipApplication.class);
        tagDescriptorMap.put(SipTagNames.SEQUENTIAL_SEARCH_TIMEOUT,
                org.jvnet.glassfish.comms.deployment.backend.SipApplication.class);        
        //session-config
        tagDescriptorMap.put(SipTagNames.SESSION_CONFIG,
                org.jvnet.glassfish.comms.deployment.backend.SipApplication.class);
        //security-constraint
        tagDescriptorMap.put(SipTagNames.SIP_BUNDLE+"/"+WebTagNames.SECURITY_CONSTRAINT,
                org.jvnet.glassfish.comms.deployment.backend.SecurityConstraint.class);
        tagDescriptorMap.put(SipTagNames.SIP_BUNDLE+"/"+
                WebTagNames.SECURITY_CONSTRAINT+"/"+
                SipTagNames.DISPLAY_NAME,
                org.jvnet.glassfish.comms.deployment.backend.SecurityConstraint.class);        
        tagDescriptorMap.put(SipTagNames.SIP_BUNDLE+"/"+
                WebTagNames.SECURITY_CONSTRAINT+"/"+
                SipTagNames.PROXY_AUTHENTICATION,
                org.jvnet.glassfish.comms.deployment.backend.SecurityConstraint.class);        
        tagDescriptorMap.put(SipTagNames.SIP_BUNDLE+"/"+
                WebTagNames.SECURITY_CONSTRAINT+"/"+
                SipTagNames.SIP_RESOURCE_COLLECTION,
                org.jvnet.glassfish.comms.deployment.backend.ResourceCollection.class);
        tagDescriptorMap.put(SipTagNames.SIP_BUNDLE+"/"+
                WebTagNames.SECURITY_CONSTRAINT+"/"+
                WebTagNames.AUTH_CONSTRAINT,
                org.jvnet.glassfish.comms.deployment.backend.SecurityConstraint.class); 
        tagDescriptorMap.put(SipTagNames.SIP_BUNDLE+"/"+
                WebTagNames.SECURITY_CONSTRAINT+"/"+
                WebTagNames.AUTH_CONSTRAINT+"/"+
                WebTagNames.ROLE_NAME,
                org.jvnet.glassfish.comms.deployment.backend.SecurityConstraint.class);         
        tagDescriptorMap.put(SipTagNames.SIP_BUNDLE+"/"+
                WebTagNames.SECURITY_CONSTRAINT+"/"+
                WebTagNames.USERDATA_CONSTRAINT,
                org.jvnet.glassfish.comms.deployment.backend.SecurityConstraint.class);
        tagDescriptorMap.put(SipTagNames.SIP_BUNDLE+"/"+
                WebTagNames.SECURITY_CONSTRAINT+"/"+
                WebTagNames.USERDATA_CONSTRAINT+"/"+
                WebTagNames.TRANSPORT_GUARANTEE,
                org.jvnet.glassfish.comms.deployment.backend.SecurityConstraint.class);
         tagDescriptorMap.put(SipTagNames.SIP_BUNDLE+"/"+
                WebTagNames.LOGIN_CONFIG,
                org.jvnet.glassfish.comms.deployment.backend.LoginConfigurationImpl.class);
         /*tagDescriptorMap.put(WebTagNames.ROLE,
                org.jvnet.glassfish.comms.deployment.backend.SipApplication.class); */        
       
    }
    
    private void initEarLifecycleListener() {
        boolean result = Boolean.getBoolean("org.jvnet.glassfish.EarLifecycle");
        if (result == false) {
            return;
        }

        Object listener = null;
        try {
            listener = Class.forName
            ("com.sun.enterprise.ee.server.AppLoaderEventListener")
            .newInstance();
            if (log.isLoggable(Level.FINE)) {
                log.log(Level.FINE, "Listener is Loaded");
            }
        }catch ( Exception e) {
            //FIX ME. This should be SEVERE?
            if (log.isLoggable(Level.FINE)) {
                log.log(Level.FINE, e.getMessage(), e);
            }
        }
        if (listener != null) {
            ApplicationLoaderEventNotifier.getInstance().addListener
            (EarLifecycleListener.class.cast(listener));
        }
    }   
}
