/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.jvnet.glassfish.comms.startup.stack;

import com.ericsson.ssa.config.Config;
import com.ericsson.ssa.config.ConfigFactory;
import com.sun.org.apache.commons.modeler.util.IntrospectionUtils;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;

/**
 * A base for all SipContainer internal Config based StackConfigProviders.
 * It defines the general behavior of such a provider and allows several 
 * occasions to influence the this general behavior by implementing or overriding
 * certain method.
 * @author ELNELBO
 */
public abstract class ConfigStackConfigProviderBase<SM extends StackModel, L extends StackLayer> {    
    protected static final List<String> KEYS_TO_OMMIT = 
            Arrays.asList(new String[]{StackConfig.CLASS_NAME_KEY,
            StackConfig.FACTORY_CLASS_NAME_KEY,
            StackConfig.HTTP_LAYER_INDICATOR_KEY}); 
    
    private ClassLoader classLoader;

    /**
     * Construct a ConfigStackConfigProviderBase. Requires a ref to a 
     * classloader, which it needs to lookup the StackLayer or Factory classes
     * from the proper place.
     * @param aClassLoader
     */
    protected ConfigStackConfigProviderBase(ClassLoader aClassLoader) {
        classLoader = aClassLoader;
    }
    
    public void configure(SM sm)
            throws StackConfigException {
        Config config = ConfigFactory.getConfig();
        String layerOrder = config.get(StackConfig.STACK_CONFIG, 
                StackConfig.LAYER_ORDER_KEY);
        List<StackLayer> stackLayers = new ArrayList<StackLayer>(); 
        
        StringTokenizer commaSep = new StringTokenizer(layerOrder, ",");
        
        //Read the config
        while (commaSep.hasMoreTokens()) {
            String id = commaSep.nextToken().trim();
            
            String layerClassName = config.get(StackConfig.STACK_LAYER+"/"+id, 
                    StackConfig.CLASS_NAME_KEY);
            
            if (layerClassName!=null && layerClassName.length()>0) {
                String factoryClassName = config.get(StackConfig.STACK_LAYER+"/"+id, 
                        StackConfig.FACTORY_CLASS_NAME_KEY);
                Map<String, String> properties = config.getAll(
                        StackConfig.STACK_LAYER+"/"+id+StackConfig.PROPERTIES);
                
                if (isAcceptedStackLayer(id, layerClassName, factoryClassName,
                        properties)) {
                    stackLayers.add(new StackLayer(id, layerClassName, 
                            factoryClassName, properties));
                }
            } else {
                throw new StackConfigException("Failed read configuration for stack-layer ["+
                        id+"], check your configuration.");
            }
        }
        
        //Instantiate the layers
        for (StackLayer stackLayer : stackLayers) {
            L layer = stackLayer.resolveLayer();
            if (layer!=null) {
                addLayerToModel(sm, layer);
            }
        }
    }

    /**
     * Override this to filter layers if needed.
     * @param id
     * @param layerClassName
     * @param factoryClassName
     * @param properties
     * @return true if not overridden
     */
    protected boolean isAcceptedStackLayer(String id, String layerClassName, 
            String factoryClassName, Map<String, String> properties) {
        return true;
    }

    /**
     * Add the layer to the StackModel. Override this to allow for other
     * side effects.
     * @param sm
     * @param layer
     * @throws org.jvnet.glassfish.comms.startup.stack.StackConfigException
     */
    protected void addLayerToModel(SM sm, L layer)
            throws StackConfigException {
        sm.addLayer(layer);        
    }
    
    /**
     * Provides the static init method that needs to be called to obtain 
     * a StackLayer instance.
     * @return layer init method
     */
    protected abstract String getLayerInitMethod();
    
    /**
     * Provides the static init method that needs to be called to register a
     * StackLayer class. 
     * a StackLayer instance.
     * @return factory init method
     */
    protected abstract String getFactoryInitMethod();
    
    /**
     * Is Layer Factorization Allowed.
     * @return true if not overridden
     */
    protected abstract boolean isLayerFactorizationAllowed();
    
    /**
     * Is Static Property Injection Allowed.
     * @return true if not overridden
     */
    protected abstract boolean isStaticPropertyInjectionAllowed();
    
    /**
     * Is Config Property Injection Allowed.
     * @return true if not overridden
     */
    protected abstract boolean isConfigPropertyInjectionAllowed();
    
    class StackLayer { 
        private String id;
        private String className;
        private String factoryClassName;        
        private Map<String, String> properties;
        
        public StackLayer(String anId, String aClassName) {
            this(anId, aClassName, null, null);
        }
        
        public StackLayer(String anId, String aClassName,
                Map<String, String> aProperties) {
            this(anId, aClassName, null, aProperties);
        }
         
        public StackLayer(String anId, String aClassName,
                String aFactoryClassName, Map<String, String> aProperties) {
            id = anId;
            className = aClassName;
            factoryClassName = aFactoryClassName;
            properties = aProperties;
        }
        
        public L resolveLayer() throws StackConfigException {
            L layer = null;
            
            try {
                Class layerClass = Class.forName(className, true, classLoader);
                
                if (isLayerFactorizationAllowed() && factoryClassName!=null) {
                    registerLayerClassInFactory(layerClass, factoryClassName);
                }
                
                Method method = layerClass.getMethod(getLayerInitMethod(),
                        (Class[]) null);
                layer = (L) method.invoke((Object) null, (Object[]) null);                
                
                //Inject the properties from the StackConfig
                if (isStaticPropertyInjectionAllowed() && properties != null) {
                    for (Map.Entry<String, String> e : properties.entrySet()) {
                        if (!KEYS_TO_OMMIT.contains(e.getKey())) {
                            injectProperty(layer, e.getKey(), e.getValue());
                        }
                    }
                }
                
                //Let Config override the properties from the StackConfig.                    
                if (isConfigPropertyInjectionAllowed()) {
                    ConfigFactory.instance().activateConfiguration(layer);                
                }
            } catch (Exception ex) {
                throw new StackConfigException("Failed to instantiate Layer["+
                        id+"] with className: "+className, ex);
            }
            
            return layer;
        }

        private void registerLayerClassInFactory(Class layerClass,
                String factoryClassName)
                throws ClassNotFoundException, 
                NoSuchMethodException, 
                IllegalAccessException, 
                InvocationTargetException {
            Class factoryClass = Class.forName(factoryClassName, true, classLoader);
            Class<?>[] paramTypes = { Class.class };
            Object[] paramObjects = { layerClass };
            Method initInstance = factoryClass.getMethod(getFactoryInitMethod(), 
                    paramTypes);
            initInstance.invoke(factoryClass, paramObjects);
        }
        
        private void injectProperty(L layer, String propertyName,
                String propertyValue) throws StackConfigException {
            try {
                IntrospectionUtils.setProperty(layer, propertyName, propertyValue);
            } catch (Exception ex) {
                //Apply a fail fast. To allow direct feedback
                throw new StackConfigException("Failed to infuse property '"+
                        propertyName+"' in Layer with className: "+className, ex);
            }
        }        
    }
}
