/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.jvnet.glassfish.comms.replication.dialogmgmt;

import com.ericsson.ssa.sip.DialogFragment;
import com.ericsson.ssa.sip.DialogFragmentManager;
import com.ericsson.ssa.sip.DialogFragmentStore;
import com.ericsson.ssa.sip.RemoteLockException;
import com.sun.appserv.ha.spi.BackingStore;
import com.sun.appserv.ha.spi.BackingStoreException;
import com.sun.appserv.ha.spi.SimpleMetadata;
import com.sun.enterprise.ee.web.sessmgmt.JxtaBackingStoreImpl;
import com.sun.enterprise.ee.web.sessmgmt.JxtaReplicationSender;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationUtil;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationState;
import com.sun.enterprise.ee.web.sessmgmt.ReplicationUtil;
import com.sun.enterprise.ee.web.sessmgmt.StorePoolElement;
import java.io.*;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.catalina.*;
import org.apache.catalina.session.IOUtilsCaller;
import org.apache.catalina.session.WebIOUtilsFactory;
import org.jvnet.glassfish.comms.replication.sessmgmt.*;
import org.jvnet.glassfish.comms.util.LogUtil;

/**
 *
 * @author HA team
 */
public class DialogFragmentStoreImpl implements StorePoolElement, DialogFragmentStore {
    
    private static final Logger _logger = LogUtil.SSR_LOGGER.getLogger();
    protected final static String MODE_SIP = ReplicationState.MODE_SIP;    
    
    /** Creates a new instance of DialogFragmentStoreImpl */
    public DialogFragmentStoreImpl() {
    }

    /**
     * gets the BackingStore used for replicating DialogFragments
     *
     * @return the BackingStore initialized for DialogFragments
     *
     */    
    protected BackingStore getBackingStore() {
        return getDialogFragmentManager().getDialogFragmentBackingStore();
    }     

    /**
     * Loads and returns the DialogFragment with the given id from
     * this store.
     *
     * @param id The id of the DialogFragment to load
     *
     * @return The DialogFragment with the given id, or null if not
     * found
     *
     * @throws IOException
     * @throws RemoteLockException
     */
    public DialogFragment load(String id)
        throws IOException, RemoteLockException {
        return load(id, null);
    }

    /**
     * Loads and returns the DialogFragment with the given id from
     * this store.
     *
     * @param id The id of the DialogFragment to load
     * @param version The version of the DialogFragment to load
     *
     * @return The DialogFragment with the given id, or null if not
     * found
     *
     * @throws IOException
     * @throws RemoteLockException
     */
    public DialogFragment load(String id, String version) 
        throws IOException, RemoteLockException {   
        HADialogFragment result = null;
        if(id == null) {
            return result;
        }
        ReplicationDialogFragmentManager repMgr 
            = (ReplicationDialogFragmentManager)DialogFragmentManager.getInstance();
        ReplicationState localCachedState 
            = repMgr.removeFromDialogFragmentReplicationCache(id);
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("DialogFragmentStoreImpl>>load:localCachedState=" + localCachedState);                       
        }
        //check if we got a hit from our own replica cache
        //and if so return it immediately
        if(version != null && localCachedState != null) {
            long versionLong = ReplicationUtil.parseLong(version);
            try {
                if(localCachedState.getVersion() == versionLong) {
                    result = this.getDialogFragment(localCachedState);
                    result.validate();
                    //save result - save will reset dirty to false
                    ((HADialogFragment)result).setDirty(true, false);
                    save(result);
                    return result;
                } 
            } catch (ClassNotFoundException ex) {
                IOException ex1 = 
                    (IOException) new IOException("Error during load: " + ex.getMessage()).initCause(ex);
                throw ex1;                
            }
        }        
        ReplicationState broadcastResultState = findDialogFragmentViaBroadcast(id, version);
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("DialogFragmentStoreImpl>>load:broadcastResultState from broadcast=" + broadcastResultState);                       
        }        
        ReplicationState bestState 
            = ReplicationState.getBestResult(localCachedState, broadcastResultState);
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("DialogFragmentStoreImpl>>load:bestState=" + bestState);                       
        }
        if(bestState != null) {
            if(_logger.isLoggable(Level.FINE)) {
                _logger.fine("DialogFragmentStoreImpl>>load:bestStateVERSION=" + bestState.getVersion());
                _logger.fine("DialogFragmentStoreImpl>>load:bestStateSTATE=" + bestState.getState());
            }            
        }

        if(bestState != null && bestState.getState() != null) {
            if(_logger.isLoggable(Level.FINE)) {
                _logger.fine("DialogFragmentStoreImpl>>load:before deserializing bestState:ver=" + bestState.getVersion());                       
            }            
            try {
                result = this.getDialogFragment(bestState); 
            } catch (ClassNotFoundException ex) {
                IOException ex1 = 
                    (IOException) new IOException("Error during load: " + ex.getMessage()).initCause(ex);
                throw ex1;                
            }                        
            if(_logger.isLoggable(Level.FINE)) {
                _logger.fine("DialogFragmentStoreImpl>>load:after deserializing dialog fragment:ver=" + result.getVersion());                       
            }
            // For DialogFragment this sending of load received ack
            // has been postponed to this point so we can check the
            // deserialized result to see if it was locked on the remote side
            // For most other artifacts the load received ack is done
            // immediately in JxtaBackingStoreImpl
            result.validate();
            //save result - save will reset dirty to false
            ((HADialogFragment)result).setDirty(true, false);
            save(result);            
            // Send acknowledgement of load receipt
            // No response to wait for in this case
            JxtaReplicationSender sender 
                = JxtaReplicationSender.createInstance();
            ReplicationState loadReceivedState = 
                ReplicationState.createBroadcastLoadReceivedState(
                    MODE_SIP, (String)bestState.getId(), 
                    getApplicationId(), bestState.getVersion(),
                    ReplicationUtil.getInstanceName(),
                    ReplicationDialogFragmentManager.MESSAGE_BROADCAST_LOAD_RECEIVED_DIALOG_FRAGMENT);
            sender.sendBroadcastQuery(loadReceivedState);
        }
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("DialogFragmentStoreImpl>>load: " + result);                       
        }         
        return result;
    }
    
    private ReplicationState findDialogFragmentViaBroadcast(String id, String version) 
        throws IOException {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("DialogFragmentStoreImpl>>findDialogFragmentViaBroadcast");                       
        }        
        BackingStore replicator = this.getBackingStore();
        JxtaBackingStoreImpl jxtaReplicator = null;
        if(replicator instanceof JxtaBackingStoreImpl) {
            jxtaReplicator = (JxtaBackingStoreImpl)replicator;
        }
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("DialogFragmentStoreImpl>>findDialogFragmentViaBroadcast: replicator: " + replicator);                       
        }
        
        SimpleMetadata queryResult = null;
        try {
            //use version aware load if possible
            if(jxtaReplicator != null && version != null) {
                queryResult = (SimpleMetadata)jxtaReplicator.load(id, version);
            } else {
                queryResult = (SimpleMetadata)replicator.load(id);
            }
        } catch (BackingStoreException ex) {
            IOException ex1 = 
                (IOException) new IOException("Error during load: " + ex.getMessage()).initCause(ex);
            throw ex1;
        }
        if(queryResult == null) {
            return null;
        }
        //queryResult.getExtraParam() here is expiration time
        byte[] extraParamState 
            = ReplicationUtil.getByteArray((Serializable) queryResult.getExtraParam());
        ReplicationState result 
            = new ReplicationState(MODE_SIP, id, this.getApplicationId(), 
                queryResult.getVersion(), 0L, 0L, null, null, 
                null, null, queryResult.getState(), null, extraParamState);
        return result;
    }
    
    /**
    * Given a byte[] containing DialogFragment data, return a DialogFragment
    * object
    *
    * @param state
    *   The byte[] with dialog fragment data
    *
    * @return
    *   A newly created DialogFragment for the given session data, and associated
    *   with this Manager
    */
    protected HADialogFragment getDialogFragment(ReplicationState replicationState) 
        throws IOException, ClassNotFoundException 
    {
        byte[] state = replicationState.getState();
        HADialogFragment dialogFragment = null;
        BufferedInputStream bis = null;
        ByteArrayInputStream bais = null;
        Loader loader = null;    
        ClassLoader classLoader = null;
        ObjectInputStream ois = null;
        long version = 0L;
        IOUtilsCaller utilsCaller = null;
            
        bais = new ByteArrayInputStream(state);
        bis = new BufferedInputStream(bais);

        String id = (String)replicationState.getId();
        //int expirationTime = ReplicationUtil.parseInt(replicationState.getExtraParam());
        version = replicationState.getVersion();    

        if(_logger.isLoggable(Level.FINEST)) {
            _logger.finest("loaded session from replicationstore, length = "+state.length);
        }
        ois = new ObjectInputStream(bis); 
        if(ois != null) {               
            try {
                dialogFragment = (HADialogFragment)ois.readObject();
            } 
            finally {
                if (ois != null) {
                    try {
                        ois.close();
                        bis = null;
                    }
                    catch (IOException e) {
                    }
                }
            }
        }       
        byte[] extraParamsState 
            = replicationState.getContainerExtraParamsState();
        dialogFragment.update(extraParamsState);        
        //currentOwnerInstanceName is in the manager already
        dialogFragment.setVersion(version);
        dialogFragment.setInternalLastAccessedTime(System.currentTimeMillis());
        dialogFragment.setDirty(false);
        dialogFragment.setReplicated(false);        
        return dialogFragment;
    }        

    /**
     * Saves the given DialogFragment to this store.
     *
     * If a DialogFragment with the same id already exists in this session
     * store, it will be replaced.
     *
     * @param session The DialogFragment to be saved
     *
     * @exception IOException
     */
    public void save(DialogFragment dialogFragment) throws IOException {
        HADialogFragment haDialogFragment = (HADialogFragment)dialogFragment;
        long previousVersion = haDialogFragment.getVersion();
        haDialogFragment.incrementVersion();
        haDialogFragment.setInternalLastAccessedTime(System.currentTimeMillis());
        try {
            if( haDialogFragment.isReplicated() && !haDialogFragment.isDirty() ) {
                this.updateContainerExtraParam(haDialogFragment);
            } else {
                haDialogFragment.setIsBeingReplicated(true);
                this.doSave(haDialogFragment);
                haDialogFragment.setReplicated(true);
            }
            haDialogFragment.setDirty(false);
        } catch (IOException ex) {
            haDialogFragment.setVersion(previousVersion);
            throw ex;
        } finally {
            haDialogFragment.setIsBeingReplicated(false);
        }
    }  

    /**
     * Updates the containerExtraParam of the specified DialogFragment into
     * this Store.
     *
     * @param df DialogFragment to be saved
     *
     * @exception IOException if an input/output error occurs
     */
    public void updateContainerExtraParam(HADialogFragment df)
            throws IOException {
        BackingStore replicator = this.getBackingStore();
        JxtaBackingStoreImpl jxtaReplicator = null;
        if(replicator instanceof JxtaBackingStoreImpl) {
            jxtaReplicator = (JxtaBackingStoreImpl)replicator;
        }
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("DialogFragmentStoreImpl>>updateContainerExtraParam: replicator: " + replicator);
        }
        try {
            replicator.updateContainerExtraParam(df.getDialogId(), //id
                    df.getInternalLastAccessedTime(), //lastaccesstime
                    df.getVersion(), //version
                    df.getExtraParameters());  //containerExtraParams
        } catch (BackingStoreException ex) {
            IOException ex1 = (IOException)
                new IOException("Error during updateContainerExtraParam: " +
                                ex.getMessage()).initCause(ex);
            throw ex1;
        }
    }

    /**
     * Saves the given DialogFragment to this store.
     *
     * If a DialogFragment with the same id already exists in this session
     * store, it will be replaced.
     *
     * @param session The DialogFragment to be saved
     *
     * @exception IOException
     */
    public void doSave(HADialogFragment haDialogFragment) throws IOException {
        byte[] sessionState = ReplicationUtil.getByteArray(haDialogFragment);
        BackingStore replicator = this.getBackingStore();
        SimpleMetadata simpleMetadata =        
            new SimpleMetadata(
                haDialogFragment.getVersion(),
                haDialogFragment.getInternalLastAccessedTime(), // internallastAccessedTime
                0L, // maxinactiveinterval
                sessionState,
                haDialogFragment.getExtraParameters() // containerExtraParam
            );
                
        try {        
            replicator.save(haDialogFragment.getDialogId(), //id
                    simpleMetadata);  //SimpleMetadata 
        } catch (BackingStoreException ex) {
            IOException ex1 = 
                (IOException) new IOException("Error during save: " + ex.getMessage()).initCause(ex);
            throw ex1;
        } 
    }

    /**
     * Removes the DialogFragment with the given id from this store.
     *
     * @param id The id of the DialogFragment to be removed
     *
     * @exception IOException
     */
    public void remove(String id) throws IOException {
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("DialogFragmentStoreImpl>>remove" + " id: " + id);                       
        } 
        BackingStore replicator = this.getBackingStore();
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("DialogFragmentStoreImpl>>remove: replicator: " + replicator);                       
        }        
        try {
            replicator.remove(id);
        } catch (BackingStoreException ex) {
            IOException ex1 = 
                (IOException) new IOException("Error during remove: " + ex.getMessage()).initCause(ex);
            throw ex1;       
        }
    } 
    
    /**
     * Removes all DialogFragments from this store.
     */
    public void clear() throws IOException {
        //FIXME
    }

    /**
     * clean up resources
     */
    public void cleanup() {
        //FIXME        
    }

    protected String getApplicationId() {
        return this.getDialogFragmentManager().getApplicationId();
    } 
    
    /**
     * Gets the session manager of this store.
     *
     * @return The session manager of this store
     */
    public ReplicationDialogFragmentManager getDialogFragmentManager() {
        return (ReplicationDialogFragmentManager)com.ericsson.ssa.sip.DialogFragmentManager.getInstance();
    }

    /**
     * Removes any expired DialogFragments from this store.
     *
     * This method is invoked by the background reaper thread.
     */
    public void processExpires() {
        ReplicationDialogFragmentManager dialogFragmentMgr =
            getDialogFragmentManager();
        if(!(dialogFragmentMgr.isThirdPartyBackingStoreInUse())) {
            dialogFragmentMgr.processExpiredDialogFragmentReplicas();
        } else {
            dialogFragmentMgr.processExpiredDialogFragmentReplicasThirdPartySPI();
        }
        dialogFragmentMgr.processExpiredReplicaMonitors();
        dialogFragmentMgr.processExpiredReplicaRemovals();
    }
    
    void sendRollingUpgradeAdvisory(String instanceName) 
        throws IOException {    
        //send rolling upgrade advisory to instanceName
        ReplicationDialogFragmentManager mgr
            = (ReplicationDialogFragmentManager)getDialogFragmentManager();
        String theCommand = mgr.MESSAGE_BROADCAST_ROLLING_UPGRADE_ADVISORY;
        ReplicationState loadAdvisoryState 
            = ReplicationState.createUnicastLoadAdvisoryState(MODE_SIP, "id", mgr.getApplicationId(), 0L, ReplicationUtil.getInstanceName(), theCommand);     
        JxtaReplicationSender sender 
            = JxtaReplicationSender.createInstance();
        sender.sendOverPropagatedPipe(loadAdvisoryState, instanceName, false);

    }     

    /**
     * Removes any DialogFragment replicas that have expired
     *
     * @return the number of expired DialogFragment replicas that were
     * removed
     */    
    public int removeExpiredDialogFragments() {
        int result = 0;
        BackingStore replicator = getBackingStore();
        try {
            result = replicator.removeExpired();
        } catch (BackingStoreException ex) {
             _logger.log(Level.WARNING, "df_remove_expired_error");
             _logger.log(Level.WARNING, ex.getMessage(), ex);
        }
        if(_logger.isLoggable(Level.FINE)) {
            _logger.fine("DiakogFragmentStoreImpl>>removeExpiredDialogFragments():number of expired dialog fragments = " + result);
        }
        return result;
    } 
}
