/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.jvnet.glassfish.comms.replication.sessmgmt;

import com.ericsson.ssa.sip.SipApplicationSessionImpl;
import java.io.*;

/**
 * Class containing the extra parameters for a ServletTimer.
 *
 * Extra parameters are parameters that are likely to change very
 * frequently. Rather than having to serialize the entire ServletTimer state
 * when any of the extra parameters change, the extra parameters are
 * serialized and replicated separately from the ServletTimer's state.
 *
 * When a ServletTimer is deserialized on a replica instance, its state is
 * overlayed with the latest version of the corresponding extra parameters.
 *
 * @author jluehe
 * @author lwhite
 */
public class ServletTimerExtraParams implements Serializable {
    
    private static final long serialVersionUID = 1L;

    /** The serialized format versioning. 1 = first version. */
    private static short serializedFormVersion  = 1;

    private transient String appId;
    private transient String parentSasId;
    private transient String currentOwnerInstanceName;
    private transient long expirationTime;
    private transient long internalLastAccessedTime;

    // The associated ServletTimer
    private transient final HAServletTimer timer;

    // The id of the associated ServletTimer
    private transient String id;

    private transient int hc;

    private enum Mode {NON_QUERY_MODE, FIND_BY_OWNER, FIND_BY_APPID, 
                       FIND_BY_APPID_AND_OWNER};

    private Mode mode = Mode.NON_QUERY_MODE;

    /**
     * Constructor
     *
     * @param timer The associated ServletTimer
     */
    public ServletTimerExtraParams(HAServletTimer timer) {
        this.timer = timer;
        //timer can be null for a criteria object
        if(timer != null) {
            this.id = timer.getId();
            this.hc = id.hashCode();
        }
    }   
    
    static ServletTimerExtraParams createSearchCriteria(String currentOwnerInstanceName) {
        ServletTimerExtraParams queryTemplate = new ServletTimerExtraParams(null);
        queryTemplate.mode = Mode.FIND_BY_OWNER;
        queryTemplate.currentOwnerInstanceName = currentOwnerInstanceName;
        return queryTemplate;
    }
    
    static ServletTimerExtraParams createSearchCriteriaAppid(String appId) {
        ServletTimerExtraParams queryTemplate = new ServletTimerExtraParams(null);
        queryTemplate.mode = Mode.FIND_BY_APPID;
        queryTemplate.appId = appId;
        return queryTemplate;
    }    

    static ServletTimerExtraParams createSearchCriteria(
            String appId, String currentOwnerInstanceName) {
        ServletTimerExtraParams queryTemplate = new ServletTimerExtraParams(null);
        queryTemplate.mode = Mode.FIND_BY_APPID_AND_OWNER;
        queryTemplate.appId = appId;
        queryTemplate.currentOwnerInstanceName = currentOwnerInstanceName;
        return queryTemplate;
    }    

    public String getId() {
        return id;
    }
    
    public String getAppId() {
        if (timer != null) {
            return timer.getAppId();
        } else {
            return appId;
        }               
    }     

    public String getParentSasId() {
        if (timer != null) {
            return timer.getParentSASId();
        } else {
            return parentSasId;
        }
    }
    
    public String getCurrentOwnerInstanceName() {
        if (timer != null) {
            return timer.getCurrentOwnerInstanceName();
        } else {
            return currentOwnerInstanceName;
        }
    }

    public long getExpirationTime() {
        if (timer != null) {
            return timer.scheduledExecutionTime();
        } else {
            return expirationTime;
        }
    }
    
    public long getInternalLastAccessedTime() {
        if (timer != null) {
            return timer.getInternalLastAccessedTime();
        } else {
            return internalLastAccessedTime;
        }               
    }    

    public boolean equals(Object obj) {
         switch (this.mode) {
         case FIND_BY_OWNER:
            //Search mode
            if (obj instanceof ServletTimerExtraParams) {
                ServletTimerExtraParams that = (ServletTimerExtraParams) obj;
                return getCurrentOwnerInstanceName().equals(that.currentOwnerInstanceName);
            } else {
                return false;
            }
         case FIND_BY_APPID:
            //Search mode
            if (obj instanceof ServletTimerExtraParams) {
                ServletTimerExtraParams that = (ServletTimerExtraParams) obj;
                return getAppId().equals(that.appId);
            } else {
                return false;
            }
         case FIND_BY_APPID_AND_OWNER:
            if (obj instanceof ServletTimerExtraParams) {
                ServletTimerExtraParams that = (ServletTimerExtraParams) obj;
                return getAppId().equals(that.appId) &&
                    getCurrentOwnerInstanceName().equals(that.currentOwnerInstanceName);
            } else {
                return false;
            }
         default:
            if (obj == this) {
                return true;
            }
    
            if (! (obj instanceof ServletTimerExtraParams)) {
                return false;
            }
    
            ServletTimerExtraParams that = (ServletTimerExtraParams) obj;
            return this.id.equals(that.id);
        }
    }

    public int hashCode() {
        if (mode != Mode.NON_QUERY_MODE) {
            throw new IllegalStateException("illegal use of criteria extra param");
        }
        return hc;
    }

    /**
     * @serialData See serialized form version 1 in #readObject(ObjectInputStream in)
     *  
     * @param oos the stream to write the object members
     * @throws IOException 
     * @throws ClassNotFoundException
     */    
    private void writeObject(ObjectOutputStream oos)
            throws IOException {
        oos.writeShort(serializedFormVersion);
        
        oos.writeObject(id);
        oos.writeObject(appId);
        oos.writeObject(getParentSasId());
        oos.writeObject(getCurrentOwnerInstanceName());
        oos.writeLong(getExpirationTime());
        oos.writeLong(getInternalLastAccessedTime());
    }

    /**
     * @serialData first field is an short and represents the serializedFormVersion.<br>
     * <h3>Data layout for serializedFormVersion = 1 follows</h3>
     * 
     * <li>field is a <b>String</b> and represents the servletTimerId field</li>
     * <li>field is a <b>String</b> and represents the applicationId field</li>     
     * <li>field is a <b>String</b> and represents the sipApplicationSessionId field</li> 
     * <li>field is a <b>String</b> and represents the name of the owner instance when this instance was serialized</li>
     * <li>field is a <b>Long</b> and represents the expirationTime field as absolute time</li>
     * <li>field is a <b>Long</b> and represents the internalLastAccessedTime field as absolute time</li>
     *  
     * @param in the stream to read the object members
     * @throws IOException is thrown when unsupported version is detected
     * @throws ClassNotFoundException
     */    
    private void readObject(ObjectInputStream in)
            throws IOException, ClassNotFoundException {
        short readSerializedFormVersion = in.readShort();
        switch(readSerializedFormVersion) {
        case 1:
            id = (String) in.readObject();
            hc = id.hashCode();
            appId = (String) in.readObject();
            parentSasId = (String) in.readObject();
            currentOwnerInstanceName = (String) in.readObject();
            expirationTime = in.readLong();
            internalLastAccessedTime = in.readLong();
        break;
        default:
            throw new IOException("Unable to deserialize into "
                    + this.getClass().getName()
                    + " due to unknown serializedFormVersion of "
                    + readSerializedFormVersion);
        }
    }
}
