/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package com.ericsson.ssa.config;

import com.ericsson.ssa.dd.ConvergedDescriptor;
import org.jvnet.glassfish.comms.deployment.backend.SipApplication;

import com.sun.enterprise.config.serverbeans.SipContainerAvailability;
import com.sun.enterprise.deployment.WebBundleDescriptor;
import com.sun.enterprise.deployment.runtime.web.SessionManager;
import com.sun.enterprise.web.Constants;
import com.sun.enterprise.web.SessionManagerConfigurationHelper;
import com.sun.enterprise.web.WebModule;
import com.sun.enterprise.web.WebModuleConfig;
import com.sun.enterprise.web.session.PersistenceType;

import org.jvnet.glassfish.comms.deployment.backend.SipBundleDescriptor;
import org.jvnet.glassfish.comms.deployment.backend.SipApplication;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.jvnet.glassfish.comms.util.LogUtil;

/**
 * Helper class for determining the persistence type, frequency, and scope
 * for the HTTP and SIP components of a converged application.
 *
 * @author jluehe
 */
public class ConvergedSessionManagerConfigurationHelper
    extends SessionManagerConfigurationHelper {
    private static final Logger SIP_LOGGER = LogUtil.SIP_LOGGER.getLogger();
    private PersistenceType sipPersistenceType = PersistenceType.MEMORY;
    private String sipPersistenceFrequency = null;
    private String sipPersistenceScope = null;
    private ConvergedServerConfigLookup lookup = null;
    private SessionManager sipSmBean = null;

    /**
     * Constructor.
     */
    public ConvergedSessionManagerConfigurationHelper(WebModule ctx,
        SessionManager httpSmBean, SessionManager sipSmBean,
        WebBundleDescriptor wbd, WebModuleConfig wmInfo) {
        super(ctx, httpSmBean, wbd, wmInfo);
        lookup = new ConvergedServerConfigLookup();
        this.sipSmBean = sipSmBean;
    }

    /**
     * Determines the persistence type, frequency, and scope for the HTTP and
     * SIP components of the converged application that was passed to the
     * constructor, taking into account configuration information from both
     * domain.xml and the application's deployment descriptors.
     *
     * This method also ensures that if in-memory replication has been turned
     * on for SIP, it must also have been turned on for HTTP, and vice versa.
     * Otherwise, we fall back to memory-only for both SIP and HTTP.
     */
    public void initializeConfiguration() {

        boolean isSipAppDistributable = false;
        SipBundleDescriptor sbd = null;
        if (_wbd instanceof SipBundleDescriptor) {
            sbd = (SipBundleDescriptor) _wbd;
            SipApplication sipApp = sbd.getSipApplication();

            if (sipApp != null) {
                isSipAppDistributable = ((SipApplication)sipApp).isDistributable();
            }

            if (sbd.isConverged()) {
                // Determine persistence configuration for HTTP
                super.initializeConfiguration();
            }
        }

        // Determine persistence configuration for SIP
        PersistenceType persistenceType = PersistenceType.MEMORY;
        String persistenceFrequency = null;
        String persistenceScope = null;

        boolean isAvailabilityEnabled = lookup.calculateSipAvailabilityEnabledFromConfig(_ctx);

        if (isAvailabilityEnabled) {
            // These are the global defaults if nothing is set at
            // domain.xml or sun-sip.xml
            persistenceType = PersistenceType.REPLICATED;
            persistenceFrequency = "sip-transaction";
            persistenceScope = "session";
        }

        // If domain.xml default exists, then use that
        PersistenceType domainPersistenceType = lookup.getSipPersistenceTypeFromConfig();

        if (domainPersistenceType != null) {
            persistenceType = domainPersistenceType;
            persistenceFrequency = lookup.getSipPersistenceFrequencyFromConfig();
            persistenceScope = lookup.getSipPersistenceScopeFromConfig();
        }

        // Consider persistence type, scope, and frequency from sun-sip.xml
        if (sipSmBean != null) {
            String pType = sipSmBean.getAttributeValue(
                SessionManager.PERSISTENCE_TYPE);
            persistenceType = PersistenceType.parseType(pType, persistenceType);

            String sipAppLevelPersistenceFrequency =
                getPersistenceFrequency(sipSmBean);
            if(sipAppLevelPersistenceFrequency != null) {
                persistenceFrequency = sipAppLevelPersistenceFrequency;
            }

            String sipAppLevelPersistenceScope =
                getPersistenceScope(sipSmBean);
            if(sipAppLevelPersistenceScope != null) {
                persistenceScope = sipAppLevelPersistenceScope;
            }
        }
        
        String frequency = null;
        String scope = null;

        // Deliberately leave frequency & scope null in memory-only case
        if (persistenceType != PersistenceType.MEMORY) {
            frequency = persistenceFrequency;
            scope = persistenceScope;
        }

        // If sip app is not distributable, and non-memory option
        // is attempted, log warning and set back to memory-only
        if (!isSipAppDistributable &&
                (persistenceType != PersistenceType.MEMORY)) {
            String sipAppName = getApplicationId(_ctx);

            // Suppress log warning for default-web-module
            // Log message only if availabilityenabled = true is attempted
            if (isAvailabilityEnabled &&
                    !sipAppName.equals(Constants.DEFAULT_WEB_MODULE_NAME) &&
                    !isSystemApp(sipAppName)) {
                SIP_LOGGER.log(Level.WARNING, "sipstack.invalid_session_manager_config",
                                   new Object[] { sipAppName, persistenceType.getType(),
                                   frequency, scope });
            }

            // Fall back to memory option
            persistenceType = PersistenceType.MEMORY;
            frequency = null;
            scope = null;
        }

        // If availability-enabled is false, reset to "memory"
        if (!isAvailabilityEnabled) {
            // Set back to memory option
            persistenceType = PersistenceType.MEMORY;
            frequency = null;
            scope = null;             
        }

        sipPersistenceType = persistenceType;
        sipPersistenceFrequency = frequency;
        sipPersistenceScope = scope;

        // Make sure that if the app is converged (i.e., contains both
        // a web.xml and sip.xml), and in-memory replication has been turned
        // on for SIP, it must also have been turned on for HTTP, and vice
        // versa.
        // Otherwise, we fall back to memory-only for both SIP and HTTP.
        if (sbd != null &&
                sbd.isConverged() &&
                (((sipPersistenceType == PersistenceType.REPLICATED) &&
                (_persistence != PersistenceType.REPLICATED)) ||
                ((sipPersistenceType != PersistenceType.REPLICATED) &&
                (_persistence == PersistenceType.REPLICATED)))) {
            SIP_LOGGER.log(Level.WARNING, "sipstack.inconsistent_persistence_type", getApplicationId(_ctx));

            // Reset SIP persistence to memory-only
            sipPersistenceType = PersistenceType.MEMORY;
            sipPersistenceFrequency = null;
            sipPersistenceScope = null;

            // Reset HTTP persistence to memory-only
            _persistence = PersistenceType.MEMORY;
            _persistenceFrequency = null;
            _persistenceScope = null;
        }
    }

    public PersistenceType getSipPersistenceType() {
        checkInitialization();

        return sipPersistenceType;
    }

    public String getSipPersistenceFrequency() {
        checkInitialization();

        return sipPersistenceFrequency;
    }

    public String getSipPersistenceScope() {
        checkInitialization();

        return sipPersistenceScope;
    }

    /**
     * Gets the bean representing the sip-container-availability element
     * in domain.xml.
     *
     * @return The bean representing the sip-container-availability element
     * in domain.xml.
     */
    public SipContainerAvailability getSipContainerAvailability() {
        return lookup.getSipContainerAvailability();
    }
}
