/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package com.ericsson.ssa.container;

import com.sun.grizzly.Context.OpType;
import com.sun.grizzly.PipelineFullException;
import com.sun.grizzly.async.AsyncWriteQueueRecord;
import com.sun.grizzly.util.net.SSLImplementation;
import com.sun.grizzly.util.net.ServerSocketFactory;
import java.io.IOException;
import java.net.ConnectException;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.SelectionKey;
import java.util.Iterator;
import java.util.Queue;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import com.ericsson.ssa.config.annotations.Configuration;
import com.ericsson.ssa.config.annotations.UpdatePolicy;
import com.ericsson.ssa.config.annotations.UsagePolicy;
import com.ericsson.ssa.container.startup.SipMonitoring;
import com.ericsson.ssa.sip.Layer;
import com.ericsson.ssa.sip.LayerHelper;
import com.ericsson.ssa.sip.SipServletMessageImpl;
import com.ericsson.ssa.sip.SipServletRequestImpl;
import com.ericsson.ssa.sip.SipServletResponseImpl;
import com.ericsson.ssa.sip.dns.SipTransports;
import com.ericsson.ssa.sip.dns.TargetResolver;
import com.ericsson.ssa.sip.dns.TargetTuple;
import com.ericsson.ssa.utils.Pair;
import com.sun.grizzly.CallbackHandler;
import com.sun.grizzly.CallbackHandlerDescriptor;
import com.sun.grizzly.ConnectorHandler;
import com.sun.grizzly.Context;
import com.sun.grizzly.Context.KeyRegistrationState;
import com.sun.grizzly.Controller;
import com.sun.grizzly.DefaultPipeline;
import com.sun.grizzly.SelectionKeyHandler;
import com.sun.grizzly.DefaultSelectionKeyHandler;
import com.sun.grizzly.IOEvent;
import com.sun.grizzly.Pipeline;
import com.sun.grizzly.ProtocolChain;
import com.sun.grizzly.DefaultProtocolChain;
import com.sun.grizzly.ProtocolChainInstanceHandler;
import com.sun.grizzly.TCPConnectorHandler;
import com.sun.grizzly.TCPSelectorHandler;
import com.sun.grizzly.UDPConnectorHandler;
import com.sun.grizzly.UDPSelectorHandler;
import com.sun.grizzly.Controller.Protocol;
import com.sun.grizzly.ProtocolFilter;
import com.sun.grizzly.SSLCallbackHandler;
import com.sun.grizzly.SSLConnectorHandler;
import com.sun.grizzly.filter.ReadFilter;
import com.sun.grizzly.filter.SSLReadFilter;
import com.sun.grizzly.SSLSelectorHandler;
import com.sun.grizzly.SelectorHandler;
import com.sun.grizzly.async.AsyncQueueWritable;
import com.sun.grizzly.async.AsyncWriteCallbackHandler;

import com.sun.grizzly.util.ByteBufferFactory.ByteBufferType;
import com.sun.grizzly.util.ThreadAttachment;
import com.sun.grizzly.util.ThreadAttachment.Mode;
import com.sun.grizzly.util.WorkerThread;
import java.io.UnsupportedEncodingException;
import java.net.Socket;
import java.nio.channels.DatagramChannel;
import java.nio.channels.SelectableChannel;
import java.nio.channels.SocketChannel;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import javax.net.ssl.SSLContext;

import org.jvnet.glassfish.comms.util.LogUtil;

/**
 * This class is responsible for initializing the Grizzly based SIP protocol
 * support in Sailfin.
 * 
 * @author ekrigro
 * @author Jeanfrancois Arcand
 */
public class GrizzlyNetworkManager extends NetworkManager implements Runnable {

    private final static String IS_CLIENT_EXECUTION = "isClientExecution";
    public final static int DEFAULT_BB_SIZE = 8192;
    public final static int DEFAULT_SSL_BB_SIZE = 20480;
    private static final String SERVER_PIPELINE_NAME = "SipContainer-servers"; //Only used for name of Pipelines.
    private static final String CLIENT_PIPELINE_NAME = "SipContainer-clients"; //Only used for name of Pipelines.
    private static final int DEFAULT_PIPELINE_DISTINCTION_PORT = 5060; //Only used for unique ID of the default Pipeline.
    public final static String SIP_CERTS = "Sip.Certs";
    private final Logger logger = LogUtil.SIP_LOGGER.getLogger();
    private Controller clientController = new Controller();
    private Controller serverController = new Controller();
    private Pipeline<Callable> serverPipeline = null;
    private Pipeline<Callable> clientPipeline = null;
    private Layer nextLayer = null;
    private static final ConcurrentLinkedQueue<ByteBuffer> byteBuffersPool =
            new ConcurrentLinkedQueue<ByteBuffer>();
    private static final ConcurrentLinkedQueue<ByteBuffer> secureByteBuffersPool =
            new ConcurrentLinkedQueue<ByteBuffer>();
    protected static final AsyncWriteCallbackHandler asyncWriteCallbackHandler = 
            new SharedAsyncWriteCallbackHandler();
    private SSLContext sslContext;
    private ConnectionManager connectionManager = null;
    //properties
    private int keepAliveTimeoutInSeconds = 600;
    private String keyStore = null;
    private String trustStore = null;
    private int maxPipelineThreads = 10; //100
    private int minPipelineThreads = 10; //20
    private int pipelineThreadsIncrement = 1;
    private int pipelineInitialByteBufferSize = 8192; //same as DefaultPipeline
    private boolean clientsShareServerThreadPool = true;
    private int maxCachedByteBuffers = 50;
    private InetSocketAddress localUDPSocketAddress;
    //[Eltjo] Sorry for this monster
    private  Boolean useUDPPortForClient = Boolean.getBoolean(
               "org.jvnet.glassfish.comms.useUDPSourcePort");
    /**
     * The model for keeping track of resources that are allocated for a specific
     * SipBindingCtx (sip-listener).
     * The structure is such:
     * Map-+-+-...
     *     |
     *     context -> Pair-+
     *                    / \
     *       SipBindingCtx  Map-+-+...
     *                          |
     *                          protocol -> Pair-+
     *                                          / \
     *                     server SelectorHandler client SelectorHandler
     * 
     * As you can see it assumes that a SipBindingCtx can only have one
     * TargetTuple per protocol, meaning only one port per protocol.
     */
    private HashMap<String, Pair<SipBindingCtx, Map<SipTransports, Pair<SelectorHandler, SelectorHandler>>>> ctxs =
            new HashMap<String, Pair<SipBindingCtx, Map<SipTransports, Pair<SelectorHandler, SelectorHandler>>>>();

    private final ConcurrentHashMap<SelectionKey,TargetTuple> 
            outboundConnectionsTuple = 
                new ConcurrentHashMap<SelectionKey,TargetTuple>();
    private static final ConcurrentHashMap<String,ByteBuffer> 
            secureByteBufferMap = 
                new ConcurrentHashMap<String,ByteBuffer>();
    
    public GrizzlyNetworkManager() {
    }

    public synchronized void start() {

        initPipeline();
        
        for (String ctx : SipBindingResolver.instance().getLocalContexts()) {
            startCtxServers(ctx);
        }

        SipBindingResolver.instance().registerSipBindingListener(new SipBindingListener() {

            public void newSipBindingCtxAvaliable(String context) {
                startCtxServers(context);
            }

            public void sipBindingCtxUpdated(String context) {
                restartCtxServers(context);
            }
            
            public void publicSipBindingCtxUpdated() {
                //ignore
            }

            public void sipBindingCtxRemoved(String context) {
                stopCtxServers(context);
            }
            });

        //Initiate the outbound connections pool
        connectionManager = new ConnectionManager();

        serverPipeline.initPipeline();
        serverPipeline.startPipeline();

        clientPipeline.initPipeline();
        clientPipeline.startPipeline();
    }

    public void startCtxServers(String ctx) {
        SipBindingCtx sipBindingCtx = SipBindingResolver.instance().getContext(ctx);
        if (sipBindingCtx != null && !sipBindingCtx.isStale()) {
            Pair<SipBindingCtx, Map<SipTransports, Pair<SelectorHandler, SelectorHandler>>> pair = new 
                    Pair<SipBindingCtx, Map<SipTransports, Pair<SelectorHandler, SelectorHandler>>>(sipBindingCtx, 
                    new HashMap<SipTransports, Pair<SelectorHandler, SelectorHandler>>());
            ctxs.put(ctx, pair);
            
            if (sipBindingCtx.isEnabled()) {
                final ProtocolChain protocolChain = getProtocolChain();
                ProtocolChainInstanceHandler instanceHandler =
                        new SimpleProtocolChainInstanceHandler(protocolChain);
                
                serverController.setPipeline(serverPipeline);
                clientController.setPipeline(clientPipeline);

                for (TargetTuple targetTuple : sipBindingCtx.getTargetTuples()) {
                    if (logger.isLoggable(Level.INFO)) {
                        logger.log(Level.INFO, "sip.stack.network.starting_sip_binding", targetTuple.toString());
                    }
                    if (logger.isLoggable(Level.FINE)) {
                        logger.log(Level.FINE, "Adding "+targetTuple.getProtocol().name()+" server for SipBindingCtx [" + sipBindingCtx.getContext() + "].");
                        logger.log(Level.FINE, "Adding "+targetTuple.getProtocol().name()+" client for SipBindingCtx [" + sipBindingCtx.getContext() + "].");
                    }
                    Pair<SelectorHandler, SelectorHandler> selHdlPair = 
                            new Pair<SelectorHandler, SelectorHandler>();
                    startSelectorHandlers(targetTuple,
                            selHdlPair,
                            instanceHandler,
                            sipBindingCtx);
                    pair.getRight().put(targetTuple.getProtocol(), selHdlPair);                    
                }
            } else {
                    // Just putting the ctx is enough.
            }
        } else {
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE, "SipBindingCtx [" + ctx + "] is stale, not starting ctx");
            }
        }
    }

    public void startSelectorHandlers(TargetTuple targetTuple, 
            Pair<SelectorHandler, SelectorHandler> selHdlPair,
            ProtocolChainInstanceHandler instanceHandler,
            SipBindingCtx sipBindingCtx) {
        switch (targetTuple.getProtocol().ordinal()) {
            case SipTransports.UDP: {                
                selHdlPair.setLeft(startUDPServer(targetTuple, instanceHandler));
                selHdlPair.setRight(startUDPClient(instanceHandler));
                break;
            }
            case SipTransports.TCP: {
                selHdlPair.setLeft(startTCPServer(targetTuple, instanceHandler));
                selHdlPair.setRight(startTCPClient(instanceHandler));
                break;
            }
            case SipTransports.TLS: {
                final ProtocolChain tlsProtocolChain = getTLSProtocolChain(sipBindingCtx);
                ProtocolChainInstanceHandler tlsInstanceHandler =
                        new SimpleProtocolChainInstanceHandler(tlsProtocolChain);
                selHdlPair.setLeft(startTLSServer(targetTuple, tlsInstanceHandler));
                selHdlPair.setRight(startTLSClient(tlsInstanceHandler));
                break;
            }
            case SipTransports.UNDEFINED://Hmm, should not happen
                break;
        }        
    }
    
    private void restartCtxServers(String ctx) {
        SipBindingCtx sipBindingCtx = SipBindingResolver.instance().getContext(ctx);
        
        Pair<SipBindingCtx, Map<SipTransports, Pair<SelectorHandler, SelectorHandler>>> pair = 
                ctxs.remove(ctx); 
              
       if (pair!=null) {
           SipBindingCtx.UpdateHint hint = 
                   pair.getLeft().getUpdateHints(sipBindingCtx);
           
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE, "Updating for SipBindingCtx["+ctx+"]: update hint = "+hint);
            }
           
           if (updateCtx(pair, sipBindingCtx, hint)) {
               pair.setLeft(sipBindingCtx);
           }                      
           ctxs.put(ctx, pair);
       } else {
            //Weird case. somehow we are updating an unknown context.
            //Lets just start it.
            //TODO does this make sense?
            startCtxServers(ctx);
       }
    }

    /**
     * 
     * @param pair
     * @param sipBindingCtx
     * @param hint
     * @return true when an update was needed and performed.
     */
    private boolean updateCtx(Pair<SipBindingCtx, 
            Map<SipTransports, Pair<SelectorHandler, SelectorHandler>>> pair,
            SipBindingCtx sipBindingCtx, SipBindingCtx.UpdateHint hint) {
        //A SipBindingCtx can be updated in several ways, it can be:
        //- disabled / enabled
        //- altered wrt ssl attributes
        //- altered wrt the TargetTuples
        //Or a combination
        //The update procedure does little assumptions on the SipBindingCtx
        //I.e. the fact that in the current config model a SipBindingCtx either
        //contains a UDP and TCP TargetTuple or only a TLS TargetTuple is not
        //taken into account. Assuming that any mix of TargetTuples can be 
        //present makes that the procedure is prepared for any changes in the
        //config model.
        boolean updated = false;
        
        switch(hint) {
        case UP_TO_DATE:
            //Okidoki
            break;
        case ENABLE_DISABLED:
            enOrDisableCtx(pair, sipBindingCtx);
            updated = true;
            break;
        case SSL_UPDATE:
            updateSSLCtx(pair, sipBindingCtx);
            updated = true;
            break;
        case TARGET_TUPLE_UPDATE:
            updateTargetTuplesCtx(pair, sipBindingCtx);
            updated = true;
            break;
        case MULTIPLE:
            //The approach for disable/enable is to crudely remove / add the
            //resources
            if (hint.contains(SipBindingCtx.UpdateHint.ENABLE_DISABLED)) {
                enOrDisableCtx(pair, sipBindingCtx);            
            } else {
                if (hint.contains(SipBindingCtx.UpdateHint.SSL_UPDATE)) {
                    updateSSLCtx(pair, sipBindingCtx);
                }
                if (hint.contains(SipBindingCtx.UpdateHint.TARGET_TUPLE_UPDATE)) {
                    updateTargetTuplesCtx(pair, sipBindingCtx);
                }
            }
            updated = true;
            break;         
        case NOT_SAME:
            //Should not happen
            break;         
        }
        
        return updated;
    }


    private void enOrDisableCtx(Pair<SipBindingCtx, 
            Map<SipTransports, Pair<SelectorHandler, SelectorHandler>>> pair, 
            SipBindingCtx sipBindingCtx) {
        if (pair.getLeft().isEnabled()) {
            Iterator<Map.Entry<SipTransports, Pair<SelectorHandler, SelectorHandler>>> i = 
                    pair.getRight().entrySet().iterator();
            while (i.hasNext()) {            
                Map.Entry<SipTransports, Pair<SelectorHandler, SelectorHandler>> e = 
                        i.next();
                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE, "Disabling "+e.getKey().name()+" server for SipBindingCtx [" + sipBindingCtx.getContext() + "].");
                    logger.log(Level.FINE, "Disabling "+e.getKey().name()+" client for SipBindingCtx [" + sipBindingCtx.getContext() + "].");
                }
                shutdownAndRemoveSelectorHandlers(e.getKey(), 
                        e.getValue().getLeft(), e.getValue().getRight());
                //Clean up the entry, a disabled ctx is expected to have 
                //no resources. This makes it easier to align with the case
                //were a new but disabled ctx is started.
                i.remove();
            }
        } else {
            for (TargetTuple targetTuple : sipBindingCtx.getTargetTuples()) {
                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE, "Enabling "+targetTuple.getProtocol().name()+" server for SipBindingCtx [" + sipBindingCtx.getContext() + "].");
                    logger.log(Level.FINE, "Enabling "+targetTuple.getProtocol().name()+" client for SipBindingCtx [" + sipBindingCtx.getContext() + "].");
                }
                final ProtocolChain udpProtocolChain = getProtocolChain();
                ProtocolChainInstanceHandler instanceHandler =
                        new SimpleProtocolChainInstanceHandler(udpProtocolChain);
                Pair<SelectorHandler, SelectorHandler> selHdlPair = 
                        new Pair<SelectorHandler, SelectorHandler>();
                startSelectorHandlers(targetTuple,
                        selHdlPair,
                        instanceHandler,
                        sipBindingCtx);
                pair.getRight().put(targetTuple.getProtocol(), selHdlPair);               
            }
        }
    }
    
    private void updateSSLCtx(Pair<SipBindingCtx, Map<SipTransports, Pair<SelectorHandler, SelectorHandler>>> pair, 
            SipBindingCtx sipBindingCtx) {
        //Consider the following cases:
        //- TLS exist in old and new
        //--and they equal
        //--->update the selectionHandler according SSL update        
        //--and they differ
        //--->ignore, the new TLS will take it        
        //-TLS exist in old or new only
        //-->ignore, the new TLS will take it
        //-TLS doesn't exist in old or new
        //-->ignore, if no TLS the SSL attributes are not used.
        //This is tollerated to allow admin to create the SSL before adding
        //or updating to TLS.        
        TargetTuple oldTlsTT = pair.getLeft().getTargetTupleForProtocol(SipTransports.TLS_PROT);        
        TargetTuple newTlsTT = sipBindingCtx.getTargetTupleForProtocol(SipTransports.TLS_PROT);      

        logger.log(Level.FINE, "updateSSLCtx: pair = "+pair);
        
        if ((oldTlsTT!=null && newTlsTT!=null) && oldTlsTT.equals(newTlsTT)) {
            Pair<SelectorHandler, SelectorHandler> selHdlPair = 
                    pair.getRight().get(SipTransports.TLS_PROT);
            if (selHdlPair!=null) {
                //disable can cause the pair to be absent.
                SSLSelectorHandler serverTlsSH = (SSLSelectorHandler)selHdlPair.getLeft();
                logger.log(Level.FINE, "updateSSLCtx: TLS server SelectorHandler = "+serverTlsSH);
                logger.log(Level.FINE, "updateSSLCtx: ProtocolChainInstanceHandler = "+serverTlsSH.getProtocolChainInstanceHandler());
                logger.log(Level.FINE, "updateSSLCtx: ProtocolChain = "+((SimpleProtocolChainInstanceHandler)serverTlsSH.getProtocolChainInstanceHandler()).getProtocolChain());
                ProtocolChain protocolChain = 
                        ((SimpleProtocolChainInstanceHandler)serverTlsSH.getProtocolChainInstanceHandler()).getProtocolChain();

                updateTLSProtocolChain(protocolChain, sipBindingCtx);
            }
        }
    }

    private void updateTargetTuplesCtx(Pair<SipBindingCtx, 
            Map<SipTransports, Pair<SelectorHandler, SelectorHandler>>> pair, 
            SipBindingCtx sipBindingCtx) {
        //Consider the following cases:
        //- A TT exist in old and new
        //--and they differ
        //--->remove the old one add the new one        
        //--and they equal
        //--->okidoki        
        //-A TT exist in new only
        //-->Add it
        //-A TT exist in old only
        //-->Remove it        
        List<TargetTuple> oldTTs = new ArrayList<TargetTuple>(Arrays.asList(pair.getLeft().getTargetTuples())); //Make sure remove is supported       
        List<TargetTuple> newTTs = Arrays.asList(sipBindingCtx.getTargetTuples());
        
        for (TargetTuple newTT : newTTs) {
            TargetTuple oldTT = pair.getLeft().getTargetTupleForProtocol(newTT.getProtocol());
            
            if (oldTT!=null) {
                if (!oldTT.equals(newTT)) {
                    //remove the old one add the new one
                    if (logger.isLoggable(Level.FINE)) {
                        logger.log(Level.FINE, "Update TT: replacing "+newTT.getProtocol().name()+" server for SipBindingCtx [" + sipBindingCtx.getContext() + "].");
                        logger.log(Level.FINE, "Update TT: replacing "+newTT.getProtocol().name()+" client for SipBindingCtx [" + sipBindingCtx.getContext() + "].");
                    }
                    //Remove
                    Pair<SelectorHandler, SelectorHandler> selHdlPair = 
                            pair.getRight().get(newTT.getProtocol());
                    if (selHdlPair!=null) { //disable can cause the pair to be absent.                      
                        shutdownAndRemoveSelectorHandlers(newTT.getProtocol(), 
                                selHdlPair.getLeft(), selHdlPair.getRight());
                        //Add
                        final ProtocolChain udpProtocolChain = getProtocolChain();
                        ProtocolChainInstanceHandler instanceHandler =
                                new SimpleProtocolChainInstanceHandler(udpProtocolChain);
                        startSelectorHandlers(newTT,
                                selHdlPair,
                                instanceHandler,
                                sipBindingCtx);
                    }
                    oldTTs.remove(oldTT);
                } else {
                    //okidoki
                    //Check updateSSL, there the SSL changes are taken into
                    //account. 
                }
            } else {
                //Add it
                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE, "Update TT: adding "+newTT.getProtocol().name()+" server for SipBindingCtx [" + sipBindingCtx.getContext() + "].");
                    logger.log(Level.FINE, "Update TT: adding "+newTT.getProtocol().name()+" client for SipBindingCtx [" + sipBindingCtx.getContext() + "].");
                }
                final ProtocolChain udpProtocolChain = getProtocolChain();
                ProtocolChainInstanceHandler instanceHandler =
                        new SimpleProtocolChainInstanceHandler(udpProtocolChain);
                Pair<SelectorHandler, SelectorHandler> selHdlPair = 
                        new Pair<SelectorHandler, SelectorHandler>();
                startSelectorHandlers(newTT,
                        selHdlPair,
                        instanceHandler,
                        sipBindingCtx);                                    
                pair.getRight().put(newTT.getProtocol(), selHdlPair);               
            }
        }
        
        for (TargetTuple oldTT : oldTTs) {
            //Remove it        
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE, "Update TT: removing "+oldTT.getProtocol().name()+" server for SipBindingCtx [" + sipBindingCtx.getContext() + "].");
                logger.log(Level.FINE, "Update TT: removing "+oldTT.getProtocol().name()+" client for SipBindingCtx [" + sipBindingCtx.getContext() + "].");
            }
            Pair<SelectorHandler, SelectorHandler> selHdlPair = 
                    pair.getRight().remove(oldTT.getProtocol());
            shutdownAndRemoveSelectorHandlers(oldTT.getProtocol(), 
                    selHdlPair.getLeft(), selHdlPair.getRight());
        }        
    }
    
    private void stopCtxServers(String ctx) {
        //Crude approach to stopping SipBindingCtx
        //TODO make gracefull
        Pair<SipBindingCtx, Map<SipTransports, Pair<SelectorHandler, SelectorHandler>>> pair = 
                ctxs.remove(ctx);
        
        if (pair!=null) {
            if (pair.getLeft().isEnabled()) {
                for (Map.Entry<SipTransports, Pair<SelectorHandler, SelectorHandler>> e : pair.getRight().entrySet()) {                
                    if (logger.isLoggable(Level.FINE)) {
                        logger.log(Level.FINE, "Stopping "+e.getKey().name()+" server for SipBindingCtx [" + ctx + "].");
                        logger.log(Level.FINE, "Stopping "+e.getKey().name()+" client for SipBindingCtx [" + ctx + "].");
                    }

                    shutdownAndRemoveSelectorHandlers(e.getKey(), 
                            e.getValue().getLeft(), e.getValue().getRight());
                }
            } else {
                //Just removing the ctx is enough.
                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE, "Stopped disabled SipBindingCtx [" + ctx + "].");
                }
            }
        }
    }
    
    private void shutdownAndRemoveSelectorHandlers(SipTransports protocol, 
            SelectorHandler server, SelectorHandler client) {
        switch (protocol.ordinal()) {
        case SipTransports.UDP: {
            serverController.removeSelectorHandler(server);
            clientController.removeSelectorHandler(client);
            break;
        }
        case SipTransports.TCP: {
            serverController.removeSelectorHandler(server);
            clientController.removeSelectorHandler(client);
            break;
        }
        case SipTransports.TLS: {
            serverController.removeSelectorHandler(server);
            clientController.removeSelectorHandler(client);
            break;
        }
        }        
    }

    private UDPSelectorHandler startUDPServer(TargetTuple targetTuple,
            ProtocolChainInstanceHandler instanceHandler) {
        UDPSelectorHandler udpSelector = createUdpEndpoint(
                targetTuple.getSocketAddress());
	localUDPSocketAddress = targetTuple.getSocketAddress();
        SelectionKeyHandler keyHandler = createSelectionKeyHandler();
        udpSelector.setSelectionKeyHandler(keyHandler);
        udpSelector.setProtocolChainInstanceHandler(instanceHandler);
        serverController.addSelectorHandler(udpSelector);
        return udpSelector;
    }

    private TCPSelectorHandler startTCPServer(TargetTuple targetTuple,
            ProtocolChainInstanceHandler instanceHandler) {
        TCPSelectorHandler tcpSelector = createTcpEndpoint(
                targetTuple.getSocketAddress());
        SelectionKeyHandler keyHandler = createSelectionKeyHandler();
        tcpSelector.setSelectionKeyHandler(keyHandler);
        tcpSelector.setProtocolChainInstanceHandler(instanceHandler);
        serverController.addSelectorHandler(tcpSelector);
        return tcpSelector;
    }

    private SSLSelectorHandler startTLSServer(TargetTuple targetTuple,
            ProtocolChainInstanceHandler tlsInstanceHandler) {
        SSLSelectorHandler tlsSelector = createTLSEndpoint(
                targetTuple.getSocketAddress());
        SelectionKeyHandler keyHandler = createSelectionKeyHandler();
        tlsSelector.setSelectionKeyHandler(keyHandler);
        tlsSelector.setProtocolChainInstanceHandler(tlsInstanceHandler);
        serverController.addSelectorHandler(tlsSelector);
        return tlsSelector;
    }

    private UDPSelectorHandler startUDPClient(
            ProtocolChainInstanceHandler instanceHandler) {
        UDPSelectorHandler udpClientSelectorHandler = new UDPSelectorHandler(true);
        udpClientSelectorHandler.setProtocolChainInstanceHandler(instanceHandler);
        SelectionKeyHandler keyHandler = createSelectionKeyHandler();
        udpClientSelectorHandler.setSelectionKeyHandler(keyHandler);
        udpClientSelectorHandler.setProtocolChainInstanceHandler(instanceHandler);
        clientController.addSelectorHandler(udpClientSelectorHandler);
        return udpClientSelectorHandler;
    }

    private TCPSelectorHandler startTCPClient(
            ProtocolChainInstanceHandler instanceHandler) {
        TCPSelectorHandler tcpClientSelectorHandler = new TCPSelectorHandler(true);
        tcpClientSelectorHandler.setProtocolChainInstanceHandler(instanceHandler);
        SelectionKeyHandler keyHandler = createSelectionKeyHandler();
        tcpClientSelectorHandler.setSelectionKeyHandler(keyHandler);
        tcpClientSelectorHandler.setProtocolChainInstanceHandler(instanceHandler);
        clientController.addSelectorHandler(tcpClientSelectorHandler);
        return tcpClientSelectorHandler;
    }

    private SSLSelectorHandler startTLSClient(
            ProtocolChainInstanceHandler tlsInstanceHandler) {
        SSLSelectorHandler tlsClientSelectorHandler = new SSLSelectorHandler(true);
        tlsClientSelectorHandler.setProtocolChainInstanceHandler(tlsInstanceHandler);
        SelectionKeyHandler keyHandler = createSelectionKeyHandler();
        tlsClientSelectorHandler.setSelectionKeyHandler(keyHandler);
        tlsClientSelectorHandler.setProtocolChainInstanceHandler(tlsInstanceHandler);
        clientController.addSelectorHandler(tlsClientSelectorHandler);
        return tlsClientSelectorHandler;
    }

    private SelectionKeyHandler createSelectionKeyHandler() {
        DefaultSelectionKeyHandler skh = new SharedSelectionKeyHandler();
        skh.setLogger(logger);
        return skh;
    }

    public synchronized void stop() throws IOException {
        clientController.stop();
        serverController.stop();
    }

    private ProtocolChain getProtocolChain() {
        DefaultProtocolChain protocolChain = new DefaultProtocolChain();
        protocolChain.setContinuousExecution(false);

        ReadFilter readFilter = new SharedReadFilter();
        readFilter.setContinuousExecution(false);

        protocolChain.addFilter(readFilter);
        protocolChain.addFilter(new MessageProcessorFilter(this));

        return protocolChain;
    }

    private ProtocolChain getTLSProtocolChain(SipBindingCtx sipBindingCtx) {
        SSLReadFilter sslReadFilter = getSSLReadFilter(sipBindingCtx);

        //Carefull dynamic reconfig update of SSL relies on the sslReadFilter
        //to be the first filter in the chain.
        ProtocolChain tlsProtocolChain = new DefaultProtocolChain();
        tlsProtocolChain.addFilter(sslReadFilter);
        tlsProtocolChain.addFilter(new MessageProcessorFilter(this));

        return tlsProtocolChain;
    }
    
    private SSLReadFilter getSSLReadFilter(SipBindingCtx sipBindingCtx) {
        SSLReadFilter sslReadFilter = new ClientCertSSLReadFilter();
        sslReadFilter.setSSLContext(getSSLContext(sipBindingCtx));
        Boolean clientAuth = Boolean.parseBoolean(
                sipBindingCtx.getSSLAttribute("ClientAuthEnabled"));
        sslReadFilter.setNeedClientAuth(clientAuth);
        
        return sslReadFilter;
    }
    
    private void updateTLSProtocolChain(ProtocolChain aProtocolChain, 
            SipBindingCtx sipBindingCtx) {
        if (aProtocolChain instanceof DefaultProtocolChain) {
            SSLReadFilter sslReadFilter = getSSLReadFilter(sipBindingCtx);
            //Carefull this relies on the sslReadFilter
            //to be the first filter in the chain.
            ((DefaultProtocolChain)aProtocolChain).setProtocolFilter(0, 
                    sslReadFilter);
        }
    }
    
    private SSLContext getSSLContext(SipBindingCtx sipBindingCtx) {
        ServerSocketFactory serverSF = null;
        try {

            //TODO This expects that the ServerSocketFactory initialized
            //is a seperated instance. A brief check of the grizzly
            //classes involved reveals that it is for the default JSSE14 factories
            SSLImplementation sslHelper = SSLImplementation.getInstance();
            serverSF = sslHelper.getServerSocketFactory();
            serverSF.setAttribute("keystoreType", "JKS");
            serverSF.setAttribute("keystore", keyStore);
            serverSF.setAttribute("truststoreType", "JKS");
            serverSF.setAttribute("truststore", trustStore);
            String keyAlias = sipBindingCtx.getSSLAttribute("CertNickname");
            serverSF.setAttribute("keyAlias", (keyAlias != null ? keyAlias : "s1as")); //Default GF s1sa
            serverSF.init();
        } catch (IOException e) {
        //TODO Logging
        } catch (ClassNotFoundException e) {
        //TODO Logging
        }
        return serverSF.getSSLContext();
    }

    void initPipeline() {
        serverPipeline = createPipeline(SERVER_PIPELINE_NAME, DEFAULT_PIPELINE_DISTINCTION_PORT);
        clientPipeline = clientsShareServerThreadPool ? serverPipeline : createPipeline(CLIENT_PIPELINE_NAME, DEFAULT_PIPELINE_DISTINCTION_PORT);

        SipContainerThreadPool tp = SipContainerThreadPool.getInstance();
        tp.initialize(serverPipeline);
    }

    private void reinitPipeline() {
        //TODO reinit the pipeline according to the reconfigured settings.
        if (logger.isLoggable(Level.FINE)) {
            logger.log(Level.FINE, "reinitPipeline not implemented.");
        }
    }

    private DefaultPipeline createPipeline(String pipelineName, int distinctionPort) {
        DefaultPipeline pipeline = new DefaultPipeline(maxPipelineThreads,
                minPipelineThreads, pipelineName, distinctionPort);
        pipeline.setThreadsIncrement(pipelineThreadsIncrement);
        pipeline.setInitialByteBufferSize(pipelineInitialByteBufferSize);
        pipeline.setByteBufferType(ByteBufferType.HEAP);
        // No limits for now.
        pipeline.setQueueSizeInBytes(-1);

        return pipeline;
    }

    public void next(SipServletRequestImpl req) {
        LayerHelper.next(req, this, nextLayer);
    }

    public void next(SipServletResponseImpl resp) {
        LayerHelper.next(resp, this, nextLayer);
    }

    public void registerNext(Layer layer) {
        nextLayer = layer;
    }

    public void dispatch(final SipServletRequestImpl req) {
        try {
            connectionManager.send(req);

            if (SipMonitoring.isEnabled(SipMonitoring.NETWORK_MANAGER)) {
                incrEasSentSipRequests();
            }


        } catch (IOException e) {    
                    if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE, "sip.stack.network.failed_to_send_request", e);
            }
            try {
                if (req.getMethod().equals("ACK")) {
                    return;
                }
                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE, "sip.stack.network.send_503");
                }
                clientPipeline.execute(new Callable() {

                    public Object call() throws Exception {
                        SipServletResponseImpl resp = req.createTerminatingResponse(503);
                        LayerHelper.next(resp, nextLayer, nextLayer);
                        return null;
                    }
                });
            } catch (PipelineFullException ex) {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.log(Level.SEVERE, "sip.stack.network.failed_to_send_503");
					logger.log(Level.SEVERE, ex.getMessage(), ex);
                }
            }        
        }
    }

    public void dispatch(SipServletResponseImpl resp) { //Failback, StreamResponseDispatchern is the normal path
        try {
            connectionManager.send(resp);

            if (SipMonitoring.isEnabled(SipMonitoring.NETWORK_MANAGER)) {
                incrEasSentSipResponses();
            }

        } catch (IOException e) {
            try{
                TargetTuple newTupple = TargetResolver.getInstance().resolveResponse(resp);
                if (newTupple != null && newTupple != resp.getRemote()) {
                    resp.setRemote(newTupple);

                    // Cannot use recurrance here because it will loop.
                    connectionManager.send(resp);

                    if (SipMonitoring.isEnabled(SipMonitoring.NETWORK_MANAGER)) {
                        incrEasSentSipResponses();
                    }                
                }
            } catch (Exception e2) {   
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.log(Level.SEVERE, "sip.stack.network.failed_to_send_response", resp.toDebugString());
					logger.log(Level.SEVERE, e2.getMessage(), e2);
                }
                // Only report the last exception
                return;
            }
                
            if (logger.isLoggable(Level.SEVERE)) {
                logger.log(Level.SEVERE, "sip.stack.network.failed_to_send_response", resp.toDebugString());
				logger.log(Level.SEVERE, e.getMessage(), e);
            }
        }
    }

    TCPSelectorHandler createTcpEndpoint(InetSocketAddress addr) {
        int linger = -1;

        final TCPSelectorHandler selectorHandler = new TCPSelectorHandler(){            
            /**
             * Intercept the accept operations and cache the associated connection.
             */
            @Override
            public boolean onAcceptInterest(SelectionKey key,
                    Context ctx) throws IOException{     
                
                SelectableChannel channel = acceptWithoutRegistration(key);        
                if (channel != null) {
                    configureChannel(channel);
                    SelectionKey readKey =
                            channel.register(selector, SelectionKey.OP_READ);
                    // Cache the connection.
                    TargetTuple tt = connectionManager.add(readKey, this); 
                    outboundConnectionsTuple.put(readKey,tt);
                }
                return false;
            }
        };
        selectorHandler.setPort(addr.getPort());
        selectorHandler.setInet(addr.getAddress());
        selectorHandler.setLinger(linger);
        selectorHandler.setLogger(logger);
        selectorHandler.setReuseAddress(true);
        selectorHandler.setSocketTimeout(keepAliveTimeoutInSeconds * 1000);
        return selectorHandler;
    }

    SSLSelectorHandler createTLSEndpoint(InetSocketAddress addr) {
        int linger = -1;

        SSLSelectorHandler selectorHandler = new SSLSelectorHandler();
        selectorHandler.setPort(addr.getPort());
        selectorHandler.setInet(addr.getAddress());
        selectorHandler.setLinger(linger);
        selectorHandler.setLogger(logger);
        selectorHandler.setReuseAddress(true);
        selectorHandler.setSocketTimeout(keepAliveTimeoutInSeconds * 1000);
        return selectorHandler;
    }

    UDPSelectorHandler createUdpEndpoint(InetSocketAddress addr) {
        UDPSelectorHandler selectorHandler = new UDPSelectorHandler();
        selectorHandler.setPort(addr.getPort());
        selectorHandler.setInet(addr.getAddress());
        selectorHandler.setLogger(logger);
        selectorHandler.setReuseAddress(true);
        return selectorHandler;
    }

    public void run() {
        try {
            // TODO: Use a thread pool
            new Thread() {

                @Override
                public void run() {
                    try {
                        clientController.start();
                    } catch (IOException ex) {
                        if (logger.isLoggable(Level.SEVERE)) {
                            logger.log(Level.SEVERE, "sip.stack.network.controller_death", new Object[] { "tcp", "client" });
							logger.log(Level.SEVERE, ex.getMessage(), ex);
                        }
                    }
                }
            }.start();

            serverController.start();
        } catch (IOException e) {
            if (logger.isLoggable(Level.SEVERE)) {
                logger.log(Level.SEVERE, "sip.stack.network.controller_death", new Object[] { "tcp", "server" });
				logger.log(Level.SEVERE, e.getMessage(), e);
            }
        }
    }

    /**
     * Return a pooled <code>ByteBuffer</code> containing the
     * SipMessage bytes.
     */
    protected static final ByteBuffer toBuffer(SipServletMessageImpl sipServletMessage)
            throws UnsupportedEncodingException {
        ByteBuffer bb = acquireBuffer();
        sipServletMessage.toBuffer(bb);
        bb.flip();
        return bb;
    }

    protected static ByteBuffer acquireBuffer() {
        ByteBuffer bb = byteBuffersPool.poll();
        if (bb == null) {
            bb = ByteBuffer.allocate(DEFAULT_BB_SIZE);
        }

        return bb;
    }

    protected static void releaseBuffer(ByteBuffer byteBuffer) {
        if (byteBuffer != null && byteBuffer.capacity() == DEFAULT_BB_SIZE) {
            byteBuffer.clear();
            byteBuffersPool.offer(byteBuffer);
        }
    }
    
    protected static ByteBuffer acquireSecureBuffer(int code, int expectedsize) {
        ByteBuffer securebb = secureByteBuffersPool.poll();
        if (securebb == null || securebb.capacity() < expectedsize){            
           securebb = ByteBuffer.allocate(expectedsize);
        }
        secureByteBufferMap.put(code + "", securebb);
        return securebb;
    }

    protected static void releaseSecureBuffer(int code) {        
        ByteBuffer securebb = secureByteBufferMap.remove(code+"");
        if (securebb != null){
            secureByteBuffersPool.offer(securebb);
        }        
    }
    class ConnectionManager {

        private int maxCachedBB = 50;
        private ConcurrentHashMap<TargetTuple, ConnectorHandler> streams = 
                new ConcurrentHashMap<TargetTuple, ConnectorHandler>();

        public ConnectionManager() {
            maxCachedBB = maxCachedByteBuffers;
        }

        private TargetTuple add(SelectionKey key, TCPSelectorHandler selectorHandler){                        
            SocketChannel socketChannel = (SocketChannel)key.channel();
            
            // TODO Cache object instead of invoking new on every request.
            TCPOutboundConnectorHandler outboundConnector = new TCPOutboundConnectorHandler();
            outboundConnector.setSelectorHandler(selectorHandler);
            outboundConnector.setUnderlyingChannel(socketChannel);

            Socket s = socketChannel.socket();
            InetSocketAddress remoteAddress = (InetSocketAddress) s.getRemoteSocketAddress();
            TargetTuple remote = new TargetTuple(SipTransports.TCP_PROT, remoteAddress);
            
            outboundConnector.setCallbackHandler(new SharedCallbackHandler(outboundConnector, remote));

            streams.put(remote, outboundConnector);
            return remote;
        }
        
        private void remove(TargetTuple tt){
            streams.remove(tt);
        }
        
        private void send(SipServletMessageImpl sipServletMessage)
                throws IOException {
            send(sipServletMessage, true, null);
        }

        private void send(SipServletMessageImpl sipServletMessage,
                boolean useCache, ByteBuffer bb) throws IOException {
            
            if (logger.isLoggable(Level.FINE)) {
                if( sipServletMessage.getMessageType() == SipServletMessageImpl.SipMessageType.SipRequest ) {
                    logger.log(Level.FINE, "Network OUT request to remote:'" + sipServletMessage.getRemote() + "' from local: '" + sipServletMessage.getLocal() + "' --> \r\n" + sipServletMessage.toString());
                } else {
                    logger.log(Level.FINE, "Network OUT response to remote:'" + sipServletMessage.getRemote() + "' from local: '" + sipServletMessage.getLocal() + "' --> \r\n" + sipServletMessage.toString());
                }
            }

            TargetTuple tt = sipServletMessage.getRemote();
            ConnectorHandler handler = null;
            if (useCache) {
                handler = streams.get(tt);

                if (handler != null && handler.getUnderlyingChannel() != null && !handler.getUnderlyingChannel().isOpen()) {
                    try {
                        handler.close();
                    } catch(Exception e) {
                        if (logger.isLoggable(Level.FINE)) {
                            logger.log(Level.FINE, 
                                    "Unexpected exception when closing ConnectorHandler: " + 
                                    handler, e);
                        }
                    }

                    connectionManager.remove(tt); //Cleanup a broken connection
                    handler = null;
                }
            }

            if (handler == null) {
                handler = createConnectorHandler(tt);
            }

            // The connection failed, returning null.
            if (handler == null) {
                throw new ConnectException("Connection refused to: " +
                        tt.getSocketAddress());
            }

            boolean cacheBuffer = false;
            if (bb == null) {
                sipServletMessage.toBufferInit();
                bb = toBuffer(sipServletMessage);
                cacheBuffer = true;
            }

            if (!bb.hasRemaining()) {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.log(Level.SEVERE, "sip.stack.network.invalid_bytebuffer");
                }
                return;
            }

            try {
                synchronized(handler.getUnderlyingChannel()) {
                    ((AsyncQueueWritable) handler).writeToAsyncQueue(bb, asyncWriteCallbackHandler);
                    while (sipServletMessage.toBufferHasRemaining()) {
                        //This is a big message
                        bb = toBuffer(sipServletMessage);
                        ((AsyncQueueWritable) handler).writeToAsyncQueue(bb, asyncWriteCallbackHandler);
                    }
                }
            } catch (IOException ex) {
                logger.log(Level.SEVERE, "Unable to write", ex);
                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE, "Unable to write", ex);
                }

                try {
                    handler.close();
                } catch (Exception e) {
                    if (logger.isLoggable(Level.FINE)) {
                        logger.log(Level.FINE,
                                "Unexpected exception when closing ConnectorHandler: " +
                                handler, e);
                    }
                }

                connectionManager.remove(tt); //Cleanup a broken connection
                if (cacheBuffer) {
                    releaseBuffer(bb);
                }
                

                // Try with a non cached connection as the connection is 
                // broken.
                if (useCache) {
                    //Changed to null since message can be multiple buffers
                    send(sipServletMessage, false, null);
                } else {
                    throw ex;
                }
            }
        }

        /**
         * Create client <code>ConnectorHandler</code> depending on <code>TargetTuple</code>
         * @param tt <code>TargetTuple</code>
         * @return <code>ConnectorHandler</code>, or null if error happened
         * @throws java.io.IOException
         */
        private ConnectorHandler createConnectorHandler(TargetTuple tt) throws IOException {
            ConnectorHandler handler = null;
            int protocol = tt.getProtocol().ordinal();
            
            if (protocol == SipTransports.TCP) {
                handler = createHandlerTCP(tt);
            } else if (handler == null && protocol == SipTransports.TLS) {
                handler = createHandlerTLS(tt);
            } else if (handler == null) {
                handler = createHandlerUDP(tt);
            }
            
            if (handler != null) {
                SelectableChannel channel = handler.getUnderlyingChannel();
                if (channel != null && channel.isOpen()) {
                    streams.putIfAbsent(tt, handler);
                } else {
                    return null;
                }
            }
            
            return handler;
        }
        
        private synchronized ConnectorHandler createHandlerTCP(final TargetTuple tt) throws IOException {
            final InetSocketAddress remote = tt.getSocketAddress();
            final TCPConnectorHandler connectorHandler = (TCPConnectorHandler) clientController.acquireConnectorHandler(Protocol.TCP);

            try {
                connectorHandler.connect(remote, new SharedCallbackHandler(connectorHandler,tt));
                return connectorHandler;
            } catch (Throwable t) {
                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE, "Connect failed:" + remote, t);
                }
                return null;
            }
        }

        private final void waitOnLatch(CountDownLatch latch,
                int timeout, TimeUnit timeUnit) throws InterruptedException {
            latch.await(timeout, timeUnit);
        }

        private synchronized ConnectorHandler createHandlerTLS(final TargetTuple tt) throws IOException {
            final InetSocketAddress remote = tt.getSocketAddress();
            final SSLConnectorHandler connectorHandler = (SSLConnectorHandler) new SSLConnectorHandler(sslContext);

            connectorHandler.setController(clientController);
            final CountDownLatch handshakeDoneLatch = new CountDownLatch(1);

            SSLCallbackHandler<Context> sslCallbackHander = 
                    new SSLSharedCallbackHandler(connectorHandler, tt, handshakeDoneLatch);

            try {
                connectorHandler.connect(remote, sslCallbackHander);
                // This makes sure the handshake operations is completed
                // before marking this connectorHandler as ready. Might
                // be a performance bottleneck.
                waitOnLatch(handshakeDoneLatch, 10, TimeUnit.SECONDS);
                return connectorHandler;
            } catch (Throwable t) {
                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE, "Connect failed:" + remote, t);
                }
                return null;
            }
        }

        private synchronized ConnectorHandler createHandlerUDP(final TargetTuple tt) throws IOException {
            final InetSocketAddress remote = tt.getSocketAddress();
            final UDPConnectorHandler connectorHandler = (UDPConnectorHandler) clientController.acquireConnectorHandler(Protocol.UDP);

            try {
				if (useUDPPortForClient){
                    connectorHandler.connect(remote, localUDPSocketAddress, new SharedCallbackHandler(connectorHandler, tt));
				} else {
                    connectorHandler.connect(remote, new SharedCallbackHandler(connectorHandler, tt));

				}
                return connectorHandler;
            } catch (Throwable t) {
                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE, "Connect failed:" + remote, t);
                }
                return null;
            }
        }
    }

    class SharedSelectionKeyHandler extends DefaultSelectionKeyHandler {
         @Override
         public void expire(Iterator<SelectionKey> keys) {
         } // The connections are never going to be closed
                    
         @Override
         public void cancel(SelectionKey key) {
              super.cancel(key);
              TargetTuple tt = outboundConnectionsTuple.remove(key);
              if (tt != null) {
                  connectionManager.remove(tt);
              }
         }                     
    }    

    class SharedReadFilter extends ReadFilter {

        private int bufferSize = 16 * 8192;

        @Override
        public boolean execute(Context ctx) throws IOException {
            SelectableChannel channel = ctx.getSelectionKey().channel();

            // Make sure we remove that attribute in case the 
            // ProtocolChain is not the default one.
            ctx.removeAttribute(ProtocolFilter.SUCCESSFUL_READ);            
            /*
             * Read filter should be executed only for READ or READ_WRITE
             * operations and never for WRITE. Printing a fine message.
             */
            if (ctx.getCurrentOpType() == Context.OpType.OP_WRITE) {
                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE, "Shared Read Filter invoked when OpType is WRITE");
                }
                return false;
            }

            if (!channel.isOpen()) {
                ctx.setKeyRegistrationState(KeyRegistrationState.CANCEL);
                return false;
            }
            WorkerThread workerThread =
                    (WorkerThread) Thread.currentThread();

            // Re-size the ByteBuffer based on the underlying OS buffer.
            if (workerThread.getByteBuffer().capacity() != bufferSize) {
                try {
                    if (ctx.getProtocol() == Protocol.TCP) {
                        bufferSize = ((SocketChannel) channel).socket().getReceiveBufferSize();
                    } else {
                        bufferSize = ((DatagramChannel) channel).socket().getReceiveBufferSize();
                    }
                } catch (IOException ex) {
                    if (logger.isLoggable(Level.WARNING)) {
                        logger.log(Level.WARNING, "sip.stack.network.resize_buffer_failed_bad_Socket_state");
						logger.log(Level.WARNING, ex.getMessage(), ex);
                    }
                    bufferSize = 16 * 8192;
                }
                workerThread.setByteBuffer(ByteBuffer.allocate(bufferSize));
            }

            boolean invokeNextFilter = super.execute(ctx);
            if (!invokeNextFilter){
                TargetTuple tt = (TargetTuple) ctx.getAttribute("tt");
                if (tt != null){
                    connectionManager.remove(tt);
                }
            }
            return invokeNextFilter;
        }
        
        @Override
        public boolean postExecute(Context ctx) throws IOException {
            Boolean isClientExecution =
                    (Boolean) ctx.removeAttribute(IS_CLIENT_EXECUTION);
            /*
             * Read filter should be executed only for READ or READ_WRITE
             * operations and never for WRITE.Print log in fine.
             */
            if (ctx.getCurrentOpType() == Context.OpType.OP_WRITE) {
                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE, "SharedReadFilter postExecute invoked when OpType is WRITE");
                }
                return true;
            }
            // isClientExecution will be null when this filter is used
            // as a server side component, and not null when used from a 
            // client side component. When used by a client side component, 
            // do not override the decision of closing or not the SelectionKey
            if (isClientExecution == null 
                    && ctx.getProtocol() == Controller.Protocol.UDP 
                    && ctx.getKeyRegistrationState() != Context.KeyRegistrationState.REGISTER) {
                // Main UDP listener should never be closed!
                ctx.setKeyRegistrationState(Context.KeyRegistrationState.REGISTER);
            } else if (isClientExecution != null &&
                    ctx.getProtocol() == Controller.Protocol.UDP
		            && useUDPPortForClient)	{
                /**
                 *  When the UDP client socket is bound to the local address to
                 *  ensure that outgoing requests are from the same UDP (server)
                 *  port, the response also have to be read from the same UDP
                 *  datagram channel from which the request was sent. The response
                 *  does not come to the main UDP channel. NIO seems to be
                 *  processing datagrams based on the soure port.
                 *  The problem here is that since this filter (chain) was 
                 *  executed from the callback
                 *  handler and the interest ops cleared, we have to register the 
                 *  read ops here so that we are able to read further messages
                 *  from the client. 
                 */
                ctx.getSelectorHandler().register(ctx.getSelectionKey(), 
                                SelectionKey.OP_READ);
				/* We have a system property to switch on this
				 * feature because there is a potential fd leak
				 * here, UDP keys do not expire and we register 
				 * one key per new datagram channel that is created.
				 */

             }

            return super.postExecute(ctx);
        }
    }


    class ClientCertSSLReadFilter extends SSLReadFilter {
        @Override
        public boolean execute(Context ctx) throws IOException {
            // Make sure we remove that attribute in case the 
            // ProtocolChain is not the default one.
            ctx.removeAttribute(ProtocolFilter.SUCCESSFUL_READ);
            
            SelectableChannel channel = ctx.getSelectionKey().channel();
            
            if (ctx.getCurrentOpType() == Context.OpType.OP_WRITE) {
                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE, "ClientSSLReadFilter" +
                            " execute invoked when OpType is WRITE");
                }
                return false;
            }
             

            if (!channel.isOpen()){
                ctx.setKeyRegistrationState
                        (KeyRegistrationState.CANCEL);
                return false;
            }

            boolean continueExecution = super.execute(ctx);         
            if (continueExecution) {
                Object[] x509Cert = null;
                ByteBuffer byteBuffer =
                        ((WorkerThread) Thread.currentThread()).getByteBuffer();
                ByteBuffer bb = acquireBuffer();
                ((WorkerThread) Thread.currentThread()).setByteBuffer(bb);
                try {
                    x509Cert =
                            doPeerCertificateChain(ctx.getSelectionKey(),
                            isNeedClientAuth());
                } catch (IOException ioe) {
                    if (logger.isLoggable(Level.WARNING)) {
                        logger.log(Level.WARNING,
                                "sip.stack.network.client_cert_error");
						logger.log(Level.WARNING, ioe.getMessage(), ioe);
                    }
                } finally {
                    releaseBuffer(bb);
                    ((WorkerThread) Thread.currentThread()).setByteBuffer(byteBuffer);
                }
                ctx.setAttribute(SIP_CERTS, x509Cert);               
            }                 
            return continueExecution;
        }

        @Override
        public boolean postExecute(Context ctx) throws IOException {
            
            if (ctx.getCurrentOpType() == Context.OpType.OP_WRITE) {
                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE, "ClientSSLReadFilter postexecute" +
                            " invoked when OpType is WRITE");
                }
                return true;
            } 
                       
            return super.postExecute(ctx);
        }
    }
        
    class SimpleProtocolChainInstanceHandler implements ProtocolChainInstanceHandler {
        ProtocolChain protocolChain = null;

        public SimpleProtocolChainInstanceHandler(ProtocolChain aProtocolChain) {
            protocolChain = aProtocolChain;
        }
        
        public ProtocolChain getProtocolChain() {
            return protocolChain;
        }
        
        public ProtocolChain poll() {
            return protocolChain;
        }

        public boolean offer(ProtocolChain instance) {
            return true;
        }
    }

    class SharedCallbackHandler implements CallbackHandler<Context>, CallbackHandlerDescriptor {
        protected ConnectorHandler connectorHandler;
        protected TargetTuple targetTuple;

        public SharedCallbackHandler(ConnectorHandler connectorHandler, TargetTuple targetTuple) {
            this.connectorHandler = connectorHandler;
            this.targetTuple = targetTuple;
        }
                
        public void onConnect(IOEvent<Context> ioEvent) {
            SelectionKey key = ioEvent.attachment().getSelectionKey();
            try {
                connectorHandler.finishConnect(key);
                key.interestOps(SelectionKey.OP_READ);
            } catch (IOException ex) {
                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE, "Connect failed" + targetTuple.getSocketAddress(), ex);
                }
                connectionManager.remove(targetTuple);
                ioEvent.attachment().getSelectorHandler().
                        getSelectionKeyHandler().cancel(key);
            }
        }

        public void onRead(IOEvent<Context> ioEvent) {
            try {
                Context ctx = ioEvent.attachment();
                ctx.setAttribute(IS_CLIENT_EXECUTION, true);
                SelectionKey key = ctx.getSelectionKey();
                if (!key.isValid()) {
                    connectionManager.remove(targetTuple);
                    return;
                }

                ctx.setAttribute("tt", targetTuple);

                // disable OP_READ on key before doing anything else
                key.interestOps(key.interestOps() & (~SelectionKey.OP_READ));
                ctx.getProtocolChain().execute(ioEvent.attachment());
            } catch (Throwable e) {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.log(Level.SEVERE, "sip.stack.network.connection_read_failed", new Object[] { "tcp" });
					logger.log(Level.SEVERE, e.getMessage(), e);
                }
            }
        }

        public void onWrite(IOEvent<Context> ioEvent) {
        }

        public boolean isRunInSeparateThread(OpType opType) {
            return opType != OpType.OP_CONNECT;
        }
    };
    
    class SSLSharedCallbackHandler extends SharedCallbackHandler implements SSLCallbackHandler<Context> {
        protected ByteBuffer handshakeAppBuffer;
        protected CountDownLatch handshakeLatch;
        
        public SSLSharedCallbackHandler(SSLConnectorHandler connectorHandler, 
                TargetTuple targetTuple, CountDownLatch handshakeLatch) {
            super(connectorHandler, targetTuple);
            this.handshakeLatch = handshakeLatch;
        }
        
        @Override
        public void onConnect(IOEvent<Context> ioEvent) {
            SSLConnectorHandler sslConnectorHandler = (SSLConnectorHandler) this.connectorHandler;
            SelectionKey key = ioEvent.attachment().getSelectionKey();
            try {
                sslConnectorHandler.finishConnect(key);
                WorkerThread workerThread = (WorkerThread)Thread.currentThread();
                ThreadAttachment attachment = 
                        workerThread.updateAttachment(Mode.SSL_ENGINE);
                key.attach(attachment);

                handshakeAppBuffer = acquireBuffer();
                if (sslConnectorHandler.handshake(handshakeAppBuffer, true)) {
                    onHandshake(ioEvent);
                }
                //key.interestOps(SelectionKey.OP_READ);
				System.out.println("ONCONNECT register");
                ioEvent.attachment().getSelectorHandler().register(key,
                        SelectionKey.OP_READ);                
				System.out.println("ONCONNECT registered");
						
            } catch (Exception ex) {
                if (logger.isLoggable(Level.FINE)) {
                    logger.log(Level.FINE, "Connect failed:" + targetTuple.getSocketAddress(), ex);
                }
                
                releaseBuffer(handshakeAppBuffer);
                handshakeAppBuffer = null;
                connectionManager.remove(targetTuple);
                ioEvent.attachment().getSelectorHandler().getSelectionKeyHandler().cancel(key);
            }
        }

        public void onRead(IOEvent<Context> ioEvent) {
            try {
                Context ctx = ioEvent.attachment();
                ctx.setAttribute(IS_CLIENT_EXECUTION, true);
                SelectionKey key = ctx.getSelectionKey();
                if (!key.isValid()) {
                    connectionManager.remove(targetTuple);
                    return;
                }

                ctx.setAttribute("tt", targetTuple);

                // disable OP_READ on key before doing anything else
                key.interestOps(key.interestOps() & (~SelectionKey.OP_READ));
                System.out.println("ONREAD SSLCALLBACK");
                ctx.getProtocolChain().execute(ioEvent.attachment());
            } catch (Throwable e) {
                if (logger.isLoggable(Level.SEVERE)) {
                    logger.log(Level.SEVERE, "sip.stack.network.connection_read_failed", new Object[] { "tcp" });
					logger.log(Level.SEVERE, e.getMessage(), e);
                }
            }
        }
        // After handshake completes
        public void onHandshake(IOEvent<Context> ioEvent) {
            releaseBuffer(handshakeAppBuffer);
            handshakeAppBuffer = null;
            handshakeLatch.countDown();
        }
    }
    
    /**
     * AsyncWriteCallbackHandler, which releases <code>ByteBuffer</code>s, allocated
     * to be used with AsyncWriteQueue
     */
    static class SharedAsyncWriteCallbackHandler implements AsyncWriteCallbackHandler {
        public void onWriteCompleted(SelectionKey key, ByteBuffer byteBuffer) {
            /*
             * Since we do not clone the byetbuffer, this bytebuffer object
             * should be the same as the one we used for calling 
             * asyncqueuewriter.write. We needed a unique identifier corresponding
             * to the securebb that was created in the dispatcher. Using the 
             * hashcode of the buffer, could not think of anything else.
             */
            releaseSecureBuffer(byteBuffer.hashCode());
            releaseBuffer(byteBuffer);            
        }

        public void onIOException(IOException exception, SelectionKey key, 
                ByteBuffer byteBuffer, Queue<AsyncWriteQueueRecord> queue) {
            releaseSecureBuffer(byteBuffer.hashCode());
            releaseBuffer(byteBuffer);
            for(AsyncWriteQueueRecord record : queue) {
                releaseBuffer(record.getByteBuffer());
            }
        }        
    }
    
    //secure
    @Configuration(key = "javax.net.ssl.keyStore", update = UpdatePolicy.STARTUP)
    public void setSslKeyStore(String aKeyStore) {
        keyStore = aKeyStore;
    }

    @Configuration(key = "javax.net.ssl.trustStore", update = UpdatePolicy.STARTUP)
    public void setSslTrustStore(String aTrustStore) {
        trustStore = aTrustStore;
    }

    //grizzly pipelines and pools
    @Configuration(key = "TimeoutInSeconds", node = "/SipService/KeepAlive", update = UpdatePolicy.STARTUP, usage = UsagePolicy.IGNORE)
    public void setKeepAliveTimeoutInSeconds(int aKeepAliveTimeoutInSeconds) {
        keepAliveTimeoutInSeconds = aKeepAliveTimeoutInSeconds;
    }

    @Configuration(key = "ThreadCount", node = "/SipService/RequestProcessing", usage = UsagePolicy.IGNORE)
    public void setMaxPipelineThreads(int aMaxThreads) {
        maxPipelineThreads = aMaxThreads;
        reinitPipeline();
    }

    @Configuration(key = "sip.network.grizzly.useServerThreadPool", update = UpdatePolicy.STARTUP)
    public void setClientsShareServerThreadPool(boolean isShared) {
        clientsShareServerThreadPool = isShared;
    }

    //Is checked with Jeanfrancios, can be mapped to the initial-thread-count.
    //This means that the min and initial are the same!
    @Configuration(key = "InitialThreadCount", node = "/SipService/RequestProcessing", usage = UsagePolicy.IGNORE)
    public void setMinPipelineThreads(int aMinThreads) {
        minPipelineThreads = aMinThreads;
        reinitPipeline();
    }

    @Configuration(key = "ThreadIncrement", node = "/SipService/RequestProcessing", usage = UsagePolicy.IGNORE)
    public void setPipelineThreadsIncrement(int aThreadsIncrement) {
        pipelineThreadsIncrement = aThreadsIncrement;
    }

    @Configuration(key = "HeaderBufferLengthInBytes", node = "/SipService/RequestProcessing", usage = UsagePolicy.IGNORE)
    public void setPipelineInitialByteBufferSize(int anInitialByteBufferSize) {
        pipelineInitialByteBufferSize = anInitialByteBufferSize;
    }

    //grizzly buffers
    //@Configuration (key="sip.network.grizzly.maxCachedByteBuffer", update=UpdatePolicy.STARTUP)    
    @Configuration(key = "MaxPendingCount", node = "/SipService/ConnectionPool", usage = UsagePolicy.IGNORE)
    public void setMaxCachedByteBuffers(int aMaxCachedByteBuffers) {
        maxCachedByteBuffers = aMaxCachedByteBuffers;
    }
}
