/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package com.ericsson.ssa.container;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;

import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketTimeoutException;

import java.util.logging.Level;

// inserted by hockey (automatic)
import java.util.logging.Logger;
import org.jvnet.glassfish.comms.util.LogUtil;


/**
 * This class encapsulates the socket and stream handling for communicating with
 * a specific host over HTTP.
 *
 * @author epiesan
 * @since Apr 16, 2006
 */
public class HttpConnection {
    private static Logger log = LogUtil.SIP_LOGGER.getLogger();
    protected final static int DEFAULT_SOCKET_TIMEOUT = 120000;
    protected final static int MAX_RECEIVE_BUFFER_SIZE = 2048;
    protected final static int MAX_SEND_BUFFER_SIZE = 2048;
    private static final byte[] NEWLINE = new byte[] { 0x0d, 0x0a };
    private HttpConnectionPool pool;
    private HttpHost host;
    private Socket socket;
    private int socketTimeout;
    private boolean isOpen;
    private InputStream inputStream;
    private OutputStream outputStream;

    /**
     * Constructor for HttpConnection.
     *
     * @param host
     *           The host this connection is for.
     */
    public HttpConnection(final HttpHost host) {
        this(host, DEFAULT_SOCKET_TIMEOUT);
    }

    /**
     * Constructor for HttpConnection.
     *
     * @param host
     *           The host this connection is for.
     * @param socketTimeout
     *           The socket timeout value for this connection's socket.
     */
    public HttpConnection(final HttpHost host, int socketTimeout) {
        this.host = host;
        this.socketTimeout = socketTimeout;
    }

    /**
     * Constructor for HttpConnection.
     *
     * @param host
     *           The host this connection is for.
     * @param pool
     *           The connection pool that manages this connection.
     */
    protected HttpConnection(final HttpHost host, HttpConnectionPool pool) {
        this(host, DEFAULT_SOCKET_TIMEOUT, pool);
    }

    /**
     * Constructor for HttpConnection.
     *
     * @param host
     *           The host this connection is for.
     * @param socketTimeout
     *           The socket timeout value for this connection's socket.
     * @param pool
     *           The connection pool that manages this connection.
     */
    protected HttpConnection(final HttpHost host, int socketTimeout,
        HttpConnectionPool pool) {
        this.host = host;
        this.socketTimeout = socketTimeout;
        this.pool = pool;
    }

    /**
     * Returns the host for this connection.
     *
     * @return The host.
     */
    public HttpHost getHost() {
        return host;
    }

    /**
     * Returns the underlyin socket of this connection.
     *
     * @return The underlying socket.
     */
    public Socket getSocket() {
        return socket;
    }

    /**
     * Checks if this connection has been opened.
     *
     * @return True if this connection as been opened.
     */
    public boolean isOpen() {
        return isOpen;
    }

    /**
     * Determines if this connection is stale, that is, a read would fail or this
     * connection is already closed.
     *
     * @return True if this connection is stale.
     * @throws IOException
     *            If the stale testing is interrupted.
     */
    public boolean isStale() throws IOException {
        boolean isStale = true;

        if (isOpen()) {
            try {
                isStale = false;

                if (inputStream.available() <= 0) {
                    socket.setSoTimeout(1);
                    inputStream.mark(1);

                    try {
                        if (inputStream.read() == -1) {
                            isStale = true;
                        } else {
                            inputStream.reset();
                        }
                    } finally {
                        socket.setSoTimeout(socketTimeout);
                    }
                }
            } catch (InterruptedIOException e) {
                if (!(e instanceof SocketTimeoutException)) {
                    throw e;
                }
            } catch (IOException e) {
                isStale = true;
            }
        }

        return isStale;
    }

    /**
     * Establishes a connection to the specified host.
     *
     * @throws IOException
     *            If the connection attempt fails.
     */
    public void open() throws IOException, IllegalStateException {
        if (isOpen()) {
            throw new IllegalStateException("The connection is already opened.");
        }

        try {
            InetSocketAddress socketAddress = new InetSocketAddress(host.getHostname(),
                    host.getPort());

            // Create the socket if not already done.
            if (socket == null) {
                socket = new Socket();

                InetAddress ia = InetAddress.getByName(OLDNetworkManager.getInstance().getCurrentLocalHost());
                InetSocketAddress localBind = new InetSocketAddress(ia, 0);
                socket.bind(localBind);
                socket.connect(socketAddress);
            }

            // Set the configured socket timeout for this connection.
            socket.setSoTimeout(socketTimeout);

            // Set the buffer sizes to the sockets buffer sizes, but limit it to
            // 2048 bytes.
            int inBufSize = socket.getReceiveBufferSize();

            if ((inBufSize <= 0) || (inBufSize > MAX_RECEIVE_BUFFER_SIZE)) {
                inBufSize = MAX_RECEIVE_BUFFER_SIZE;
            }

            int outBufSize = socket.getSendBufferSize();

            if ((outBufSize <= 0) || (outBufSize > MAX_SEND_BUFFER_SIZE)) {
                outBufSize = MAX_SEND_BUFFER_SIZE;
            }

            // Wrap the sockets streams in buffered streams.
            inputStream = new BufferedInputStream(socket.getInputStream(),
                    inBufSize);
            outputStream = new BufferedOutputStream(socket.getOutputStream(),
                    outBufSize);

            // Uncomment this to log all output and input to System.out.
            // outputStream = new TeeOutputStream(outputStream, System.out);
            // inputStream = new TeeInputStream(inputStream, System.out);

            // Mark this connection as properly opened.
            isOpen = true;
        } catch (IOException e) {
            // Unable to properly open the connection, so close it.
            closeConnection();

            // Rethrow the exception.
            throw e;
        }
    }

    /**
     * Closes the connection.
     */
    public void close() {
        closeConnection();
    }

    /**
     * Releases this connection to the connection pool.
     *
     * @throws UnsupportedOperationException
     *            If this connection is not managed by any connection pool.
     */
    public void release() throws UnsupportedOperationException {
        if (pool == null) {
            throw new UnsupportedOperationException(
                "This connection is not managed by any connection pool.");
        }

        // Release this connection.
        pool.releaseConnection(this);
    }

    /**
     * Returns an OutputStream suitable for writing the request to.
     *
     * @throws IllegalStateException
     *            If the connection is not yet opened.
     * @return The OutputStream to write the request to.
     */
    public OutputStream getOutputStream() throws IllegalStateException {
        assertOpened();

        return outputStream;
    }

    /**
     * Returns an InputStream suitable for reading the response from.
     *
     * @throws IllegalStateException
     *            If the connection is not yet opened.
     * @return The InputStream to read the response from.
     */
    public InputStream getInputStream() throws IllegalStateException {
        assertOpened();

        return inputStream;
    }

    /**
     * Flushes the output stream of tis connection.
     *
     * @throws IOException
     *            if any IO problem occurs.
     */
    public void flushOutputStream() throws IOException {
        assertOpened();

        outputStream.flush();
    }

    /**
     * Reads a complete line from the connections input stream and strips of the
     * last \r\n characters. This method default to using the ISO-8859-1 charset
     * when converting the read bytes to characters.
     *
     * @return The read line.
     */
    public String readLine() throws IOException {
        return readLine("ISO-8859-1");
    }

    /**
     * Reads a complete line from the connections input stream and strips of the
     * last \r\n characters
     *
     * @param charset
     *           The charset to use when converting from bytes to characters.
     * @return The read line.
     */
    public String readLine(String charset) throws IOException {
        assertOpened();

        ByteArrayOutputStream buf = new ByteArrayOutputStream();
        int ch;

        while ((ch = inputStream.read()) >= 0) {
            buf.write(ch);

            if (ch == '\n') { // be tolerant (RFC-2616 Section 19.3)

                break;
            }
        }

        if (buf.size() == 0) {
            return null;
        }

        byte[] buffer = buf.toByteArray();

        // strip CR and LF from the end
        int len = buffer.length;
        int offset = 0;

        if (len > 0) {
            if (buffer[len - 1] == '\n') {
                offset++;

                if (len > 1) {
                    if (buffer[len - 2] == '\r') {
                        offset++;
                    }
                }
            }
        }

        return new String(buffer, 0, len - offset, charset);
    }

    /**
     * Writes the specified bytes to the output stream.
     *
     * @param data
     *           The data to be written
     * @throws IOException
     *            If an IO problem occurs.
     * @throws IllegalStateException
     *            if not connected
     */
    public void write(byte[] data) throws IOException, IllegalStateException {
        this.write(data, 0, data.length);
    }

    /**
     * Writes bytes to the output stream.
     *
     * @param data
     *           Byte array containing the data to be written.
     * @param offset
     *           The start offset.
     * @param length
     *           The number of bytes to write.
     * @throws IOException
     *            if any IO problem occurs.
     * @throws IllegalStateException
     *            If the connection is not already opened.
     */
    public void write(byte[] data, int offset, int length)
        throws IOException, IllegalStateException {
        assertOpened();
        outputStream.write(data, offset, length);
    }

    /**
     * Writes the specified bytes, followed by a newline ("\r\n") to the output
     * stream of this connection.
     *
     * @param data
     *           The bytes to be written.
     * @throws IOException
     *            If any IO problem occurs.
     * @throws IllegalStateException
     *            if the connection is not already opened.
     */
    public void writeLine(byte[] data)
        throws IOException, IllegalStateException {
        write(data);
        writeLine();
    }

    /**
     * Writes a newline ("\r\n") to the output stream.
     *
     * @throws IOException
     *            If an /O problem occurs.
     * @throws IllegalStateException
     *            if the connection is not already opened.
     */
    public void writeLine() throws IOException, IllegalStateException {
        write(NEWLINE);
    }

    /**
     * Prints the specified string to the output stream using the default charset
     * "ISO-8859-1".
     *
     * @param data
     *           The string to print.
     * @throws IOException
     *            If any IO problem occurs.
     * @throws IllegalStateException
     *            If this connection is not yet opened.
     */
    public void print(String data) throws IOException, IllegalStateException {
        print(data, "ISO-8859-1");
    }

    /**
     * Prints the specified string to the output stream using the specified
     * charset.
     *
     * @param data
     *           The string to print.
     * @param charset
     *           The charset to use.
     * @throws IOException
     *            If any IO problem occurs.
     * @throws IllegalStateException
     *            If this connection is not yet opened.
     */
    public void print(String data, String charset)
        throws IOException, IllegalStateException {
        try {
            write(data.getBytes(charset));
        } catch (UnsupportedEncodingException e) {
            log.log(Level.WARNING,
                "Unsupported encoding '" + charset +
                "'. Using default system encoding instead.");

            write(data.getBytes());
        }
    }

    /**
     * Prints the specified line to the output stream using the specified
     * charset. Uses ISO-8859-1.
     *
     * @param data
     *           The line to print.
     * @throws IOException
     *            If any IO problem occurs.
     * @throws IllegalStateException
     *            If this connection is not yet opened.
     */
    public void printLine(String data)
        throws IOException, IllegalStateException {
        printLine(data, "ISO-8859-1");
    }

    /**
     * Prints the specified line to the output stream using the specified
     * charset.
     *
     * @param data
     *           The line to print.
     * @param charset
     *           The charset to use.
     * @throws IOException
     *            If any IO problem occurs.
     * @throws IllegalStateException
     *            If this connection is not yet opened.
     */
    public void printLine(String data, String charset)
        throws IOException, IllegalStateException {
        try {
            writeLine(data.getBytes(charset));
        } catch (UnsupportedEncodingException e) {
            log.log(Level.WARNING,
                "Unsupported encoding '" + charset +
                "'. Using default system encoding instead.");

            writeLine(data.getBytes());
        }
    }

    @Override
    public String toString() {
        String socketInfo = "Socket not yet created";

        if (socket != null) {
            socketInfo = "localaddr=" + socket.getLocalSocketAddress() +
                ", remoteaddr=" + socket.getRemoteSocketAddress();
        }

        return "Connection(isOpen=" + isOpen() + ", SocketInfo=" + socketInfo +
        ")";
    }

    /**
     * Closes streams and the socket.
     */
    private void closeConnection() {
        if (inputStream != null) {
            try {
                inputStream.close();
            } catch (IOException e) {
                // Ignore
            } finally {
                inputStream = null;
            }
        }

        if (outputStream != null) {
            try {
                outputStream.close();
            } catch (IOException e) {
                // Ignore
            } finally {
                outputStream = null;
            }
        }

        if (socket != null) {
            try {
                socket.close();

                if (log.isLoggable(Level.FINE)) {
                    log.log(Level.FINE,
                        "The socket was closed for connection: " + this);
                }
            } catch (IOException e) {
                // Ignore
            } finally {
                socket = null;
            }
        }

        isOpen = false;
    }

    /**
     * Throws an exception if this connection is not opened.
     *
     * @throws IllegalStateException
     *            If the connection is not opened.
     */
    private void assertOpened() throws IllegalStateException {
        if (!isOpen()) {
            throw new IllegalStateException(
                "The connection must to be opened first.");
        }
    }
}
