/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package com.ericsson.ssa.container;

import java.net.URI;


/**
 * HttpHost holds all information necessary to describe
 * an HTTP connection to a specific host.
 *
 * @author epiesan
 * @since Apr 16, 2006
 */
public final class HttpHost implements Cloneable {
    private final static int HTTP_DEFAULT_PORT = 80;
    private String hostname;
    private int port;

    /**
     * Constructor for HttpHost.
     *
     * @param hostname The hostname (may not be null).
     * @param port The port. Specify -1 to use the default port for HTTP.
     */
    public HttpHost(final String hostname, int port) {
        if (hostname == null) {
            throw new IllegalArgumentException("Host name must be specified.");
        }

        this.hostname = hostname;

        if (port < 0) {
            this.port = HTTP_DEFAULT_PORT; // Set default port for HTTP
        } else {
            this.port = port;
        }
    }

    /**
     * Constructor for HttpHost.
     *
     * @param hostname The hostname (may not be null).
     */
    public HttpHost(final String hostname) {
        this(hostname, -1);
    }

    /**
     * Constructor for HttpHost.
     *
     * @param uri The URI.
     */
    public HttpHost(final URI uri) {
        this(uri.getHost(), uri.getPort());
    }

    /**
     * Constructor needed for implementation of clone().
     *
     * @param source The source HttpHost to copy values from.
     */
    private HttpHost(final HttpHost source) {
        this.hostname = source.hostname;
        this.port = source.port;
    }

    /**
     * Returns the host name.
     *
     * @return The host name.
     */
    public String getHostname() {
        return hostname;
    }

    /**
     * Returns the port of this host.
     *
     * @return The port.
     */
    public int getPort() {
        return port;
    }

    /**
     * Clones this HttpHost instance.
     */
    public Object clone() {
        return new HttpHost(this);
    }

    /**
     * @see java.lang.Object#equals(java.lang.Object)
     */
    public boolean equals(final Object obj) {
        if (obj instanceof HttpHost) {
            if (obj == this) {
                return true;
            }

            HttpHost that = (HttpHost) obj;

            if (!this.hostname.equalsIgnoreCase(that.hostname)) {
                return false;
            }

            if (this.port != that.port) {
                return false;
            }

            return true;
        } else {
            return false;
        }
    }

    /**
     * @see java.lang.Object#hashCode()
     */
    public int hashCode() {
        int hash = 7;

        hash = (31 * hash) + hostname.hashCode();
        hash = (31 * hash) + port;

        return hash;
    }

    /**
     * Returns a string representation of the HttpHost instance.
     * The return string is in the form 'http://<host>:<port>'.
     *
     * @return The string representation of this HttpHost instance.
     */
    public String toString() {
        StringBuffer s = new StringBuffer();
        s.append("http://");
        s.append(this.hostname);

        if (this.port != HTTP_DEFAULT_PORT) {
            s.append(':');
            s.append(this.port);
        }

        return s.toString();
    }
}
