/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package com.ericsson.ssa.container;

import java.io.IOException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;

import javax.servlet.ServletException;

import org.jvnet.glassfish.comms.httplayers.HttpLayer;
import org.jvnet.glassfish.comms.util.LogUtil;
import java.util.logging.Logger;

import com.ericsson.ssa.config.ConfigFactory;
import com.ericsson.ssa.config.annotations.Configuration;
import com.ericsson.ssa.container.callflow.CallflowResolver;
import com.ericsson.ssa.container.callflow.Reporter;
import com.ericsson.ssa.fm.FmEventSender;
import com.ericsson.ssa.sip.Layer;
import com.ericsson.ssa.sip.LayerHelper;
import com.ericsson.ssa.sip.SipServletRequestImpl;
import com.ericsson.ssa.sip.SipServletResponseImpl;
import com.ericsson.ssa.sip.timer.GeneralTimer;
import com.ericsson.ssa.sip.timer.GeneralTimerListener;
import com.ericsson.ssa.sip.timer.TimerServiceImpl;
import com.ericsson.util.os.OSUtil;
import com.ericsson.util.os.OSUtilFactory;
import com.sun.grizzly.tcp.Request;
import com.sun.grizzly.tcp.Response;

/**
 * Protects the container from overload. <br>
 * When the container detects overload (see below for details on when this is) two
 * actions can be taken:
 * <list>
 * <li>the request is rejected with a 503 response</li>
 * <br>
 * Status code 503 is
 * indicating that the server is currently unable to handle the request due to a
 * temporary overloading or maintenance of the server. 
 * <li>
 * the request is silently dropped
 * <br>This is done in case of high load, where a 'nice' handling of the 
 * request would still consume too many resource.
 * </li>
 * </list>
 * <p>
 * <br>
 * Overload protection applies both to memory consumption and CPU usage
 * <br>
 * There are different thresholds defined for CPU and Memory usage, and different
 * thresholds for HTTP and SIP traffic.
 * Normally there are three counters in increasing order:<br>
 * A threshold were initial requests are rejected (<code>IrThreshold</code>, <code>MemIrThreshold</code>, <code>HttpThreshold</code>, <code>HttpMemThreshold</code>)<br>
 * A threshold were subsequent requests are rejected (<code>SrThreshold</code>) This is not applicable for memory,
 * since subsequent requests are expected to cause less memory consumption.<br>
 * A threshold were all requests are dropped (<code>MmThreshold</code>, <code>MemMmThreshold</code>). These are shared between sip and http.
 * <br><br>
 * The actions taken by this manager (e.g., reject with 503 or drop) are <em>only</em>
 * when the actual values exceed the thresholds for a consecutive number of
 * times. This value is controlled by the <code>NumberOfSamples</code>
 * parameter. The measurement interval is controlled by the <code>SampleRate</code>
 * parameter. So in an overload it takes at least <code>NumberOfSample * 
 * SampleRate</code> seconds before the overload manager kicks in.
 * <p>
 * Overload protection can be dynamically enabled and disabled using the boolean parameters:
 * <code>cpuOverloadRegulationEnabled</code> and <code>memOverloadRegulationEnabled</code>.
 * @author ehsroha
 * @author erikvandervelden
 * @since 2006-mar-29
 * @reviewed ehswolm 2007-mar-22
 * @reviewed ehswolm 2007-jun-18
 */
public final class OverloadProtectionManager implements Layer,
        GeneralTimerListener, HttpLayer {
    /**
     * Default threshold percentage level of the CPU for initial SIP requests.
     */
    private static final int DEFAULT_IR_CPU_THRESHOLD = 70;

    /**
     * Default Threshold percentage level of the memory for initial SIP
     * requests.
     */
    private static final int DEFAULT_IR_MEM_THRESHOLD = 85;

    /**
     * Default threshold percentage level of the CPU for subsequent SIP
     * requests.
     */
    private static final int DEFAULT_SR_CPU_THRESHOLD = 90;

    /**
     * Default threshold percentage level of the CPU for all messages. Requests
     * will be dropped above this threshold.
     */
    private static final int DEFAULT_MM_CPU_THRESHOLD = 99;

    /**
     * Default threshold percentage level of the memory for all messages.
     * Requests will be dropped above this threshold.
     */
    private static final int MM_MEM_THRESHOLD = 99;

    /**
     * Default threshold percentage level of the CPU for HTTP requests.
     */
    private static final int HTTP_CPU_THRESHOLD = 70;

    /**
     * Default threshold percentage level of the memory for HTTP requests.
     */
    private static final int HTTP_MEM_THRESHOLD = 85;

    /**
     * Default sample rate in seconds.
     */
    private static final int DEFAULT_SAMPLE_RATE = 2; // 2 seconds

    /**
     * Default number of samples the threshold is exceeded before concluding
     * overload.
     */
    private static final int DEFAULT_NR_OF_SAMPLES = 5; // 5 samples
    private static final Logger LOGGER = LogUtil.SIP_LOGGER.getLogger();
    private static OverloadProtectionManager _singletonInstance = new OverloadProtectionManager();
    private Layer _nextLayer = null;
    private GeneralTimer _timer = null;
    private boolean _cpuOverloadRegulationEnabled = false;
    private boolean _memOverloadRegulationEnabled = false;
    private int _sampleRate = DEFAULT_SAMPLE_RATE;
    private int _numberOfSamples = DEFAULT_NR_OF_SAMPLES;
    private int _initialRequestCPUThreshold = DEFAULT_IR_CPU_THRESHOLD;
    private int _initialRequestMemThreshold = DEFAULT_IR_MEM_THRESHOLD;
    private int _subsequentRequestCPUThreshold = DEFAULT_SR_CPU_THRESHOLD;
    private int _maxMessageCPUThreshold = DEFAULT_MM_CPU_THRESHOLD;
    private int _maxMessageMemThreshold = MM_MEM_THRESHOLD;
    private int _httpCPUThreshold = HTTP_CPU_THRESHOLD;
    private int _httpMemThreshold = HTTP_MEM_THRESHOLD;
    private final OSUtil _osUtil;
    private AtomicInteger _overloadDetectedInitialRequestCounter = new AtomicInteger(
            0);
    private AtomicInteger _memOverloadDetectedInitialRequestCounter = new AtomicInteger(
            0);
    private AtomicInteger _overloadDetectedSubsequentRequestCounter = new AtomicInteger(
            0);
    private AtomicInteger _overloadDetectedMaxLevelCounter = new AtomicInteger(
            0);
    private AtomicInteger _memOverloadDetectedMaxLevelCounter = new AtomicInteger(
            0);
    private AtomicInteger _overloadDetectedHttpCounter = new AtomicInteger(0);
    private AtomicInteger _memOverloadDetectedHttpCounter = new AtomicInteger(0);
    private boolean _alarmRaised = false;
    private boolean _alarmRaisedMemory = false;

    // private String _hostName = "";
    private Reporter _reporter;

    private boolean isStarted = false;

    private OverloadProtectionManager() {
        String allProcess = System.getProperty("AllProcess");

        // If AllProcess will be set to true default, if it not set as -D
        // attribute.
        String allProcessFlag = "true"; // non-null means set which is default

        if ("false".equals(allProcess)) {
            allProcessFlag = null; // unset the flag
        }

        if (LOGGER.isLoggable(Level.FINEST)) {
            LOGGER.log(Level.FINEST, "logging cpu for all processes? "
                    + ((allProcess == null) ? "false" : "true"));
        }

        _osUtil = OSUtilFactory.getOSUtil(allProcessFlag);

        if (_osUtil == null) {
            LOGGER.log(Level.WARNING, OverloadProtectionManager.class.getCanonicalName()
                    + ".no_os_utils");
            // lets ensure that this class is not used since no util class was
            // found.
            _cpuOverloadRegulationEnabled = false;
            _memOverloadRegulationEnabled = false;
        }
    }

    /**
     * Note! The configuration are in preference tree under tm. A list of the
     * parameter:<br> - CpuOverloadRegulation - MemOverloadRegulation -
     * SampleRate - NumberOfSamples - HttpThreshold - MemHttpThreshold -
     * IrThreshold - MemIrThreshold - SrThreshold - MmThreshold - MemMmThreshold
     * <br>
     * <br>
     * -DAllProcess = false: Monitoring is sun specific monitoring for both
     * linux and windows, which monitors the running jvm total cpu busy
     * activity. <br>
     * -DAllProcess=true, overrides the sunspecific behaviour and is only valid
     * on Linux. It enables accumulated cpu busy monitoring for all process.
     * <br>
     * The default behaviour is equivalent to -DAllprocess = true
     * @return the singleton
     */
    public static OverloadProtectionManager getInstance() {
        if (shouldInsertOLPM()) {
            return _singletonInstance;
        } else {
            return null;
        }
    }

    private static boolean shouldInsertOLPM() {
        return "true".equals(ConfigFactory.getConfig().get("/SipContainer", "olpInserted"));
    }

    /**
     * @see getInstance
     * @return the singleton
     */
    public static HttpLayer getHttpLayerInstance() {
        if (shouldInsertOLPM()) {
            return _singletonInstance;
        } else {
            return null;
        }
    }

    /**
     * Initializes this Singleton. It will start to monitor the CPU load in case
     * of raising an alarm.
     */
    public void start() {
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.log(Level.FINE,
                    "Starting OverloadProtection for CPU and/or Memory");
        }

        startSampleTimer();
        isStarted = true;
    }

    /**
     * stop the layer.
     */
    public void stop() {
        // XXX remove method when configuration activation is part of layer
        // handling
        isStarted = false;
        ConfigFactory.instance().deactivateConfiguration(this);

        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.log(Level.FINE,
                    "Stopped OverloadProtection for CPU and/or Memory");
        }
    }

    /**
     * start the timer if needed
     */
    private synchronized void startSampleTimer() {
        if (isEnabled() && (_timer == null)){
            // Overload timer not started yet.
            doStartSampleTimer();
        }
    }

    private synchronized void doStartSampleTimer() {
        try {
            if (LOGGER.isLoggable(Level.FINEST)) {
                LOGGER.log(Level.FINEST,
                        "Starting timer for OverloadProtection functionality, SampleRate: "
                                + _sampleRate + " * 1000");
            }

            _timer = TimerServiceImpl.getInstance().createTimer(this,
                    _sampleRate * 1000, _sampleRate * 1000, true, null);
        } catch (Exception e) {
            LOGGER.log(Level.SEVERE, OverloadProtectionManager.class.getCanonicalName()
                    + ".could_not_start_timer");
			LOGGER.log(Level.SEVERE, e.getMessage(), e);
        }
    }

    /**
     * ensure that the timer is stopped
     */
    private synchronized void stopSampleTimer() {
        if (_timer != null) {
            if (LOGGER.isLoggable(Level.FINEST)) {
                LOGGER.log(Level.FINEST,
                        "Stopping timer for OverloadProtection functionality");
            }

            _timer.cancel();
            _timer = null;
        }
    }

    private boolean isEnabled() {
        return (_osUtil != null)
                && (_cpuOverloadRegulationEnabled || _memOverloadRegulationEnabled);
    }

    private boolean isOverloadDetectedInitialRequest() {
        return _overloadDetectedInitialRequestCounter.get() >= _numberOfSamples;
    }

    private boolean isMemOverloadDetectedInitialRequest() {
        return _memOverloadDetectedInitialRequestCounter.get() >= _numberOfSamples;
    }

    private boolean isOverloadDetectedSubsequentRequest() {
        return _overloadDetectedSubsequentRequestCounter.get() >= _numberOfSamples;
    }

    private boolean isOverloadDetectedHttpRequest() {
        return _overloadDetectedHttpCounter.get() >= _numberOfSamples;
    }

    private boolean isMemOverloadDetectedHttpRequest() {
        return _memOverloadDetectedHttpCounter.get() >= _numberOfSamples;
    }

    private boolean isOverloadDetectedMaxLevel() {
        return _overloadDetectedMaxLevelCounter.get() >= _numberOfSamples;
    }

    private boolean isMemOverloadDetectedMaxLevel() {
        return _memOverloadDetectedMaxLevelCounter.get() >= _numberOfSamples;
    }

    // EVDV replaced by req.isInitial()
    // private boolean isInitial(SipServletMessageImpl m)
    // {
    // try
    // {
    // return m.getAddressHeader(Header.TO).getParameter(AddressImpl.TAG_PARAM)
    // == null;
    // }
    // catch(ServletParseException e)
    // {
    // return true;
    // }
    // }
    
    /**
     * check on threshold violoations and:
     * @see com.ericsson.ssa.sip.Layer#next(com.ericsson.ssa.sip.SipServletRequestImpl)
     */
    public void next(final SipServletRequestImpl req) {
        if (isEnabled()) {
            if (LOGGER.isLoggable(Level.FINEST)) {
                LOGGER.log(Level.FINEST, "Method next (req) counters "
                        + printCounters());
            }
            
            if (!isOverloadDetectedMaxLevel()
                    && !isMemOverloadDetectedMaxLevel()) {
                boolean initial = req.isInitial();

                if (initial) {
                    if (isOverloadDetectedInitialRequest()
                            || isMemOverloadDetectedInitialRequest()) {
                        SipServletResponseImpl resp = req
                                .createTerminatingResponse(503);

                        if (LOGGER.isLoggable(Level.FINE)) {
                            LOGGER
                                    .log(Level.FINE,
                                            "Method next (req): rejecting initial request with 503");
                        }

                        // TR HH52078
                        if (resp == null) {
                            return;
                        }

                        // Remote has been set when creating the terminating
                        // response
                        resp.popDispatcher().dispatch(resp);
                        incrEasOverloadRejectedSipRequests();

                        return;
                    }
                } else {
                    // lets handle ACK as an initial request, it's a risk
                    // otherwise
                    // that INVITE is stopped but ACK is sent to next layer...
                    if (req.getMethod().equals("ACK")) {
                        if (isOverloadDetectedInitialRequest()
                                || isMemOverloadDetectedInitialRequest()) {
                            // drop ACK
                            if (LOGGER.isLoggable(Level.FINE)) {
                                LOGGER.log(Level.FINE,
                                        "Method next (req): dropping ACK ");
                            }

                            incrEasOverloadRejectedSipRequests();

                            return;
                        }
                    } else if (isOverloadDetectedSubsequentRequest()) {
                        // TR HH52078
                        SipServletResponseImpl resp = req
                                .createTerminatingResponse(503);

                        if (resp == null) {
                            return;
                        }

                        if (LOGGER.isLoggable(Level.FINE)) {
                            LOGGER
                                    .log(Level.FINE,
                                            "Method next (req): rejecting subsequent request with 503");
                        }

                        resp.popDispatcher().dispatch(resp);
                        incrEasOverloadRejectedSipRequests();

                        return;
                    }
                }
            } else {
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER
                            .log(Level.FINE,
                                    "Max load, drop all sip requests");
                }

                incrEasOverloadRejectedSipRequests();

                return;
            }
        }

        LayerHelper.next(req, this, _nextLayer);
    }

    public void next(final SipServletResponseImpl resp) {
        if (isEnabled() && (isOverloadDetectedMaxLevel() || isMemOverloadDetectedMaxLevel())) {
            if (LOGGER.isLoggable(Level.FINEST)) {
                LOGGER.log(Level.FINEST, "Method next (resp) counters "
                        + printCounters());
            }

            if (LOGGER.isLoggable(Level.FINE)) {
                LOGGER
                        .log(
                                Level.FINE,
                                "Method next (resp):" 
                                + "Max load, drop all sip responses");
            }

            return;
        }

        LayerHelper.next(resp, this, _nextLayer);
    }

    public void registerNext(final Layer layer) {
        _nextLayer = layer;
    }

    public void dispatch(final SipServletRequestImpl req) {
        req.popDispatcher().dispatch(req);
    }

    public void dispatch(final SipServletResponseImpl resp) {
        resp.peekDispatcher().dispatch(resp);
    }

    /**
     * Calculates threshold for memory or cpu usage If the given threshold has
     * been exceeded, a counter is incremented.
     * 
     */
    private void calculateThreshold(final int usage,
            final AtomicInteger levelCounter, final Integer threshold,
            final String typeOfThreshold) {
        if (LOGGER.isLoggable(Level.FINEST)) {
            LOGGER.log(Level.FINEST, "calculateThreshold: Type="
                    + typeOfThreshold + " Usage(%)=" + usage + " levelCounter="
                    + levelCounter.toString() + " threshold="
                    + threshold.toString());
        }

        if (usage > threshold) {
            if (levelCounter.get() < _numberOfSamples) {
                levelCounter.incrementAndGet();
            }
        } else {
            levelCounter.set(0);
        }
    }

    /**
     * With the set sample rate the overload levels will be re-calculated
     * periodically and a flag will be raised per level if the levels are higher
     * than the protection levels that are set.
     */
    public void timeout(final GeneralTimer timer) {
        if (_cpuOverloadRegulationEnabled) {
            int busy = cpuBusy(); // fetch cpu level from os...

            if (LOGGER.isLoggable(Level.FINEST)) {
                LOGGER.log(Level.FINEST, "timeout - cpu: " + busy);
            }

            // calculate thresholds...
            calculateThreshold(busy, _overloadDetectedInitialRequestCounter,
                    _initialRequestCPUThreshold, "CPU_SIP_IR");
            calculateThreshold(busy, _overloadDetectedSubsequentRequestCounter,
                    _subsequentRequestCPUThreshold, "CPU_SIP_SR");
            calculateThreshold(busy, _overloadDetectedHttpCounter,
                    _httpCPUThreshold, "CPU_HTTP");
            calculateThreshold(busy, _overloadDetectedMaxLevelCounter,
                    _maxMessageCPUThreshold, "CPU_MAX");

            handleCpuAlarm(busy);
        }

        if (_memOverloadRegulationEnabled) {
            int memoryusage = memUsage(); // fetch memory
            // usage...

            if (LOGGER.isLoggable(Level.FINEST)) {
                LOGGER.log(Level.FINEST, "timeout - memory: " + memoryusage);
            }

            // calculate thresholds...
            calculateThreshold(memoryusage,
                    _memOverloadDetectedInitialRequestCounter,
                    _initialRequestMemThreshold, "MEM_SIP_IR");
            calculateThreshold(memoryusage, _memOverloadDetectedHttpCounter,
                    _httpMemThreshold, "MEM_HTTP");
            calculateThreshold(memoryusage,
                    _memOverloadDetectedMaxLevelCounter,
                    _maxMessageMemThreshold, "MEM_MAX");

            handleMemoryAlarm(memoryusage);
        }
    }

    private void handleMemoryAlarm(int anMemoryusage) {
        // Check if any threshold has been exceeded, if so raise alarm, if not
        // (and alarm is raised), cease it.
        if (isMemOverloadDetectedInitialRequest()
                || isMemOverloadDetectedHttpRequest()
                || isMemOverloadDetectedMaxLevel()) {
            raiseMemoryAlarm(anMemoryusage);
        } else { // No threshold exceeded
            clearMemoryAlarm();
        }
    }

    private void clearMemoryAlarm() {
        if (_alarmRaisedMemory) { // An alarm has been raised, cease it.

            if (LOGGER.isLoggable(Level.INFO)) {
                LOGGER
                        .info("The memory overload limit is below the threshold, will clear an alarm.");
            }

            FmEventSender.clearAlarmMemoryLoadLimitExceeded();
            _alarmRaisedMemory = false;
        }
    }

    private void raiseMemoryAlarm(int anMemoryusage) {
        if (!_alarmRaisedMemory) {
            try {
            if (LOGGER.isLoggable(Level.INFO)) {
                LOGGER
                        .info("The Memory overload limit is exceeded, will raise an alarm.");
            }
            } catch (Throwable e) { 
                System.err.println("WHAAAAAAAAA");
                e.printStackTrace();
            }

            _alarmRaisedMemory = true;

            // Raise the alarm.
            // HH34467 remove reason from raiseAlarmCpuLoadLimitExceeded
            FmEventSender.raiseAlarmMemoryLoadLimitExceeded("measured usage: " + anMemoryusage);
        }
    }
    
    /**
     * Raises the alarm(s) if a threshold has been exceeded. Ceases the alarm(s)
     * if level falls below the thresholds
     * @param anBusy 
     * 
     */
    private void handleCpuAlarm(int anBusy) {
        // Check if any threshold has been exceeded, if so raise alarm, if not
        // (and alarm is raised), cease it.
        if (isOverloadDetectedInitialRequest()
                || isOverloadDetectedSubsequentRequest()
                || isOverloadDetectedHttpRequest()
                || isOverloadDetectedMaxLevel()) {
            raiseCpuAlarm(anBusy);
        } else { // No threshold exceeded
            clearCpuAlarm();
        }
    }

    private void clearCpuAlarm() {
        if (_alarmRaised) { // An alarm has been raised, cease it.

            if (LOGGER.isLoggable(Level.INFO)) {
                LOGGER
                        .info("The CPU overload limit is below the threshold, will clear an alarm.");
            }
            FmEventSender.clearAlarmCpuLoadLimitExceeded();
            _alarmRaised = false;
        }
    }

    private void raiseCpuAlarm(int anBusy) {
        if (!_alarmRaised) {
            if (LOGGER.isLoggable(Level.INFO)) {
                LOGGER
                        .info("The CPU overload limit is exceeded, will raise an alarm.");
            }
            _alarmRaised = true;
            // Raise the alarm.
            // HH34467 remove reason from raiseAlarmCpuLoadLimitExceeded
            FmEventSender.raiseAlarmCpuLoadLimitExceeded("measured load: " + anBusy);
        }
    }

    private int cpuBusy() {
        return (_osUtil != null) ? _osUtil.cpuBusy() : 0;
    }

    private int memUsage() {
        return (_osUtil != null) ? _osUtil.memUsage() : 0;
    }

    void incrEasOverloadRejectedSipRequests() {
        OverloadManager.getInstance().incrEasOverloadRejectedSipRequests();
    }

    void incrEasOverloadRejectedHttpRequests() {
        OverloadManager.getInstance().incrEasOverloadRejectedHttpRequests();
    }

    /**
     * Returns the threshold of the cpu for initial requests (range 0-100%). A
     * 503 response will be returned when this level is reached. Default
     * IR_CPU_THRESHOLD.
     * @return the threshold of the cpu for initial requests.
     */
    public Integer getIrThreshold() {
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.log(Level.FINE, "IrThreshold: "
                    + _initialRequestCPUThreshold);
        }

        return _initialRequestCPUThreshold;
    }

    /**
     * Sets the threshold of the cpu for initial requests (range 0-100%). A 503
     * response will be returned when this level is reached. Default
     * IR_CPU_THRESHOLD.
     * @param threshold
     *                The threshold to set.
     */
    @Configuration(key = "IrThreshold", node = "/SipContainer")
    public void setIrThreshold(final Integer threshold) {
        if (LOGGER.isLoggable(Level.INFO)) {
            LOGGER.log(Level.INFO, "IrThreshold value will be set to "
                    + threshold.intValue());
        }

        // default IR_CPU_THRESHOLD, answer 503
        if ((threshold >= 0) && (threshold <= 100)) {
            _initialRequestCPUThreshold = threshold;
        } else {
            LOGGER.log(Level.WARNING, OverloadProtectionManager.class.getCanonicalName()
                    + ".illegal_param_value",
                    new Object[] { threshold.intValue(), "IrThreshold" });
        }
    }

    /**
     * Returns the threshold of the memory for initial requests (range 0-100%).
     * A 503 response will be returned when this level is reached. Default
     * IR_MEM_THRESHOLD.
     * @return the threshold of the cpu for initial requests.
     */
    public Integer getMemIrThreshold() {
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.log(Level.FINE, "MemIrThreshold: "
                    + _initialRequestMemThreshold);
        }

        return _initialRequestMemThreshold;
    }

    /**
     * Sets the threshold of the memory for initial requests (range 0-100%). A
     * 503 response will be returned when this level is reached. Default
     * IR_MEM_THRESHOLD.
     * @param threshold
     *                The threshold to set.
     */
    @Configuration(key = "MemIrThreshold", node = "/SipContainer")
    public void setMemIrThreshold(final Integer threshold) {
        if (LOGGER.isLoggable(Level.INFO)) {
            LOGGER.log(Level.INFO, "MemIrThreshold value will be set to "
                    + threshold.intValue());
        }

        // default IR_MEM_THRESHOLD, answer 503
        if ((threshold >= 0) && (threshold <= 100)) {
            _initialRequestMemThreshold = threshold;
        } else {
            LOGGER.log(Level.WARNING, OverloadProtectionManager.class.getCanonicalName()
                    + ".illegal_param_value",
                    new Object[] { threshold.intValue(), "MemIrThreshold" });
        }
    }

    /**
     * Returns the threshold of the cpu for subsequent requests (range 0-100%).
     * A 503 response will be returned when this level is reached. Default
     * SR_CPU_THRESHOLD.
     * @return the threshold of the cpu for subsequent requests.
     */
    public Integer getSrThreshold() {
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.log(Level.FINE, "SrThreshold: "
                    + _subsequentRequestCPUThreshold);
        }

        return _subsequentRequestCPUThreshold;
    }

    /**
     * Sets the threshold of the cpu for subsequent requests (range 0-100%). A
     * 503 response will be returned when this level is reached. Default
     * SR_CPU_THRESHOLD.
     * @param threshold
     *                The threshold to set.
     */
    @Configuration(key = "SrThreshold", node = "/SipContainer")
    public void setSrThreshold(final Integer threshold) {
        if (LOGGER.isLoggable(Level.INFO)) {
            LOGGER.log(Level.INFO, "SrThreshold value will be set to "
                    + threshold.intValue());
        }

        // default HTTP_CPU_THRESHOLD, answer 503
        if ((threshold >= 0) && (threshold <= 100)) {
            _subsequentRequestCPUThreshold = threshold;
        } else {
            LOGGER.log(Level.WARNING, OverloadProtectionManager.class.getCanonicalName()
                    + ".illegal_param_value",
                    new Object[] { threshold.intValue(), "SrThreshold" });
        }
    }

    /**
     * Returns the threshold of the cpu for http requests (range 0-100%). A 503
     * response will be returned when this level is reached. Default
     * HTTP_CPU_THRESHOLD.
     * @return the threshold of the cpu for subsequent requests.
     */
    public Integer getHttpThreshold() {
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.log(Level.FINE, "HttpThreshold: " + _httpCPUThreshold);
        }

        return _httpCPUThreshold;
    }

    /**
     * Sets the threshold of the cpu for http requests (range 0-100%). A 503
     * response will be returned when this level is reached. Default
     * HTTP_CPU_THRESHOLD.
     * @param threshold
     *                The threshold to set.
     */
    @Configuration(key = "HttpThreshold", node = "/SipContainer")
    public void setHttpThreshold(final Integer threshold) {
        if (LOGGER.isLoggable(Level.INFO)) {
            LOGGER.log(Level.INFO, "HttpThreshold value will be set to "
                    + threshold.intValue());
        }

        // default SR_CPU_THRESHOLD, answer 503
        if ((threshold >= 0) && (threshold <= 100)) {
            _httpCPUThreshold = threshold;
        } else {
            LOGGER.log(Level.WARNING, OverloadProtectionManager.class.getCanonicalName()
                    + ".illegal_param_value",
                    new Object[] { threshold.intValue(), "HttpThreshold" });
        }
    }

    /**
     * Returns the threshold of the memory for http requests (range 0-100%). A
     * 503 response will be returned when this level is reached. Default
     * HTTP_MEM_THRESHOLD.
     * @return the threshold of the memory for subsequent requests.
     */
    public Integer getMemHttpThreshold() {
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.log(Level.FINE, "MemHttpThreshold: " + _httpMemThreshold);
        }

        return _httpMemThreshold;
    }

    /**
     * Sets the threshold of the memory for http requests (range 0-100%). A 503
     * response will be returned when this level is reached. Default
     * HTTP_MEM_THRESHOLD.
     * @param threshold
     *                The threshold to set.
     */
    @Configuration(key = "MemHttpThreshold", node = "/SipContainer")
    public void setMemHttpThreshold(final Integer threshold) {
        if (LOGGER.isLoggable(Level.INFO)) {
            LOGGER.log(Level.INFO, "MemHttpThreshold value will be set to "
                    + threshold.intValue());
        }

        // default HTTP_MEM_THRESHOLD, answer 503
        if ((threshold >= 0) && (threshold <= 100)) {
            _httpMemThreshold = threshold;
        } else {
            LOGGER.log(Level.WARNING, OverloadProtectionManager.class.getCanonicalName()
                    + ".illegal_param_value",
                    new Object[] { threshold.intValue(), "MemHttpThreshold" });
        }
    }

    /**
     * Returns the threshold of the cpu for maximum load possible for handling
     * messages (range 0-100%). All messages will be dropped when this level is
     * reached. Default MM_CPU_THRESHOLD.
     * @return the threshold of the cpu for max messages
     */
    public Integer getMmThreshold() {
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER
                    .log(Level.FINE, "MmThreshold: "
                            + _maxMessageCPUThreshold);
        }

        return _maxMessageCPUThreshold;
    }

    /**
     * Sets the threshold of the cpu for max load possible for handling messages
     * (range 0-100%). All messages will be dropped when this level is reached.
     * Default MM_CPU_THRESHOLD.
     * @param threshold
     *                The threshold to set.
     */
    @Configuration(key = "MmThreshold", node = "/SipContainer")
    public void setMmThreshold(final Integer threshold) {
        if (LOGGER.isLoggable(Level.INFO)) {
            LOGGER.log(Level.INFO, "MmThreshold value will be set to "
                    + threshold.intValue());
        }

        // default MM_CPU_THRESHOLD, drop all messages
        if ((threshold >= 0) && (threshold <= 100)) {
            _maxMessageCPUThreshold = threshold;
        } else {
            LOGGER.log(Level.WARNING, OverloadProtectionManager.class.getCanonicalName()
                    + ".illegal_param_value",
                    new Object[] { threshold.intValue(), "MmThreshold" });
        }
    }

    /**
     * Returns the threshold of the memory for maximum load possible for
     * handling messages (range 0-100%). All messages will be dropped when this
     * level is reached. Default MM_MEM_THRESHOLD.
     * @return the threshold of the cpu for max messages
     */
    public Integer getMemMmThreshold() {
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.log(Level.FINE, "MemMmThreshold: "
                    + _maxMessageMemThreshold);
        }

        return _maxMessageMemThreshold;
    }

    /**
     * Sets the threshold of the memory for max load possible for handling
     * messages (range 0-100%). All messages will be dropped when this level is
     * reached. Default MM_MEM_THRESHOLD.
     * @param threshold
     *                The threshold to set.
     */
    @Configuration(key = "MemMmThreshold", node = "/SipContainer")
    public void setMemMmThreshold(final Integer threshold) {
        if (LOGGER.isLoggable(Level.INFO)) {
            LOGGER.log(Level.INFO, "MemMmThreshold value will be set to "
                    + threshold.intValue());
        }

        // default MM_MEM_THRESHOLD, drop all messages
        if ((threshold >= 0) && (threshold <= 100)) {
            _maxMessageMemThreshold = threshold;
        } else {
            LOGGER.log(Level.WARNING, OverloadProtectionManager.class.getCanonicalName()
                    + ".illegal_param_value",
                    new Object[] { threshold.intValue(), "MemMmThreshold" });
        }
    }

    /**
     * Returns whether overload protection is used. If true overload protection
     * is turned on.
     * @return whether overload protection is turned on.
     */
    public Boolean getCpuOverloadRegulation() {
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.log(Level.FINE, "CpuOverloadRegulation: "
                    + _cpuOverloadRegulationEnabled);
        }

        return _cpuOverloadRegulationEnabled;
    }

    /**
     * Enable/disable the overload Regulation.
     * @param enabled
     *                if true over load protection is used
     */
    @Configuration(key = "CpuOverloadRegulation", node = "/SipContainer")
    synchronized public void setCpuOverloadRegulation(final Boolean enabled) {
        _cpuOverloadRegulationEnabled = enabled.booleanValue();

        if (LOGGER.isLoggable(Level.INFO)) {
            LOGGER.log(Level.INFO, "CpuOverloadRegulation will be changed to "
                    + _cpuOverloadRegulationEnabled);
        }
        
        if (isStarted) {
            if (!isEnabled()) {
                stopSampleTimer();
            }
            if (!_cpuOverloadRegulationEnabled) {
                resetCpuCounters();
                clearCpuAlarm();
            }
            startSampleTimer();
        }
    }

    private void resetCpuCounters() {        
        _overloadDetectedInitialRequestCounter.set(0);
        _overloadDetectedSubsequentRequestCounter.set(0);
        _overloadDetectedHttpCounter.set(0);
        _overloadDetectedMaxLevelCounter.set(0);
    }

    /**
     * Returns the sample rate of updating the overload protection levels.
     * @return the sample rate of updating the overload protection levels.
     */
    public Integer getSampleRate() {
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.log(Level.FINE, "SampleRate: " + _sampleRate);
        }

        return _sampleRate;
    }

    /**
     * Sets the sample rate of updating the overload protection levels. Must be
     * a positive value.
     * @param rate
     */
    @Configuration(key = "SampleRate", node = "/SipContainer")
    public void setSampleRate(final Integer rate) {
        if (LOGGER.isLoggable(Level.INFO)) {
            LOGGER.log(Level.INFO, "SampleRate value will be set to "
                    + rate.intValue());
        }

        if (rate > 0) {
            _sampleRate = rate;

            // take into account at next timeout.
            // XXX
            // alt: cancel the current timer and start a new one.
        }
    }

    /**
     * Returns the number of consequence samples that is needed before overload
     * is raised.
     * @return the number of consequence samples that is needed before overload
     *         is raised.
     */
    public Integer getNumberOfSamples() {
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.log(Level.FINE, "NumberOfSamples: " + _numberOfSamples);
        }

        return _numberOfSamples;
    }

    /**
     * Sets the number of consequence samples that is needed before overload is
     * raised. The sample rate could minimum be set to 2.
     * @param rate
     *                the sample rate
     */
    @Configuration(key = "NumberOfSamples", node = "/SipContainer")
    public void setNumberOfSamples(final Integer rate) {
        if (LOGGER.isLoggable(Level.INFO)) {
            LOGGER.log(Level.INFO, "NumberOfSamples value will be set to "
                    + rate.intValue());
        }

        _numberOfSamples = (rate < 2) ? 2 : rate;
    }

    /**
     * Returns whether overload protection for memory is used. If true overload
     * protection for memory is turned on.
     * @return whether overload protection form memory is turned on.
     */
    public Boolean getMemOverloadRegulation() {
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.log(Level.FINE, "MemOverloadRegulation: "
                    + _memOverloadRegulationEnabled);
        }

        return _memOverloadRegulationEnabled;
    }

    /**
     * Enable/disable the overload protection for memory.
     * @param enabled
     *                if true over load protection for memory is used
     */
    @Configuration(key = "MemOverloadRegulation", node = "/SipContainer")
    synchronized public void setMemOverloadRegulation(final Boolean enabled) {
        _memOverloadRegulationEnabled = enabled.booleanValue();
        if (LOGGER.isLoggable(Level.INFO)) {
            LOGGER.log(Level.INFO, "MemOverloadRegulation will be changed to "
                    + _memOverloadRegulationEnabled);
        }

        if (isStarted) {
            if (!isEnabled()) {
                stopSampleTimer(); // both CPU and Mem OLP are not enabled
            }
            if (!_memOverloadRegulationEnabled) {
                resetMemCounters();
                clearMemoryAlarm();
            }
            startSampleTimer(); // will only start if mem and CPU OLP enabled
        }
    }

    private void resetMemCounters() {
        _memOverloadDetectedHttpCounter.set(0);
        _memOverloadDetectedInitialRequestCounter.set(0);
        _memOverloadDetectedMaxLevelCounter.set(0);
    }

    /**
     * @return info
     */
    public String getInfo() {
        return "Overload Regulation for SIP&HTTP";
    }

    /* (non-Javadoc)
     * @see org.jvnet.glassfish.comms.httplayers.HttpLayer#invoke(com.sun.grizzly.tcp.Request, com.sun.grizzly.tcp.Response)
     */
    public boolean invoke(final Request request, final Response response) {
        if (isEnabled()) {
            if (LOGGER.isLoggable(Level.FINEST)) {
                LOGGER.log(Level.FINEST,
                        "Method invoke: _cpuOverloadRegulationEnabled=true"
                                + printCounters());
            }
            if (isOverloadDetectedMaxLevel()
                    || isMemOverloadDetectedMaxLevel()) {
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.log(Level.FINE,
                            "Method invoke: Max load, drop all http messages");
                }
                // XXX check if this is OK and whether we should increase
                // rejected
                // the old code did not count dropped as rejected.
                // but this is consistent with the SIP path
                response.setStatus(503);
                incrEasOverloadRejectedHttpRequests();
                return false;
            } else if (isOverloadDetectedHttpRequest()
                    || isMemOverloadDetectedHttpRequest()) {
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER
                            .log(Level.FINE,
                                    "Method invoke: is(Mem)OverloadDetectedHttpRequest, return 503");
                }
                response.setStatus(503);
                incrEasOverloadRejectedHttpRequests();
                return false;
            }
        }

        return true;
    }

    public void postInvoke(final Request request, final Response response)
            throws IOException, ServletException {
        //
    }

    public Reporter getReporter() {
        return _reporter;
    }

    public void setReporters(final String reporters) {
        _reporter = CallflowResolver.getInstance().getReporter(reporters);
    }

    private StringBuffer printCounters() {
        StringBuffer result = new StringBuffer();
        result.append("\nCPU").append("\nMAX:\t").append(
                _overloadDetectedMaxLevelCounter).append("\nHTTP:\t").append(
                _overloadDetectedHttpCounter).append("\nSIP Initial:\t")
                .append(_overloadDetectedInitialRequestCounter).append(
                        "\nSIP subsequent:\t").append(
                        _overloadDetectedSubsequentRequestCounter).append(
                        "\nMEMORY").append("\nMAX:\t").append(
                        _memOverloadDetectedMaxLevelCounter)
                .append("\nHTTP:\t").append(_memOverloadDetectedHttpCounter)
                .append("\nSIP Initial:\t").append(
                        _memOverloadDetectedInitialRequestCounter);

        return result;
    }
}
