/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package com.ericsson.ssa.container.processor;

import com.ericsson.ssa.container.SipContainerThreadPool;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.logging.Level;

// inserted by hockey (automatic)
import java.util.logging.Logger;
import org.jvnet.glassfish.comms.util.LogUtil;


/**
 * This is a processor which executes tasks, provided through a queue, in a
 * separate thread. In case no processor thread exist, one is taken from the
 * thread pool.
 * <p>
 * The thread will be active for a configurable time, if no new tasks are
 * received within that time the thread is killed.
 * <p>
 * Before the processor thread takes a task from the input queue the queue is
 * scavenged from erroneous tasks. For each of the erroneous tasks the error
 * action of the task is invoked.
 * <p>
 * In case the queue is full when putting a task, the error action of the task
 * is invoked.
 * <p>
 * The processor can be shutdown; when doing that the processor thread is killed
 * and each task is removed from the input queue and the error action is invoked
 * on the task.
 *
 * @author ejoelbi
 * @since 2006 nov 1
 * @reveiwed ehsroha 2006-nov-14
 */
public class QueuedProcessor implements Callable {
    public static final String SHUTDOWN_OF_PROCESSOR = "Shutdown of processor";
    public static final String QUEUE_LENGTH_EXCEEDED = "Queue length exceeded";
    private Logger log = LogUtil.SIP_LOGGER.getLogger();
    private Object mutex = new Object();
    private List<QueuedTask> inputQueue = new LinkedList<QueuedTask>();
    private int maxQueueLength;
    private long inactiveWorkerTimeout;
    private State state = State.IDLE;

    /** This is a signalling flag; it is set to stop the current processor thread */
    private boolean doStop;

    public QueuedProcessor(int maxQueueLength, long inactiveWorkerTimeout) {
        this.maxQueueLength = maxQueueLength;
        this.inactiveWorkerTimeout = inactiveWorkerTimeout;
    }

    /**
     * Checks if the processor thread is active or not.
     *
     * @return true if active
     */
    public boolean isActive() {
        return state == State.ACTIVE;
    }

    /**
     * Checks if the processor has shutdown
     *
     * @return true if shutdown
     */
    public boolean hasShutdown() {
        return state == State.TERMINATED;
    }

    /**
     * Puts the specified task in the queue for later processing.
     *
     * @param task
     *        the task
     */
    public void put(QueuedTask task) throws ProcessorException {
        synchronized (mutex) {
            if (!hasShutdown() && (inputQueue.size() < maxQueueLength)) {
                inputQueue.add(task);
                activateProcessor();
                mutex.notifyAll();
            } else {
                throw new ProcessorException(doStop ? SHUTDOWN_OF_PROCESSOR
                                                    : QUEUE_LENGTH_EXCEEDED,
                    null);
            }
        }
    }

    /**
     * Shutdown this processor. The error action is performed for all tasks in
     * the input queue.
     */
    public void shutdown() {
        synchronized (mutex) {
            doStop = true;
            state = State.TERMINATED;

            for (Iterator<QueuedTask> iter = inputQueue.iterator();
                    iter.hasNext();) {
                QueuedTask task = iter.next();
                iter.remove();
                task.doErrorAction(SHUTDOWN_OF_PROCESSOR);
            }

            mutex.notifyAll();
        }
    }

    /**
     * Activate the processor as a thread in in the SipContainerThreadPool, if
     * not already existing.
     */
    private void activateProcessor() {
        if (state == State.IDLE) {
            state = State.ACTIVE;
            SipContainerThreadPool.getInstance().execute(this);
        }
    }

    @Override
    public String toString() {
        synchronized (mutex) {
            return "[current queue length = " + inputQueue.size() +
            "; max queue length = " + maxQueueLength + "; active = " +
            isActive() + "; hasShutdown = " + hasShutdown() + "]";
        }
    }

    /**
     * Go through the 10 oldest tasks and remove those that are in error state
     * from the input queue and perform error action for each of them.
     * This is to avoid purging queues that conatins several of thousands of items.
     */
    private void removeOldestErrorTasks() {
        int i = 0;

        for (Iterator<QueuedTask> iter = inputQueue.iterator(); iter.hasNext();) {
            QueuedTask task = iter.next();
            String errorDescription;

            if ((errorDescription = task.getError()) != null) {
                iter.remove();
                task.doErrorAction(errorDescription);
            }

            if (++i > 10) {
                break;
            }
        }
    }

    /**
     * The processor.
     */
    public Object call() throws Exception{
        while (true) {
            QueuedTask task = null;

            synchronized (mutex) {
                // The processor thread has been ordered to stop
                if (doStop) {
                    doStop = false;

                    if (state == State.ACTIVE) {
                        state = State.IDLE;
                    }

                    return null;
                }

                // Try to find a non-erroneous task
                task = getTaskFromQueue(task);

                // If no non-erroneous task found, wait or stop
                if (task == null) {
                    if (inactiveWorkerTimeout > 0) {
                        try {
                            mutex.wait(inactiveWorkerTimeout);
                        } catch (InterruptedException e) {
                            // Ignore interrupts
                        }

                        if (inputQueue.size() == 0) {
                            // No new tasks in input queue (we have timed out), time to
                            // die...
                            // Order the thread to stop in the next loop
                            doStop = true;
                        }
                    } else {
                        doStop = true;
                    }
                }
            }

            // If a task has been found, do the processing; otherwise go back to
            // the top
            if (task != null) {
                process(task);
            }
        }
    }

    private QueuedTask getTaskFromQueue(QueuedTask task) {
        // Scavenge queue before doing further processing
        removeOldestErrorTasks();

        if (inputQueue.size() > 0) {
            return inputQueue.remove(0);
        } else {
            return null;
        }
    }

    /**
     * Process a task. Called from the {@link #run} method. Note that it is
     * called without locking {@link #mutex}
     *
     * @param task
     */
    protected void process(QueuedTask task) {
        try {
            task.run();
        } catch (Throwable e) {
            log.log(Level.WARNING, "Unexpected exception when executing task", e);
        }
    }

    public class ProcessorException extends Exception {
        public ProcessorException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    /**
     * The state of the processor:
     * <ul>
     * <li>IDLE: The processor is in operation but has nothing in the list and
     * no active processor thread.
     * <li>ACTIVE: The processor is in operation and has an active processor
     * thread (may also have tasks in the list).
     * <li>TERMINATED: The processor has been shutdown and can no longer process
     * tasks (and there is no way to wake it up).
     * </ul>
     *
     * @author ejoelbi
     */
    private enum State {IDLE,
        ACTIVE,
        TERMINATED;
    }
}
