/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package com.ericsson.ssa.container.sim;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.TreeMap;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.servlet.Servlet;
import javax.servlet.sip.Address;
import javax.servlet.sip.ServletParseException;
import javax.servlet.sip.SipApplicationRouter;
import javax.servlet.sip.SipApplicationRouterInfo;
import javax.servlet.sip.SipApplicationRoutingDirective;
import javax.servlet.sip.SipApplicationRoutingRegion;
import javax.servlet.sip.SipApplicationRoutingRegionType;
import javax.servlet.sip.SipRouteModifier;
import javax.servlet.sip.TooManyHopsException;

import org.apache.catalina.ContainerEvent;
import org.apache.catalina.ContainerListener;
import org.apache.catalina.LifecycleEvent;
import org.apache.catalina.LifecycleListener;
import org.jvnet.glassfish.comms.util.LogUtil;

import com.ericsson.ssa.container.callflow.CallflowResolver;
import com.ericsson.ssa.container.callflow.Reporter;
import com.ericsson.ssa.sip.Dispatcher;
import com.ericsson.ssa.sip.Header;
import com.ericsson.ssa.sip.Layer;
import com.ericsson.ssa.sip.LayerHelper;
import com.ericsson.ssa.sip.RemoteLockException;
import com.ericsson.ssa.sip.RemoteLockRuntimeException;
import com.ericsson.ssa.sip.ServiceHandler;
import com.ericsson.ssa.sip.SipFactoryImpl;
import com.ericsson.ssa.sip.SipServletReadOnlyRequestImpl;
import com.ericsson.ssa.sip.SipServletRequestImpl;
import com.ericsson.ssa.sip.SipServletResponseImpl;
import com.ericsson.ssa.sip.SipSessionBase;
import com.ericsson.ssa.sip.SipSessionImplBase;

/**
 * The ApplicationDispatcher is the Layer in the sip-stack that is responsible
 * for invoking applications. It uses the Application Router to determine which
 * application to invoke.
 * 
 * ApplicationDispatcher maintains two references to application routers.
 * One is the application router that is present in the system by default,
 * typically the AlphabeticalRouter (unless configured otherwise).
 * 
 * The administrator has the possibility of deploying a custom application
 * router by using the 'asadmin deploy' command line.
 * 
 * @author Robert Handl, Per Pettersson, Kristoffer Gronowski, Yvo Bogers
 */
public class ApplicationDispatcher implements Layer,
    ContainerListener, LifecycleListener, ServiceHandler {
    public static final String APPLICATION_DISPATCHER = "com.ericsson.ssa.container.sim.ApplicationDispatcher/1.0";

    /* Singleton instance */
    private static ApplicationDispatcher m_singletonInstance = new ApplicationDispatcher();
    private static TreeMap<String, ServletDispatcher> m_contextMapping = new TreeMap<String, ServletDispatcher>();
    private Logger m_logger = LogUtil.SIP_LOGGER.getLogger();
    private Layer m_nextLayer = null;
    
    /*
     * The runtime router is determined at startup
     */ 
    private SipApplicationRouter runtimeRouter = null;
    
    /*
     * A default router reference is remembered when a custom router is deployed
     * over it. The ApplicationDispatcher resorts to the default whenever the 
     * custom router is undeployed again. 
     */
    private SipApplicationRouter defaultRouter = null;
    private ClassLoader saved = null;

    private Reporter _reporter;
    
    public void setReporters(String reporters){
        _reporter = CallflowResolver.getInstance().getReporter(reporters);
    }
    
    public Reporter getReporter(){
        return _reporter;
    }
        
    protected ApplicationDispatcher() {
        super();
    }

    public synchronized void start() {
        if (runtimeRouter == null) {
            throw new RuntimeException(
                "Can not start without application router.");
        }

        setRouterContext();
        runtimeRouter.init(new ArrayList<String>(m_contextMapping.keySet()));
        resetRouterContext();
    }

    public synchronized void stop() {
        setRouterContext();
        runtimeRouter.destroy();
        resetRouterContext();
    }

    public void lifecycleEvent(LifecycleEvent event) {
    }

    public void containerEvent(ContainerEvent event) {
        if (m_logger.isLoggable(Level.FINE)) m_logger.log(Level.FINE, "Got Container event!");
    }

    /**
     * The method is a part of the dispatch chain. For the initial request the
     * method will try to call the ServletDispatcher to invoke some servlets,
     * otherwise it will pop up the next dispatcher to invoke.
     *
     * @param request
     *           A <code>SipServletRequestImpl</code>.
     */
    public void dispatch(SipServletRequestImpl request) {
        // Check if request is initial
        if (request.isInitial()) {

            // JSR 289, ask the Application Router which application to invoke.
            invokeServletDispatchers(request);
        } else {
            
            // Subsequent request, should go through the default proxy.
            dispatchViaStatelessProxy(request);
        }
    }

    /*
     *  This method implements section 15.4.1 of JSR289.
     */
    private void invokeServletDispatchers(SipServletRequestImpl request) {
        request.setSentOnThread(false);

        Serializable stateInfo = (Serializable)request.getStateInfo();
        String regionStr = null;
        SipApplicationRoutingRegion region = null;
        SipSessionBase session = request.getSessionImpl();
        if (session != null){
                session.access(); // modify the access times
        	regionStr = ((SipSessionImplBase)session).getRegion();
        	if (regionStr != null){
        		region = new SipApplicationRoutingRegion(regionStr, 
        				SipApplicationRoutingRegionType.valueOf(regionStr));	
        	}
        }

        setRouterContext();
              
        // If routing directive is NEW, we should treat this as a new request
        // i.e. reset the Application Router state.
        if (SipApplicationRoutingDirective.NEW.equals(
            request.getRoutingDirective())){
            // clear the stateInfo
            request.setStateInfo(null);
            stateInfo = null;
        }
        
        SipApplicationRouterInfo info = null;
        // issue 192, guard for malicious application routers
        try {
            info = runtimeRouter.getNextApplication(
				new SipServletReadOnlyRequestImpl(request),
				region, 
				request.getRoutingDirective(),
				stateInfo);
        } catch (RuntimeException e){
            m_logger.log(Level.SEVERE, 
                "Application Router threw RuntimeException: " + e.getMessage(),
                e);
            SipServletResponseImpl response = 
                request.createTerminatingResponse(
                    SipServletResponseImpl.SC_SERVER_INTERNAL_ERROR);
            response.popDispatcher().dispatch(response);
            return;
        }
    		
        resetRouterContext();

        Serializable newStateInfo = null;

        if (info != null) {
            newStateInfo = info.getStateInfo();
        }

        request.setStateInfo(newStateInfo);

        while (info != null) {
            
            SipRouteModifier mod = info.getRouteModifier();
            String route = info.getRoute();
            if (SipRouteModifier.ROUTE.equals(mod)) {
                try {
                    Address a = 
                        SipFactoryImpl.getInstance().createAddress(route);
                    request.pushRoute(a);
                } catch (ServletParseException e) {
                    m_logger.log(
                            Level.SEVERE, "The following address in ApplicationRouterInfo " + 
                            "does not have a correct format and will not be " +
                            "added: " + route);
                }
                dispatchViaStatelessProxy(request);
                return;
            } else if (SipRouteModifier.ROUTE_BACK.equals(mod)){
                // First push a route that will eventually send the request 
            	// back to this server 
            	try{
            	    // TODO what is the correct way to create a route to this host?
            	    Address a = SipFactoryImpl.getInstance().createAddress("localhost");
            	    request.pushRoute(a);
            	} catch (ServletParseException e) {
            	    m_logger.log(Level.SEVERE, "Could not push a route back to this host");
            	}
            	// now push the remaining route
            	try {
            	    Address a = 
            	        SipFactoryImpl.getInstance().createAddress(route);
            	    request.pushRoute(a);
            	} catch (ServletParseException e) {
            	    m_logger.log(
            	            Level.SEVERE, "The following address in ApplicationRouterInfo " + 
            	            "does not have a correct format and will not be " +
            	            "added: " + route);
            	}
            	dispatchViaStatelessProxy(request);
                return;
            } else if (SipRouteModifier.CLEAR_ROUTE.equals(mod)) {
            	// clear_route -> remove popped route
                request.setPoppedRoute(null);
                dispatchViaStatelessProxy(request);
                return;
            }

            // else treat as NO_ROUTE and look for application name
            String application = info.getNextApplicationName();

            if (application == null) {
                if (m_logger.isLoggable(Level.FINE)){
                    m_logger.log(Level.FINE, 
                        "Application Dispatcher: no application and NO_ROUTE");
                }
                Dispatcher dispatcher = request.popDispatcher();
               
                if(m_logger.isLoggable(Level.FINE)) { 
                    m_logger.log(Level.FINE,
                       "AD: popped dispatcher " + dispatcher);
                } 
                dispatchViaStatelessProxy(request);
                return;
            }

            ServletDispatcher sd = m_contextMapping.get(application);
            
            //Do lazy registeration to be less dependend on timeing between
            //reporter initialization and ServlertDispatcher initialization.
            addReporter(sd);

            if (sd==null){
                m_logger.log(
                        Level.SEVERE, "Unable to invoke application " + application 
            		+ ", check your AR configuration!");

                SipServletResponseImpl response = 
                        request.createTerminatingResponse(
                            SipServletResponseImpl.SC_SERVER_INTERNAL_ERROR,
                            "Application Router error");
                response.popDispatcher().dispatch(response);
                return;
            }
            boolean matched = false;
            try {
                request.activateSentOnThread(); //Activate the function
                
                // Push the ApplicationDispatcher on to the application stack
                // so we come back into the dispatch() method when the request
                // is proxied, for example.
                request.pushApplicationDispatcher(this);
                matched = sd.invoke(request, info);

                
            } catch (TooManyHopsException tmhe) {
                if (m_logger.isLoggable(Level.FINE)) {
                    m_logger.log(Level.FINE, "Too Many Hops Exception ", tmhe);
                }

                SipServletResponseImpl response = 
                        request.createTerminatingResponse(
                            SipServletResponseImpl.SC_TOO_MANY_HOPS);
                response.popDispatcher().dispatch(response);

                return;
            } catch (RemoteLockRuntimeException e) {
                if (m_logger.isLoggable(Level.FINE)) {
                    m_logger.log(Level.FINE, "Dialog was remotely locked ", e);
                }

                SipServletResponseImpl response = request.createTerminatingResponse(500);
                response.setHeader(Header.RETRY_AFTER, "5");
                response.popDispatcher().dispatch(response);

                return;
            } catch (Exception e) {
                if (m_logger.isLoggable(Level.FINE)) {
                    m_logger.log(Level.FINE, "Error invoking service ", e);
                    e.printStackTrace();
                }

                SipServletResponseImpl response = 
                        request.createTerminatingResponse(
                            SipServletResponseImpl.SC_SERVER_INTERNAL_ERROR);
                response.popDispatcher().dispatch(response);

                return;
            } finally {
                request.clearSentOnThread(); //Important not to leak mem
            }

            if ( matched ) {
                // The invoked application has either proxied the request, or
                // replied with a response.
                return;
            } else {
                
                // pop the ApplicationDispatcher 
                request.popDispatcher();
                
                // No match, cleanup and take another round in the Application Router
                setRouterContext();
                // issue 192, guard for malicious application routers
                try {
                    info = runtimeRouter.getNextApplication(
                            new SipServletReadOnlyRequestImpl(request),
                            region,
                            SipApplicationRoutingDirective.CONTINUE, 
                            newStateInfo);
                } catch (RuntimeException e){
                    m_logger.log(Level.SEVERE, 
                    "Application Router threw RuntimeException: " + e.getMessage(),
                    e);
                    SipServletResponseImpl response = 
                        request.createTerminatingResponse(
                            SipServletResponseImpl.SC_SERVER_INTERNAL_ERROR,
                            "Application Router error");
                    response.popDispatcher().dispatch(response);
                    return;
                }
                resetRouterContext();

                newStateInfo = null;

                if (info != null) {
                    newStateInfo = info.getStateInfo();
                }

                request.setStateInfo(newStateInfo);
            }
        }
        //No apps matching, doing stateless SIP proxying. If not the correct behavior
        //then it is up to the user to deploy an app that always returns 404!
        dispatchViaStatelessProxy(request);
    }

    /**
     * The method will pop up and invoke the next <code>Dispatcher</code>.
     *
     * @param response
     *           A <code>SipServletResponseImpl</code>.
     */
    public void dispatch(SipServletResponseImpl response) {
        // The transaction layer populates the dispatchers
        // for the request, so here we pop the dispatchers
        try {
            response.popDispatcher().dispatch(response);
        } catch (Exception ex) {
            m_logger.log(Level.SEVERE, "Unhandled exception ", ex);
        }
    }

    private void dispatchViaStatelessProxy(SipServletRequestImpl request) {
        
        // Internal stateless proxy, decrease Max-Forwards by one.
        int maxForwards = request.getMaxForwards();
        --maxForwards;

        if (maxForwards <= 0) {
            SipServletResponseImpl response = request.createTerminatingResponse(
                SipServletResponseImpl.SC_TOO_MANY_HOPS);

            if (response != null) {
                dispatch(response);
            }

            return;
        }
        
        if (m_logger.isLoggable(Level.FINE)){
            m_logger.log(Level.FINE, 
                "ApplicationDispatcher: dispatching " + request.getMethod() + 
                " request via internal stateless proxy. The Max-forwards " +
                "header value is decreased by one.");
        }

        request.setMaxForwards(maxForwards);
        SipSessionBase session = request.getSessionImpl();
        if (session != null){
            session.access(); // modify the access times
        }

        Dispatcher dispatcher = request.popDispatcher();
        dispatcher.dispatch(request);
    }

    /**
     * Registers a Servlet Dispatcher associated to a application Name
     *
     * @param applicationName
     *           The application name.
     * @param servletDispatcher
     *           A <code>ServletDispatcher</code>.
     */
    public void addServletDispatcher(String applicationName,
        ServletDispatcher servletDispatcher) {
        m_contextMapping.put(applicationName, servletDispatcher);

        // Notify the application router
        List<String> apps = new ArrayList<String>(1);
        apps.add(applicationName);

        setRouterContext();
        runtimeRouter.applicationDeployed(apps);
        resetRouterContext();
    }

    /**
     * Deregisters a Servlet Dispatcher for a specified application
     *
     * @param applicationName
     */
    public void removeServletDispatcher(String applicationName) {
        ServletDispatcher sd = m_contextMapping.remove(applicationName);
        removeReporter(sd);
        
        // Notify the application router
        List<String> apps = new ArrayList<String>(1);
        apps.add(applicationName);

        setRouterContext();
        runtimeRouter.applicationUndeployed(apps);
        resetRouterContext();
    }

    // Helper method for serialization
    public ServletDispatcher getServletDispatcher(String applicationName) {
        return m_contextMapping.get(applicationName);
    }

    /**
     * Returns the SipServlet of the application name and SipSession handler
     *
     * @param applicationName
     *           the application name
     * @param handler
     *           the SipSession handler
     * @return SipServlet of the application name and SipSession handler
     */
    public Servlet getHandler(String applicationName, String handler) {
        ServletDispatcher disp = m_contextMapping.get(applicationName);

        if (disp != null) {
            return disp.getHandler(handler);
        }

        return null;
    }

    /**
     * @param request
     */
    public void next(SipServletRequestImpl request) {
        LayerHelper.next(request, this, m_nextLayer);
    }

    /**
     * @param response
     */
    public void next(SipServletResponseImpl response) {
        LayerHelper.next(response, this, m_nextLayer);
    }

    /**
     * @param layer
     */
    public void registerNext(Layer layer) {
        m_nextLayer = layer;
    }

    /**
     * Return an instance of <code>ApplicationDispatcher</code>.
     *
     * @return A <code>ApplicationDispatcher</code>.
     */
    public static ApplicationDispatcher getInstance() {
        return m_singletonInstance;
    }

    /**
     * Set a custom Application Router. A custom deployed application router
     * will take precedence over the default application router.
     * 
     * @param aRouter
     */
    public void setCustomApplicationRouter(SipApplicationRouter aRouter) {
        m_logger.log(Level.FINE, "Setting custom Application Router: " + aRouter);

        setRouterContext();
        // initialize the new application router with the currently
        // active list of applications.
        aRouter.init(new ArrayList<String>(m_contextMapping.keySet()));
        
        // allow the currently active runtime application router to cleanup
        // its resources
        if (runtimeRouter != null){
            if (m_logger.isLoggable(Level.FINE)){
                m_logger.log(Level.FINE, 
                    "Overwriting previously active AR: " 
                    + runtimeRouter);
            }
            runtimeRouter.destroy();
        }
        resetRouterContext();

        // use the new router as runtime application router
        this.runtimeRouter = aRouter;
    }
    
    /**
     * Fall back to the default application router
     * 
     * @return false if no default is available, in which case custom router is
     * not unset. true otherwise.
     */
    public boolean unsetCustomApplicationRouter(){
        if (defaultRouter != null){
            m_logger.log(Level.INFO, 
                "unsetCustomApplicationRouter(). SIP request routing will now " +
                "be done using " + defaultRouter + ". To override, deploy a new"
                + " application router jar.");
            
            setRouterContext();
            // initialize the default application router with the currently
            // active list of applications.
            defaultRouter.init(new ArrayList<String>(m_contextMapping.keySet()));
            
            // allow the currently active runtime application router to cleanup
            // its resources
            runtimeRouter.destroy();
            resetRouterContext();
            
            // the default router becomes the new runtime.
            this.runtimeRouter = defaultRouter;
            return true;
        }
        else return false;
    }

    /**
     * Specify the default Application Router classname. The Application 
     * Dispatcher will use java reflection to instantiate the default 
     * application router. The default application router is used for routing
     * SIP traffic. It is possible to override this behaviour by calling 
     * setCustomApplicationRouter.
     * 
     * This method will not initialize the router yet. This will happen when
     * the start() method is called. 
     * 
     * @param className
     */
    public void setApplicationRouterClass(String className) {
        if (m_logger.isLoggable(Level.FINE))
            m_logger.log(Level.FINE, "Instantiating default Application Router : " + className);

        try {
            Class ar = Class.forName(className);
            defaultRouter = (SipApplicationRouter) ar.newInstance();
            
            // Check if a custom router was already deployed. If not, use this
            // one as the runtime instance.
            if (runtimeRouter == null){
                if (m_logger.isLoggable(Level.FINE))
                    m_logger.log(Level.FINE, "No custom Application Router has been deployed yet, using default.");
                runtimeRouter = defaultRouter;
            } else {
                if (m_logger.isLoggable(Level.FINE))
                    m_logger.log(Level.FINE, 
                    "A custom Application Router has been deployed on the system." 
                    + "The default applicaiton router will not be used.");
            }
            
            // If annotaion would be supported, do it here
        } catch (ClassNotFoundException e) {
            m_logger.log(Level.WARNING,
                "Failed to instantiate ApplicationRouter", e);
        } catch (InstantiationException e) {
            m_logger.log(Level.WARNING,
                "Failed to instantiate ApplicationRouter", e);
        } catch (IllegalAccessException e) {
            m_logger.log(Level.WARNING,
                "Failed to instantiate ApplicationRouter", e);
        }
    }

    /**
     * Push a service dispatcher on top of the application stack.
     *
     * @param request
     *           On which the service dispatcher will be pushed.
     */
//    public void pushServiceDispatcher(SipServletRequestImpl request) {
//        request.pushApplicationDispatcher(m_serviceDispatcher);
//    }

    /**
     * Return descriptive information about this Valve implementation.
     */
    public String getInfo() {
        return (APPLICATION_DISPATCHER);
    }

    /**
     * Utility method for setting the context classloader. 
     * Move this to common module?
     */
    private ClassLoader _setContextClassLoader(ClassLoader newClassLoader) {

        // Can reference only final local variables from dopriveleged block
        final ClassLoader classLoaderToSet = newClassLoader;

        final Thread currentThread = Thread.currentThread();
        ClassLoader originalClassLoader = currentThread.getContextClassLoader();

        if (classLoaderToSet != originalClassLoader) {
            if (System.getSecurityManager() == null) {
                currentThread.setContextClassLoader(classLoaderToSet);
            } else {
                java.security.AccessController.doPrivileged(
                        new java.security.PrivilegedAction() {
                            public java.lang.Object run() {
                                currentThread.setContextClassLoader
                                (classLoaderToSet);
                                return null;
                            }
                        }
                );
            }
        }
        return originalClassLoader;
    }

    /**
     * Sets the context for invoking the application router
     */
    private void setRouterContext() {
        try {
            ClassLoader routerCL = 
            runtimeRouter.getClass().getClassLoader();
            saved = _setContextClassLoader(routerCL);
        } catch (Exception e) {
            if (m_logger.isLoggable(Level.WARNING)) {
                m_logger.log(Level.WARNING, e.getMessage(), e);
            }
        }
    }

    /**
     * Resets the context after invoking the application router
     */
    private void resetRouterContext() {
        try {
            _setContextClassLoader(saved);
        } catch (Exception e) {
            if (m_logger.isLoggable(Level.WARNING)) {
                m_logger.log(Level.WARNING, e.getMessage(), e);
            }
        }
    }

    private void addReporter(ServletDispatcher sd) {
        if (sd!=null && _reporter!=null) {
            sd.addReporter(_reporter);
        }
    }
    
    private void removeReporter(ServletDispatcher sd) {
        if (sd!=null && _reporter!=null) {
            sd.removeReporter();            
        }
    }
    
    public void logActiveCaches() {
        for (ServletDispatcher sd : m_contextMapping.values()) {
            sd.logActiveCaches();
        }
    }
}
