/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package com.ericsson.ssa.sip;


import com.ericsson.ssa.container.startup.SipMonitoring;

import java.util.ArrayList;
import java.util.Iterator;
import org.apache.catalina.Lifecycle;
import org.apache.catalina.LifecycleException;
import org.apache.catalina.LifecycleListener;
import org.apache.catalina.util.LifecycleSupport;

import org.jvnet.glassfish.comms.util.LogUtil;
import java.util.logging.Logger;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;


/**
 * A singleton manager for dialog fragments.
 */
public class DialogFragmentManager implements Lifecycle {
    private static final Logger SIP_LOGGER = LogUtil.SIP_LOGGER.getLogger();

    private static DialogFragmentManager dfManager = null;

    static {
        try {
            String dfManagerClassName = System.getProperty("sipContainer.df.dfManagerClass", 
                    "org.jvnet.glassfish.comms.replication.dialogmgmt.ReplicationDialogFragmentManager");
            dfManager = (DialogFragmentManager) Class.forName(dfManagerClassName).newInstance();
        } catch (Throwable t) {
            throw new ExceptionInInitializerError(t);
        }
    }

    private AtomicLong concurrentSipDialogs;
    private AtomicLong totalSipDialogCount;
    private AtomicLong totalSipDialogLifeTime;
    private AtomicLong expiredSipDialogs;

    // Active cache of dialog fragments
    protected ConcurrentHashMap<String, DialogFragment> dialogFragments;

    // Has this DialogFragmentManager been started?
    protected AtomicBoolean isStarted = new AtomicBoolean(); // defaults to false

    // Has this DialogFragmentManager been initialized?
    protected AtomicBoolean isInitialized = new AtomicBoolean(); //defaults to false

    // The lifecycle event support.
    protected LifecycleSupport lifecycle = new LifecycleSupport(this);

    /**
     * Constructor.
     *
     * Scope is protected since this is a singleton.
     */
    protected DialogFragmentManager() {
        dialogFragments = new ConcurrentHashMap<String, DialogFragment>();
        concurrentSipDialogs = new AtomicLong();
        totalSipDialogCount = new AtomicLong();
        totalSipDialogLifeTime = new AtomicLong();
        expiredSipDialogs = new AtomicLong();
    }

    /**
     * Gets the singleton DialogFragmentManager instance.
     *
     * @return The singleton DialogFragmentManager instance
     */
    public static DialogFragmentManager getInstance() {
        
        // FIXME.   Quick fix to make sure DialogFragmentManager gets started.
        if (dfManager != null && ! dfManager.isInitialized.get()) {
            try {
                // FIXME
                // Would like to have this called from org.jvnet.glassfish.comms.startup.SipServiceListener.lifecycleevent()
                // Call DialogFragmentManager.getInstance().start() after startSipStack() in lifecyleEvent().  
                // Also, have DialogFragmentManager.getInstance().stop() called after stopSipStack() in lifecycleEvent.
                dfManager.start();
            } catch (LifecycleException le) {
                // FIXME.  Log caught exception.
            }
        }
        return dfManager;
    }
   
    /**
     * Factory method to instansiate DialogFragments
     * @param ds
     * @param shouldBeConfirmed 
     * @return
     */
    public DialogFragment createDialogFragment(DialogSet ds) {
        return new DialogFragment(ds);
    }

    /**
     * Adds the given DialogFragment.
     *
     * @param d The DialogFragment to add
     */
    public void registerDialogFragment(DialogFragment d) {
 
        if (SipMonitoring.isEnabled(SipMonitoring.SESSION_MANAGER) && 
                (d.isInitialDialog())) {
            concurrentSipDialogs.incrementAndGet();
        }
        
        dialogFragments.put(d.getDialogId(), d);
    }

    /**
     * Gets the DialogFragment with the given id.
     *
     * @param id The DialogFragment id to match
     *
     * @return The DialogFragment with matching id, or null if not found
     *
     * @throws RemoteLockException
     */
    public DialogFragment findDialogFragment(String id)
            throws RemoteLockException {
        return dialogFragments.get(id);
    }

    /**
     * Removes the given DialogFragment.
     *
     * @param df The DialogFragment to be removed
     * @param hasTimedOut true if the DialogFragment is removed because it has
     * expired, false otherwise
     */
    public void removeDialogFragment(DialogFragment df, boolean hasTimedOut) {
        DialogFragment storedDlg = dialogFragments.remove(df.getDialogId());

        // if dialogfragment storedDlg is found/removed it was a "real" dialog
        if ((storedDlg != null) && SipMonitoring.isEnabled(SipMonitoring.SESSION_MANAGER) &&
                  (df.isInitialDialog())) {
            concurrentSipDialogs.decrementAndGet();
            totalSipDialogCount.incrementAndGet();
            totalSipDialogLifeTime.addAndGet(df.getDialogLifetime());

            if (SIP_LOGGER.isLoggable(Level.FINE)) {
                long nDialogs = totalSipDialogCount.get();
                double cumulativeLifetime = totalSipDialogLifeTime.doubleValue();
                double average = cumulativeLifetime / nDialogs;
                SIP_LOGGER.log(Level.FINE,
                    "DialogLifetime=" + df.getDialogLifetime() + "; nDialogs=" +
                    nDialogs + "; cumulativeLifetime=" + cumulativeLifetime +
                    ";averageLifetime=" + average);
            }

            if (hasTimedOut) {
                expiredSipDialogs.incrementAndGet();
            }
        }
    }

    /**
     * Gets the total number of dialog fragments that were ever created.
     *
     * @return The total number of dialog fragments that were ever created
     */
    public long getEasTotalSipDialogCount() {
        return totalSipDialogCount.longValue();
    }

    /**
     * Gets the number of dialog fragments that are currently active.
     *
     * @return The number of dialog fragments that are currently active.
     */
    public long getEasConcurrentSipDialogs() {
        return concurrentSipDialogs.longValue();
    }

    /**
     * Gets the cumulative lifetime of all dialog fragments.
     *
     * @return The cumulative lifetime of all dialog fragments
     */
    public long getEasTotalSipDialogLifeTime() {
        return totalSipDialogLifeTime.longValue();
    }

    /**
     * Gets the total number of dialog fragments that have expired.
     *
     * @return The total number of dialog fragments that have expired
     */
    public long getEasExpiredSipDialogs() {
        return expiredSipDialogs.longValue();
    }

    /**
     * Starts this DialogFramgmentManager.
     */
    public void start() throws LifecycleException {
        if (isInitialized.compareAndSet(false, true)) {
            init();
        }

        if (isStarted.compareAndSet(false, true)) {
            doStart();
        }
    }

    /**
     * Stops this DialogFragmentManager.
     */
    public void stop() throws LifecycleException {
        if (isStarted.compareAndSet(true, false)) {
            // only stop once
            doStop();
        }
    }

    /**
     * Adds a lifecycle event listener to this SipSessionManager.
     *
     * @param listener The lifecycle event listener to add
     */
    public void addLifecycleListener(LifecycleListener listener) {
        lifecycle.addLifecycleListener(listener);
    }

    /**
     * Gets the lifecycle event listeners of this SipSessionManager.
     *
     * @return Array (possibly of zero length) containing the lifecycle
     * event listeners of this SipSessionManager
     */
    public LifecycleListener[] findLifecycleListeners() {
        return lifecycle.findLifecycleListeners();
    }

    /**
     * Removes the give lifecycle event listener from this
     * SipSessionManager.
     *
     * @param listener The lifecycle event listener to remove
     */
    public void removeLifecycleListener(LifecycleListener listener) {
        lifecycle.removeLifecycleListener(listener);
    }

    protected void init() {
    }
    
    protected void doStart() {
    }
    
    protected void doStop() {
        // FIXME Clean up dialog fragments, ..
    }
    
    /**
     * Internal method for logging only.
     * @return
     */
    public Iterable<DialogFragment> getDialogs() {
        return new ArrayList<DialogFragment>(dialogFragments.values()); 
    }

    /*
     * Removes all DialogFragments that contain a pathnode with a SipSession
     * that matches the given appId.
     *
     * @param appId the application id to match
     */
    void removeDialogFragments(String appId) {

        synchronized (dialogFragments) {
            Iterator<DialogFragment> dfIt =
                dialogFragments.values().iterator();
            while (dfIt.hasNext()) {
                DialogFragment df = dfIt.next();
                for (Iterator<PathNode> pnIt = df.getCallee2CallerPath();
                        pnIt.hasNext();) {
                    SipSessionDialogImpl ss = (SipSessionDialogImpl)
                            pnIt.next().getSipSession();
                    if (ss != null && appId.equals(ss.getApplicationId())) {
                        dfIt.remove();
                        DialogSet ds = df.getDialogSet();
                        if (ds != null) {
                            ds.removeDialog(df);
                        }
                        break;
                    }
                }
            }
        }
    }
}
