/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package com.ericsson.ssa.sip;

import com.ericsson.ssa.container.SipContainerThreadPool;
import com.ericsson.ssa.sip.persistence.ReplicationUnitOfWork;

import java.util.concurrent.Callable;
import java.util.logging.Level;

import javax.servlet.sip.SipServletMessage;


/**
 * This represents the finite state machine of a UAC, UAS or combined
 *
 * @author ehsroha
 */
public abstract class FSM {
    protected FSM() {
    }

    /**
     * Will return and if needed create a FSM responsible of the processing of
     * the message or null if no FSM was capable.
     *
     * @param req
     *        the incoming request is the key to decide which FSM to create
     * @return a suitable FSM or null if no one is found
     */
    public static FSM createFSM(SipServletMessage m) {
        FSM fsm = INVITESession.createFSM(m);

        if (fsm != null) {
            return fsm;
        }

        fsm = SUBSCRIBE_REFERSession.createFSM(m);

        if (fsm != null) {
            return fsm;
        }

        fsm = GeneralSession.createFSM(m);

        if (fsm != null) {
            return fsm;
        }

        // list of rest of fsm if any
        return null;
    }

    /**
     * Returns a clone of current FSM
     *
     * @return a clone of current FSM
     */
    public abstract Object clone();

    /**
     * Whether this FSM is responsible of processing the message or not.
     *
     * @param m
     *        the incoming message
     * @return if this instance is responsible for the message or not
     */
    public abstract boolean isResponsible(SipServletMessage m);

    /**
     * Whether it's safe to remove this instance or not
     *
     * @return if it's safe to delete it or not
     */
    public abstract boolean isDeletable();

    /**
     * Handles an incoming request from the local UAC. Updates the session and
     * forwards the request to the remote UAS upon success.
     *
     */
    public void send(SipServletRequestImpl req, UA uac)
        throws IllegalStateException {
        final SipServletRequestImpl forward = req;
        forward.getSessionImpl().updateSipSessionState(forward, uac.getType());

        SipContainerThreadPool.getInstance().execute(new Callable() {
                public Object call() throws Exception {
                    // Start the dialog lifecycle u-o-w
                    forward.getDialog().getDialogLifeCycle().setThreadLocalUnitOfWork();

                    // push the request to the dispatcher
                    forward.popDispatcher().dispatch(forward);

                    return null;
                }
            });
    }

    /**
     * Handles an incoming response from the local UAS. Updates the session and
     * forwards the response to the remote UAC upon success.
     */
    public void send(SipServletResponseImpl resp, UA uas)
        throws IllegalStateException {
        final SipServletResponseImpl forward = resp;

        SipContainerThreadPool.getInstance().execute(new Callable() {
                public Object call() throws Exception {
                    // Activate the dialog lifecycle u-o-w
                    forward.getDialog().getDialogLifeCycle().setThreadLocalUnitOfWork();

                    // push the request to the dispatcher
                    forward.popDispatcher().dispatch(forward);

                    return null;
                }
            });
    }

    /**
     * Handles an incoming request from the remote UAC and updates the session.
     */
    public abstract void dispatch(SipServletRequestImpl req, UA uas);

    /**
     * Handles an incoming response from the remote UAS and updates the session.
     */
    public abstract void dispatch(SipServletResponseImpl resp, UA uac);
}
