/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package com.ericsson.ssa.sip;

import com.ericsson.ssa.container.datacentric.DataCentricUtil;

public final class SipApplicationSessionUtil {

    private static final DataCentricUtil dataCentricUtil =
        DataCentricUtil.getInstance();

    /**
     * Gets the actual server instance (i.e, with respect to the actual
     * cluster shape) to which the given SipApplicationSession id currently
     * maps.
     *
     * @param sasId the SipApplicationSession id to be mapped
     *
     * @return the server instance to which the specified 
     * SipApplicationSession id currently maps
     */
    public static String getActualServerInstance(String sasId) {
        return dataCentricUtil.getActualServerInstance(
            getSipApplicationKey(sasId));
    }

    /**
     * Gets the server instance to which the specified SipApplicationSession
     * id would map if the cluster were complete (i.e., if all cluster
     * instances were up).
     *
     * @param sasId the SipApplicationSession id to be mapped
     *
     * @return the ideal server instance to which the specified 
     * SipApplicationSession id would map if all cluster instances were up
     */
    public static String getIdealServerInstance(String sasId) {
        return dataCentricUtil.getIdealServerInstance(
            getSipApplicationKey(sasId));
    }

    /**
     * Checks if the specified SipApplicationSession id maps to the current
     * (local) instance.
     *
     * @param sasId the SipApplicationSession id 
     *
     * @return true if the given SipApplicationSession id maps to the local
     * server instance; false otherwise
     */
    public static boolean isLocal(String sasId) {
        return dataCentricUtil.isLocal(getSipApplicationKey(sasId));
    }

    /**
     * Extracts the application (hash) key portion of the given
     * SipApplicationSession id and returns it.
     *
     * Format: see {@link #createSasId(String, String, String)}
     *
     * @param sasId the SipApplicationSession id
     *
     * @return the application (hash) key portion of the given
     * SipApplicationSession id
     */
    public static String getSipApplicationKey(String sasId) {
        try {
            int indexOfComma = sasId.indexOf(',');
            int indexOfSecondComma = sasId.indexOf(',', indexOfComma+1);
            int uuidIndex = Integer.parseInt(sasId.substring(0, indexOfComma));
            if (uuidIndex != 0) {
                return sasId.substring(indexOfSecondComma+1).substring(0, uuidIndex);
            }
            return null;
        } catch (Throwable t) {
            throw new IllegalArgumentException("The SAS-ID was malformed: "+sasId, t);
        }
    }
    
    /**
     * Extracts the application name portion of the given
     * SipApplicationSession id and returns it.
     *
     * Format: see {@link #createSasId(String, String, String)}
     *
     * @param sasId the SipApplicationSession id
     *
     * @return the application name portion of the given
     * SipApplicationSession id
     */
    public static String getApplicationName(String sasId) {
        try {
            int indexOfComma = sasId.indexOf(',');
            int indexOfSecondComma = sasId.indexOf(',', indexOfComma+1);
            int appNameIndex = Integer.parseInt(sasId.substring(indexOfComma+1, indexOfSecondComma));
            String idParts = sasId.substring(indexOfSecondComma+1);
            if (idParts.length() != appNameIndex) {
                return idParts.substring(appNameIndex);
            }
            return null;
        } catch (Throwable t) {
            throw new IllegalArgumentException("The SAS-ID was malformed: "+sasId, t);
        }
    }

    /**
     * Extracts the random portion of the given
     * SipApplicationSession id and returns it.
     *
     * Format: see {@link #createSasId(String, String, String)}
     *
     * @param sasId the SipApplicationSession id
     *
     * @return the random portion of the given
     * SipApplicationSession id
     */
    public static String getRandomPart(String sasId) {
        try {
            int indexOfComma = sasId.indexOf(',');
            int indexOfSecondComma = sasId.indexOf(',', indexOfComma+1);
            int uuidIndex = Integer.parseInt(sasId.substring(0, indexOfComma));
            int appNameIndex = Integer.parseInt(sasId.substring(indexOfComma+1, indexOfSecondComma));
            if (uuidIndex != appNameIndex) {
                return sasId.substring(indexOfSecondComma+1).substring(uuidIndex, appNameIndex);
            }
            return null;
        } catch (Throwable t) {
            throw new IllegalArgumentException("The SAS-ID was malformed: "+sasId, t);
        }
    }

    /**
     * Creates a SAS-Id based on the specified values.
     * 
     * Format: <index to uuid>,<index to appname>;<key><uuid><appname>, e.g "4,10;key1uuid23MyApp". 
     * The first two numbers are indices to uuid part and ap name part, respectively.
     *  
     * @param key the key that indicates to which instance the SAS belongs; a key that 
     * can be used for look up in the consistent hash (e.g, {@link DataCentricUtil#isLocal(String)}) 
     * @param appName the name of the application to which the SAS belongs
     * @param uuid an optional random part
     * @return the SAS-Id
     */
    public static String createSasId(String key, String appName, String uuid) {
        StringBuilder sb = new StringBuilder();
        int uuidIndex = key!=null?key.length():0;
        int appNameIndex = uuid!=null?uuidIndex+uuid.length():uuidIndex;
        sb.append(uuidIndex).append(',').append(appNameIndex).append(',');
        if (key!=null) sb.append(key);
        if (uuid!=null) sb.append(uuid);
        if (appName!=null) sb.append(appName);
        return sb.toString();
    }
}
