/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package com.ericsson.ssa.sip;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;
import java.util.logging.Level;
import org.jvnet.glassfish.comms.util.LogUtil;

import javax.servlet.sip.URI;


/**
 * Part of the Proxy implementation. There might be several TargetSets per
 * Proxy.
 *
 * @author ehsroha
 */
public abstract class TargetSet {
    private static final Logger _log = LogUtil.SIP_LOGGER.getLogger();
    private static final List<TargetSet> EMPTY_TARGET_SET = new ArrayList<TargetSet>(0);
    protected int _proxyTimeout = 0;
    private final ProxyImpl _proxy;
    private TargetSet _parent = null;

    /**
     * The constructor of the TargetSet implementation.
     *
     * @param p
     *           the Proxy this set belongs to
     */
    protected TargetSet(ProxyImpl p) {
        _proxy = p;
    }

    protected List<TargetSet> getTargetSets() {
        return EMPTY_TARGET_SET;
    }

    protected void setParent(TargetSet t) {
        _parent = t;
    }

    public TargetSet getParent() {
        return _parent;
    }

    /**
     * Return ProxyBranch matching URI. Should only be used by reponse of initial
     * request
     *
     * @param uri
     * @return searched ProxyBranch if found. Null is returned if branch is not
     *         found
     */
    public ProxyBranchImpl findBranch(URI uri) {
        ProxyBranchImpl branch = null;

        for (TargetSet t : getTargetSets()) {
            branch = t.findBranch(uri);

            if (branch != null) {
                return branch;
            }
        }

        return null;
    }

    public void addTopLevelBranch(List<ProxyBranchImpl> branches) {
        for (TargetSet t : getTargetSets()) {
            t.addTopLevelBranch(branches);
        }
    }

    /**
     * Returns the response of 2xx. If no 2xx exist the response of 6xx is
     * returned, otherwise the response of the lowest status number found is
     * returned. If the target set is not finished or not started null is
     * returned.
     *
     * @return the response of 2xx. If no 2xx exist the response of 6xx is
     *         returned, otherwise the response of the lowest status number found
     *         is returned. NOTE: if TargetSet is not finished or not started
     *         null is returned
     */
    public SipServletResponseImpl findBestResponse() {
        if (!isStarted()) {
            return null;
        }

        // lets assume target set has finished
        boolean isRunning = false;
        SipServletResponseImpl bestResponse = null;
        SipServletResponseImpl currentResp = null;
        int currentStatus = -1;
        int bestStatus = 700; // one higher than possible

        for (TargetSet t : getTargetSets()) {
            currentResp = t.findBestResponse();

            if (currentResp == null) {
                // haven't got all answers, but 2xx or 6xx might already be in
                // set.
                isRunning = true;
            } else if (!currentResp.isAlreadyRedirected()) {
                currentStatus = currentResp.getStatus();

                // 2xx response should always be returned...
                if ((currentStatus / 100) == 2) {
                    // can't be better, lets return...
                    return currentResp;
                }
                // ...otherwise save 6xx response (might find 2xx later)...
                else if ((currentStatus / 100) == 6) {
                    bestStatus = currentStatus;
                    bestResponse = currentResp;
                }
                // ...otherwise save the response of the lowest status number
                else if (((bestStatus / 100) != 6) &&
                        (bestStatus > currentResp.getStatus())) {
                    bestStatus = currentResp.getStatus();
                    bestResponse = currentResp;
                }
            }
        }

        if (isRunning) {
            // haven't got all answers, lets wait for better response context
            return null;
        }

        // have collected all answers
        return bestResponse;
    }

    public abstract void proxyTo() throws IllegalStateException;

    public abstract void cancel() throws IllegalStateException;

    public void add(TargetSet t) throws IllegalStateException {
        throw new IllegalStateException(
            "Not allowed to add targets in this target set");
    }

    public boolean hasNext() {
        return false;
    }

    public void next() {
        // nothing to do
    }

    public ProxyImpl getProxyImpl() {
        return _proxy;
    }

    public abstract boolean getParallel();

    public abstract boolean isStarted();

    public abstract boolean getRecurse();

    public void setProxyTimeout(int t) {
        _proxyTimeout = t;
    }

    public int getProxyTimeout() {
        return _proxyTimeout;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder(getProxyImpl().toString());
        sb.append(" isParallel: ");
        sb.append(getParallel());
        sb.append(" isRecurse: ");
        sb.append(getRecurse());

        return sb.toString();
    }
}
