/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2008 Sun Microsystems, Inc. All rights reserved.
 * Copyright (c) Ericsson AB, 2004-2008. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package com.ericsson.ssa.sip.persistence;

import java.util.logging.Level;
import java.util.logging.Logger;

import org.jvnet.glassfish.comms.util.LogUtil;

import com.ericsson.ssa.container.callflow.CallflowResolver;
import com.ericsson.ssa.container.callflow.Reporter;
import com.ericsson.ssa.sip.DialogFragment;
import com.ericsson.ssa.sip.Layer;
import com.ericsson.ssa.sip.LayerHelper;
import com.ericsson.ssa.sip.SipFactoryImpl;
import com.ericsson.ssa.sip.SipServletRequestImpl;
import com.ericsson.ssa.sip.SipServletResponseImpl;


/**
 * Replication layer intercepting message flow
 *
 * @author epakdsz
 */
public class ReplicationManager implements Layer {
    private static final ReplicationManager singletonInstance = new ReplicationManager();
    private Logger logger = LogUtil.SIP_LOGGER.getLogger();
    private Layer nextLayer = null;
    private Reporter reporter;

    // Enforce Singleton pattern
    private ReplicationManager() {
        // Empty
    }

    public void setReporters(String reporters) {
        reporter = CallflowResolver.getInstance().getReporter(reporters);
    }

    public Reporter getReporter() {
        return reporter;
    }

    public static ReplicationManager getInstance() {
        if (singletonInstance.logger.isLoggable(Level.FINE)) {
            singletonInstance.logger.log(Level.FINE, "Created ReplicationManager");
        }
        return singletonInstance;
    }

    /**
     * Delegate to next layer.
     * Incoming UAS request.
     */
    public void next(SipServletRequestImpl req) {
        req.pushTransactionDispatcher(this);
        req.pushApplicationDispatcher(this);

        // Paranoid extra check
        if (ReplicationUnitOfWork.getThreadLocalUnitOfWork() != null) {
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE, "An UOW was set on the thread here. Should not happen!!! UOW: "+ReplicationUnitOfWork.getThreadLocalUnitOfWork());
            }
            ReplicationUnitOfWork.clearThreadLocal();
        }

        // Send onwards
        LayerHelper.next(req, this, nextLayer);

        // The following will be executed after the layers above have executed
        replicateOnRequest(req, true);
        
        ReplicationUnitOfWork.clearThreadLocal();
    }

    /**
     * Delegate to next layer.
     * Incoming responses for UAC.
     */
    public void next(SipServletResponseImpl resp) {
        // Paranoid extra check
        if (ReplicationUnitOfWork.getThreadLocalUnitOfWork() != null) {
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE, "An UOW was set on the thread here. Should not happen!!! UOW: "+ReplicationUnitOfWork.getThreadLocalUnitOfWork());
            }
            ReplicationUnitOfWork.clearThreadLocal();
        }

        // Send onwards
        LayerHelper.next(resp, this, nextLayer);
        
        // The following will be executed after the layers above have executed
        replicateOnResponse(resp, true);
        
        // Try to finish the current UOW
        DialogFragment dialog = resp.getDialog();
        if (dialog != null) {
            dialog.getDialogLifeCycle().trigDelayedFinish();
        }
        
        ReplicationUnitOfWork.clearThreadLocal();
    }

    private void replicateOnResponse(SipServletResponseImpl resp, boolean inbound) {
        if (logger.isLoggable(Level.FINEST)) 
            logger.log(Level.FINEST, "Enter replicateOnResponse");
        DialogFragment df = resp.getDialog();

        if (df != null) {
            if (inbound) {
                if (df.containsUA()) {
                    df.getDialogLifeCycle().save();
                } 
            } else {
                df.getDialogLifeCycle().save();
            }
        }
    }

    public void registerNext(Layer layer) {
        nextLayer = layer;
    }

    /**
     * Delegate to next layer, UAC
     */
    public void dispatch(SipServletRequestImpl req) {
        // Send onwards to next lower layer
        req.popDispatcher().dispatch(req);
        
        replicateOnRequest(req, false);

        ReplicationUnitOfWork.clearThreadLocal();
    }

    private void replicateOnRequest(SipServletRequestImpl req, boolean inbound) {
        if (logger.isLoggable(Level.FINEST)) 
            logger.log(Level.FINEST, "Enter replicateOnRequest");
        DialogFragment df = req.getDialog();

        if (df != null) {
            if ("ACK".equalsIgnoreCase(req.getMethod()) && df.isConfirmed()) {
                if (inbound) {
                    if (df.containsUA()) {
                        df.getDialogLifeCycle().save();
                    } 
                } else {
                    df.getDialogLifeCycle().save();
                }
            }
        }
    }

    /**
     * Delegate to next layer.
     * Replicate on UAS outgoing responses.
     */
    public void dispatch(SipServletResponseImpl resp) {
        // Send onwards to next lower layer
        resp.popDispatcher().dispatch(resp);
        
        replicateOnResponse(resp, false);

        ReplicationUnitOfWork.clearThreadLocal();
    }
}
